/*
 * File: CipherFrame.java
 * @author R. Morelli <ralph.morelli@trincoll.edu>
 * 
 * <P>Description: Implements a Java 1.1 version of a GUI tool for using and analyzing
 *  historical ciphers.

 * <P>Credits: This program is modelled after the ImageJ program which is written by
 *  Wayne Rasband of the National Institutes of Health. ImageJ is in the public domain.
 *

 * <P>Copyright: This program is in the public domain. You can modify it as you 
 *  see fit as long as you properly acknowledge its original authors (Morelli
 *  and Rasband). It would also be nice if you forwarded your changes to 
 *  <A HREF= "mailto:ralph.morelli@trincoll.edu">ralph.morelli@trincoll.edu</A> so 
 *  they can possibly be added to the "official" version.
 *
 * MODIFICATIONS:
 * 7/30/02 -- Modified the file open and save operations to use file_encoding
 *  information entered by the user in a TextField.
 */
package cryptotoolj;

import hcrypto.analyzer.*;
import hcrypto.cipher.*;
import hcrypto.provider.*;
import hcrypto.engines.*;

import java.io.*;
import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.datatransfer.*;

/**
 * This class provides a user interface for performing encryption and 
 * decryption operations on a particular message. It supports file I/O
 * (for the application version). 
 */

public class CipherFrame extends Frame implements ActionListener, ItemListener,
                                        WindowListener, ClipboardOwner
{
    public static final int WIDTH = 500;
    public static final int HEIGHT = 500;
 
    private CryptoToolJ cryptoTool;     //reference to the CryptoToolJ
    private Cipher cipher;              // reference to a Cipher
    private HistoricalKey key;
    private String keyClassName;
    
    private TextAreaPlus inputField, outputField, textAreaInFocus = null;
    private TextField encodingField;
    private Button clearInput, clearOutput, encodeButton, decodeButton, keyButton;
    private Panel inputPanel, buttonPanel, outputPanel;
    private Choice selectionBox; //for storing Cipher selections
    private String keywordString;	
    private String keyPromptString;
    private String windowId = "";
    public boolean isActivated = false;

    /**
     * This constructor creates a frame with a menubar and text
     *  areas for plaintext and ciphertext. The menubar includes
     *  a drop down list of the available cipher engines.
     */ 
    public CipherFrame(CryptoToolJ ctj) {           //constructor
        super("Cipher");
        cryptoTool = ctj;
	    
        this.createCipherFrame();
        addWindowListener(this);

        setMenuBar(Menus.getMenuBar());	    
        setResizable(true);
        setSize(WIDTH, HEIGHT);
	pack();
        setVisible(true);
        Tools.centerWindow(this);
        requestFocus();
    }  //end constructor()
    
    /**
     * Renames the window with descriptive title that refers
     *  to the type of cipher engine being used.
     */ 
    public void renameWithWindowId(String id) {
        String s = getTitle();
        int index = s.indexOf(":");
        setTitle(s.substring(0, index) + id + s.substring(index));
    }

    /**
     *  Manages the detailed operations of setting up the user interface,
     *  including setting up a ChoiceBox containing the names of both
     *  the  built-in and plugin ciphers.
     */ 
   private void createCipherFrame() {
      selectionBox = new Choice();
      java.util.StringTokenizer st = 
           new java.util.StringTokenizer(Provider.getCipherNames(), Provider.DELIMITER);    
      while (st.hasMoreTokens()) {
	  String s = st.nextToken();
	  selectionBox.addItem(s);   
      }
      selectionBox.select("Caesar (Default)");
      selectionBox.addItemListener(this);    
      
      if (!cryptoTool.isApplet()) {                  // Plugins not available in applet version
          File file = new File("plugins");
          String plugins[] = file.list();
          if (plugins != null) {
              for (int k = 0; k < plugins.length; k++) {
                  String className = plugins[k];
                  if (className.endsWith("Spi.class")) {
                      String algoName = className.substring(0, className.indexOf("Spi.class")); 
                      algoName = algoName + " (Plugin)";
                      selectionBox.addItem(algoName);
		  }          
	      }
	  }
      }
      setSelectedCipher();
      setTitle("Cipher: " + cipher.getAlgorithm() + " (" +  cipher.getProvider() +")");	 
        
      inputField = new TextAreaPlus(this, "Input Text Area");
      inputField.setEditable(true);
      outputField = new TextAreaPlus(this, "Output Text Area");
      outputField.setEditable(true);
        
      buttonPanel = new Panel();
      keyButton = new Button("Set Key");
      encodeButton = new Button("Encrypt");
      decodeButton = new Button("Decrypt");
      encodingField = new TextField("default");
      encodeButton.setEnabled(false);
      decodeButton.setEnabled(false);
      keyButton.addActionListener(this);
      encodeButton.addActionListener(this);
      decodeButton.addActionListener(this);  
      buttonPanel.add(selectionBox);     
      buttonPanel.add(keyButton);
      buttonPanel.add(encodeButton);
      buttonPanel.add(decodeButton);
      buttonPanel.add(new Label("Encoding:"));
      buttonPanel.add(encodingField);
        
      add(buttonPanel, "North");
      add(inputField, "Center");
      add(outputField, "South");
   }
    
    /**
     * Creates an instance of the cipher selected by the user. 
     */ 
    private void setSelectedCipher() {
        String cipherSelection = (String) selectionBox.getSelectedItem();
        String algorithm = cipherSelection.substring(0, cipherSelection.indexOf(" "));
        String provider = cipherSelection.substring(cipherSelection.indexOf("(") + 1, cipherSelection.length()-1);
        if (provider.indexOf("Plugin") != -1) {
          try {
              System.out.println("Using Plugin algorithm " + algorithm);
              cipher = Cipher.getInstance(algorithm, provider);
              keyClassName ="plugins." +  algorithm + "Key";
              key = null;
          } catch (Exception e) {
              System.out.println("Exception: " + e.getMessage());
              e.printStackTrace();
          }
         } else {
            cipher = Cipher.getInstance(algorithm, provider);
            keyClassName ="hcrypto.engines." +  algorithm + "Key";
            key = null;
         }        
    }

   /**
    * Opens a file and reads its text into the InputField
    */  
    public void openFile() {
	//        inputField.openFile();
        inputField.openFile(encodingField.getText());
        inputField.markFileSaved();
    }
    
   /**
    * Saves the text is response to Save or SaveAs commands.
    */  
    public boolean saveTextInTextAreas(boolean rename) {
        int result = 0;
        if (inputField.isDirty()) {
            result = inputField.saveToFile();
            if (result == FileSaveMessageDialog.CANCEL_OPTION)  // Cancel stops close
                return false;
            else if (result == FileSaveMessageDialog.SAVE_OPTION)
                inputField.save(encodingField.getText(), inputField.getFile() == null);    // Save or don't
        }
                
        if (outputField.isDirty()) {
            result = outputField.saveToFile();
            if (result == FileSaveMessageDialog.CANCEL_OPTION)  // Cancel stops close
                return false;
            else if (result == FileSaveMessageDialog.SAVE_OPTION)
                outputField.save(encodingField.getText(), outputField.getFile() == null);    // Save or don't
        }
        return true;        
    }

   /**
    * Handles user's choices in the ChoiceBox.
    */  
    public void itemStateChanged(ItemEvent e) {
        if (e.getSource() == selectionBox) {
            setSelectedCipher();
            String s = getTitle();
            int index = s.indexOf(":");
            setTitle(s.substring(0, index) + ":" + cipher.getAlgorithm() + " (" +  cipher.getProvider() +")");
            encodeButton.setEnabled(false);
            decodeButton.setEnabled(false);
            keyButton.setEnabled(true);        
        }
    }
    
   /**
    * Opens a separate frame and displays certain useful statistics
    *  for the selected text.
    */  
    public void showStatistics() {
	TextAnalyzer analyzer = new TextAnalyzer();
        TextAreaPlus ta = getTextAreaWithFocus();
        if (ta != null) {
	    analyzer.setup(ta.getText());
	}	    
    }

   /**
    * Opens a separate frame and displays certain useful statistics
    *  for the selected text.
    */  
    public void getEnglishFacts() {
	EnglishFacts analyzer = new EnglishFacts();
	analyzer.run();
    }


   /**
    * Lets user search for pattern words.
    */  
    public void getPatternWords() {
	PatternWordSearcher analyzer = new PatternWordSearcher();
	//	analyzer.run();
    }

   /**
    * Opens a separate frame and displays certain useful statistics
    *  for the selected text.
    */  
    public void affineAnalyzer() {
	AffineAnalyzer analyzer = new AffineAnalyzer();
        TextAreaPlus ta = getTextAreaWithFocus();
        if (ta != null) {
	    analyzer.setup(ta.getText());
	}	    
    }

   /**
    * Opens a separate frame and displays certain useful statistics
    *  for the selected text.
    */  
    public void caesarAnalyzer() {
	CaesarAnalyzer analyzer = new CaesarAnalyzer();
        TextAreaPlus ta = getTextAreaWithFocus();
        if (ta != null) {
	    analyzer.setup(ta.getText());
	}	    
    }

   /**
    * Opens a separate frame and displays certain useful statistics
    *  for the selected text.
    */  
    public void substitutionAnalyzer() {
	CryptogramAnalyzer analyzer = new CryptogramAnalyzer();
        TextAreaPlus ta = getTextAreaWithFocus();
        if (ta != null) {
	    analyzer.setup(ta.getText());
	}	    
    }
   /**
    * Opens a separate frame and displays certain useful statistics
    *  for the selected text.
    */  
    public void vigenereAnalyzer() {
	VigenereAnalyzer analyzer = new VigenereAnalyzer();
        TextAreaPlus ta = getTextAreaWithFocus();
        if (ta != null) {
	    analyzer.setup(ta.getText());
	}	    
    }
    
   /**
    * Handles all menu selections for the encode, decode and set key buttons.
    */  
    public void actionPerformed(ActionEvent e) {
        KeyDialog keyDialog = null;
        if (e.getSource() == encodeButton) {
            try {
                cipher.init(key);
                outputField.setText(cipher.encrypt(inputField.getText()) );  //Encrypt
            }
            catch (Exception ee) {
                System.out.println("Exception: " + ee.getMessage());
                ee.printStackTrace();
            } // END: try
        } else if (e.getSource() == decodeButton) {
            try {
                cipher.init(key);
                outputField.setText(cipher.decrypt(inputField.getText()) );  //Decrypt
	        } 
    	    catch (Exception ee) {
                System.out.println("Exception: " + ee.getMessage());
                ee.printStackTrace();
            }  // END: try                
        } else if (e.getSource() == keyButton) {
           if (key == null) {
              try {  
                 keywordString  = (String) Class.forName(keyClassName).getDeclaredField("DEFAULT_KEYWORD_STRING").get(null);
                 keyPromptString = (String) Class.forName(keyClassName).getDeclaredField("DEFAULT_KEY_DESCRIPTOR_PROMPT_STRING").get(null);
              } catch(Exception ee) { //specific catches can be implemented
                 System.out.println(ee.getMessage());
                 ee.printStackTrace();
              }   
           }
 //           System.out.println("Opening Key Dialog");
           if (keyDialog == null) 
	       keyDialog = new KeyDialog(this, cipher.getAlphabetRangeOptions(), 
				 "Enter " + cipher.getAlgorithm() + " Key and Choose Alphabet Parameters");
           keyDialog.setKeyText(keywordString);
           int dialogOption = keyDialog.showKeyDialog();	        

   //	   cryptoTool.showStatus("Returned from key dialog " + dialogOption);
	   //	   requestFocus();
	   //	   cryptoTool.showStatus("Requested focus");

           if (dialogOption == KeyDialog.APPROVE_OPTION) {
//	       cryptoTool.showStatus("Approved Key");
	       try {  
		 key = HistoricalKey.getInstance(cipher.getAlgorithm(), cipher.getProvider());
		 key.init(keyDialog.getKeyAreaText() + "/" + keyDialog.getSelectedAlphabetChoice());
		 if (key != null) {
		     encodeButton.setEnabled(true);
		     decodeButton.setEnabled(true);
		 } 
//		 cryptoTool.showStatus("Done setting buttons");
             } catch(Exception ex) {
		 ex.printStackTrace();
	     }
          } //end if APPROVE_OPTION          
       }
    } //actionPerformed()    
    
    
   /**
    * Returns a reference to the TextArea that currently has focus.
    */  
    public TextAreaPlus getTextAreaWithFocus() {
        if (inputField.isInFocus()) 
            return inputField;
        else if (outputField.isInFocus()) 
            return outputField;
        else 
            return null;
    }
        
   /**
    *  Closes this window, first saving the text in the TextAreas if necessary.
    */  
    public boolean close() {
        if (!saveTextInTextAreas(false))
            return false;
        setVisible(false);
        WindowManager.removeWindow(this);
        dispose();
        return true;    
    }
    
   /**
    *  Implementation of the Save and SaveAs commands.
    */  
    public void save(boolean rename) {
        TextAreaPlus tap = getTextAreaWithFocus();
        if (tap == null)
            return;
        if (rename || tap.isDirty())
	    //            tap.save(rename);
            tap.save(encodingField.getText(), rename);
    }
    
   /**
    *  Sets this window's id number.
    */  
    public void setWindowId( int n ) {
        windowId = "" + n;
    }
   /**
    *  Gets this window's id number.
    */  
    public String getWindowId() { return windowId; }

   /**
    *  Causes this window to be the front window.
    */  
    public void activate() {
        isActivated = true;
//        cryptoTool.showStatus("CipherFrame" + windowId + " activated");
        this.toFront();
        this.setMenuBar(cryptoTool.getMenus().getMenuBar());
	Menus.updateMenus();
    }
    
    public void windowActivated(WindowEvent e) {
        isActivated = true;
	this.toFront();
        this.setMenuBar(cryptoTool.getMenus().getMenuBar());
	Menus.updateMenus();
//        activate();
    }
	
    public void windowDeactivated(WindowEvent e) {
        isActivated = false;
    }

   /**
    *  Closes this window and is subsidiary windows, if any.
    */  
    public void windowClosing(WindowEvent e) {
        cryptoTool.doCommand("Close Cipher...");
	//        if (statsframe != null) {
	//            statsframe.dispose();
	//            statsframe = null;
	//        }
    }

    // Interface methods
    public void windowClosed(WindowEvent e) {}
    public void windowDeiconified(WindowEvent e) {}
    public void windowIconified(WindowEvent e) {}
    public void windowOpened(WindowEvent e) {}    

    public void lostOwnership (Clipboard clip, Transferable cont) {}
} // CipherFrame
