/*
 * File: CryptoToolJ.java
 * @author R. Morelli <ralph.morelli@trincoll.edu>
 * 
 * <P>Description: Implements a Java 1.1 version of a GUI tool for using and analyzing
 *  historical ciphers.

 * <P>Credits: This program is modelled after the ImageJ program which is written by
 *  Wayne Rasband of the National Institutes of Health. ImageJ is in the public domain.
 *

 * <P>Copyright: This program is in the public domain. You can modify it as you 
 *  see fit as long as you properly acknowledge its original authors (Morelli
 *  and Rasband). It would also be nice if you forwarded your changes to 
 *  <A HREF= "mailto:ralph.morelli@trincoll.edu">ralph.morelli@trincoll.edu</A> so 
 *  they can possibly be added to the "official" version.
 */
 
package cryptotoolj;

import hcrypto.cipher.*;
import hcrypto.provider.*;
import hcrypto.engines.*;
import hcrypto.analyzer.*; // New Packager for Analyzers

import analyzers.*;

import java.io.*;
import java.awt.*;
import java.awt.event.*;   
import java.util.*;
import java.awt.datatransfer.*;

/** 
  This class is the top-level window for the CryptoToolJ application. It
  sets up the user interface, loads the various built-in cipher engines
  and plugin cipher engines, and manages control of the application 
  throughout its execution.

  <P>This class implements the main window and menu system for the
  application.  When the user creates a new file or opens an existing
  file, CryptoToolJ opens a separate window, which is implemented by
  the <A href="CipherFrame.html">CipherFrame</A> class.

  <P>See also
  <BLOCKQUOTE>
      <BR><A href="CipherFrame.html">CipherFrame</A>
      <BR><A href="Menus.html">Menus</A>
  </BLOCKQUOTE>
*/


public class CryptoToolJ extends Frame implements ActionListener, WindowListener,
                                       ClipboardOwner
{

    private java.applet.Applet applet;
	
    private Menus menus;
    private CipherFrame cf;         // Current CipherFrame

    public TextArea display;// = new TextArea();

   /**
    * Creates a CryptoToolJ. Set applet to null to run as an application.
    *  This constructor method sets up the user interface and installs
    *  the various built-in and plug-in ciphers.
    * @param applet, a reference to an applet.
    */
    public CryptoToolJ(java.applet.Applet applet) { 
	super("CryptoToolJ");
        this.applet = applet;

	if (applet == null)
	    System.getProperties().list(System.out); // List the system properties
        
        display = new TextArea();
        display.setEditable(false);
        add("Center", display);
        display.setText("Starting CryptoToolJ v1.3\n");
        if (applet != null)
            display.setText("Applet Version: Plugins are not available.\n Applet Version: File I/O is disabled\n");
        addWindowListener(this);

        menus = new Menus(this, applet);
        menus.addMenuBar();
        WindowManager.init(this, applet);
        
        Provider.addProvider(new DefaultProvider("Default")); // Add default and built-in providers
        Provider.addProvider(new RamProvider("Ram"));

	// Install plugin providers for the application version only
        if (applet != null)
            System.out.println("Applet version does not support plugins");
        else
	    loadPluginProviders();

	//        setResizable(true);
        Point loc = getPreferredLocation();
//		Dimension tbSize = toolbar.getPreferredSize();
        setBounds(loc.x, loc.y, 400, 200);
        setVisible(true);
        requestFocus();
    } 
    
    /**
     * loadProviders() loads both the Default and Ram providers as well as
     *  any providers it finds in the (System Dependent) provider directory.
     *  Plugin providers must belong to the providers package.
     */      
    private void loadPluginProviders() {
        
//        String pluginDir = Tools.getPluginDirectory();      
        String className="";
        try {
            String packagePrefix = Tools.getProviderPackageName() + ".";  
            File file = new File(Tools.getProviderDirectory());
            String providers[] = file.list();
            if (providers != null) 
                for (int k = 0; k < providers.length; k++) {
                    className = providers[k];
                
                    if (className.endsWith("Provider.class")) {
                        className = packagePrefix  + className.substring(0, className.indexOf("."));
                        display.append("Adding plugin provider: " + className + "\n");    
//                        display.append("Name: " + Class.forName(className).getName() + "\n");                    
//                        System.out.println("Adding provider " + className);
//                        System.out.println("Name: " + Class.forName(className).getName());
                        Class paramtypes[] = { Class.forName("java.lang.String") };
                        java.lang.reflect.Constructor constr = Class.forName(className).getConstructor(paramtypes);
                        Object constargs[] = { new String("Plugin") };
                        Provider.addProvider((Provider)constr.newInstance(constargs));
                    }
                }
            else {
                display.append("No providers found\n");
            }
		}
		catch (ClassNotFoundException e) {
		    showStatus("Exception: Plugin not found: " + className);
		}
		catch (InstantiationException e) {
		    showStatus("Exception: Unable to load plugin (Instantiation)");
		}
		catch (IllegalAccessException e) {
		    showStatus("Exception: Unable to load plugin (Access)");
		}
        catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
            e.printStackTrace();
        }
        
//        display.append("Ciphers loaded: " + Provider.getCipherNames() + "\n");
        display.append("Ciphers loaded: \n");
	StringTokenizer st = new StringTokenizer(Provider.getCipherNames(),",");
	while (st.hasMoreTokens()) {
	    display.append(" > "  + st.nextToken() + "\n");
	}
    } // addProviders()

    /**
     * Calculates a preferred location for the main application window
     *  based on the screen size.
     */
    public Point getPreferredLocation() {
        int screenWidth = Toolkit.getDefaultToolkit().getScreenSize().width;
        int windowWidth = 500;//tbsize.width+10;
        double percent;
        if (screenWidth > 832)
            percent = 0.8;
        else
            percent = 0.9;
        int windowX = (int)(percent * (screenWidth - windowWidth));
        if (windowX < 10)
            windowX = 10;
        int windowY = 32;
        return new Point(windowX, windowY);
    }

    /** 
     * Handles all menu events whenever the user selects a menu item.
     */
    public void actionPerformed(ActionEvent e) {
	//        System.out.println("Action Event" + e.paramString());
        if ((e.getSource() instanceof MenuItem)) {
             MenuItem item = (MenuItem)e.getSource();
             String cmd = e.getActionCommand();
             if (((Menu)item.getParent()).getLabel().equals("Analysis"))
                 doAnalyzerCommand(cmd);             
             else // if (cmd != null)
                 doCommand(cmd);
        }
    }
	
    /**
     * doAnalyzerCommand() does both built-in and plugin analyzers. Plugin analyzers
     *  are assumed to be in the analyzer package.
     * @param cmd is the menu command
     * All analyzers implement the Analyzer interface. They should simply
     *  be setup() and then run(). 
     */
    public void doAnalyzerCommand(String cmd) {
	    CipherFrame cf = (CipherFrame)WindowManager.getActiveCipherFrame();
	    if (cf == null)
	        return;      // Return if a cipher frame is not selected
	    
//        showStatus("Analyzing " + cmd);
        if (cmd.indexOf("Index Of Coincidence") != -1) {
             Analyzer analyzer = new IndexOfCoincidence();
             analyzer.setup(cf.getTextAreaWithFocus().getText());
             analyzer.run();
             display.append(analyzer.getReport());
        } 
        else if (cmd.indexOf("Text Analyzer") != -1) {
//            showStatus("Text Analyzer");
            cf.showStatistics();
	} 
        else if (cmd.indexOf("English FAQ") != -1) {
            cf.getEnglishFacts();
	} 
        else if (cmd.indexOf("Pattern Word Searcher") != -1) {
            cf.getPatternWords();
	} 
	else if (cmd.indexOf("Affine Analyzer") != -1) {
//            showStatus("Affine Analyzer");
            cf.affineAnalyzer();
        } 
	else if (cmd.indexOf("Caesar Analyzer") != -1) {
//            showStatus("Caesar Analyzer");
            cf.caesarAnalyzer();
        } 
	else if (cmd.indexOf("Simple Substitution Analyzer") != -1) {
//            showStatus("Simple Substitution Analyzer");
            cf.substitutionAnalyzer();
        } 
	else if (cmd.indexOf("Vigenere Analyzer") != -1) {
//            showStatus("Vigenere Analyzer");
            cf.vigenereAnalyzer();
        } 
	else {                        // Drop through logic: It must be a plugin
            try {
                String className = Tools.getAnalyzerPackageName() + "."  + cmd;
                Analyzer a = (Analyzer)Class.forName(className).newInstance();
		//                a.setup(cf.getTextAreaWithFocus(), display);
                a.setup(cf.getTextAreaWithFocus().getText());
                a.run();
                display.append(a.getReport());
            } 
		    catch (ClassNotFoundException e) {
		        showStatus("Exception: Analyzer not found: " + cmd);
		    }
		    catch (InstantiationException e) {  
		        showStatus("Exception: Unable to load analyzer (Instantiation)");
		    }
		    catch (IllegalAccessException e) {
		        showStatus("Exception: Unable to load analyzer (Access)");
		    }
         }    
    }

    /** 
     * Executes the command given by its parameter.
     * @param a String that specifies the desired command.
     */
    public void doCommand(String cmd) {   
        CipherFrame cf = (CipherFrame)WindowManager.getActiveCipherFrame();
//    	showStatus("doCommand " + cmd);
        if (cmd.equals("About CryptoToolJ...")) 
            showAboutBox();
        if (cmd.equals("New Cipher...")) {
//            showStatus("New Cipher...");
            cf = new CipherFrame(this);
            WindowManager.addWindow(cf);
        }
        else if (cmd.equals("Open File..."))  {
//            showStatus("Open File...");
            if (cf == null) {
                cf = new CipherFrame(this);
                WindowManager.addWindow(cf);
            }
            cf.openFile();	
        } 
        else if (cmd.equals("Close Cipher...")) {   
//            showStatus("Close Cipher...");
            if (cf != null) {
                cf.close();
            }
        }
        else if (cmd.equals("Save")) {
//            showStatus("Save Cipher...");
            if (cf != null) 
                cf.save(false);
        }
        else if (cmd.equals("Save As")) {
//            showStatus("Save As Cipher...");
            if (cf != null) 
               cf.save(true);
        }
        else if (cmd.equals("Print"))
            showStatus("PrintCipher... not yet implemented");
        else if (cmd.equals("Cut")) {
//            showStatus("Cut");
            TextManager.copyText(true);
        }
        else if (cmd.equals("Copy")) {
//            showStatus("Copy");
            TextManager.copyText(false);
        }
        else if (cmd.equals("Paste")) {
//            showStatus("Paste");
            TextManager.pasteText();
        }
        else if (cmd.equals("Select All")) {
//            showStatus("Select All");
            TextManager.selectAll();
	}
        else if (cmd.equals("CryptoToolJ")) {
            activate();
        }
        else if (cmd.indexOf("CipherFrame") != -1) {
            WindowManager.activateWindow(cmd);
        } 
        else if (cmd.equals("Quit"))
            this.quit();
     }    

    /**
     *  Displays the "About CryptoToolJ" blurb.
     */
     void showAboutBox() {   
//	    display.append("About CryptoToolJ\n");
	MessageDialog d = new MessageDialog(this, "About...", 
		"         CryptoToolJ " + "v1.3" + "\n" +
		" \n" +
		"Ralph Morelli (ralph.morelli@trincoll.edu)\n" +
		"Trinity College, Hartford, CT, USA\n" +
		" \n" +
		"CryptoToolJ is in the public domain");
    }
	
    /**
     *  Provides a controlled quit of the program for either
     *   its applet or application versions.
     */
    private void quit() {  
        display.append("Quitting\n"); 
        if (!WindowManager.closeAllWindows()) {
            display.append("Canceled\n"); 
            return;            // If any FileSave dialog is canceled, don't quit
        }
	if (applet == null) {
            System.exit(0);
        }
        display.append("Setting invisible\n"); 
        setVisible(false);  
        applet.destroy();
    }
    
    /**
     *  Returns a reference to the programs Menus object.
     */
    public Menus getMenus() {
        return menus;
    }

    /**
     *  Brings the top-level CryptoToolJ frame to the front.
     */
    private void activate() {
//       showStatus("CryptoToolJ Window activated");
       this.toFront();	
       this.setMenuBar(menus.getMenuBar());
       Menus.updateMenus();
    }
	
    /**
     *  Responds to a window closing event.
     */
    public void windowClosing(WindowEvent e) {
        quit();
    }
    public void windowActivated(WindowEvent e) { 
	this.toFront();
	this.setMenuBar(menus.getMenuBar());
	Menus.updateMenus();
//	activate(); 
    }
    public void windowClosed(WindowEvent e) {}
    public void windowDeactivated(WindowEvent e) {}
    public void windowDeiconified(WindowEvent e) {}
    public void windowIconified(WindowEvent e) {}
    public void windowOpened(WindowEvent e) {}

    /**
     *  Displays a message on the CryptoToolJ's display.
     */
    public void showStatus(String msg) {
        display.append(msg + "\n");
    }
     
    public void lostOwnership (Clipboard clip, Transferable cont) {}

    /**
     *  Returns true iff this is an applet version.
     */
    public boolean isApplet() {
        return applet != null;
    }

    /**
     *  Creates an instance of CryptoToolJ and when run in application mode.
     */
    public static void main(String args[]) {  //main method
        CryptoToolJ cda = new CryptoToolJ(null); 
    }  //end main()
}
