/* *****************************************************************
 * File: MakeAlbertiFiles.java
 * Author: R. Walde and R. Morelli
 * Description: This command-line program lets you create randomly
 *  encrypted messages using easy or hard Alberti encryption (Type I or Type II).
 *  The files created have the following format:
 * 
 *       Encrypted message without line breaks
 *       $$$
 *       Original plaintext message without line breaks
 *       #permutationkey,shiftkeyword/alphabettype(shiftkeylength)#
 * 
 *  Here's an example of a 50 character message:
 *
 *       awnpf dvg krptxwfa wbwf dvg jrpzvrfgic wkizn ne gic
 *       $$$
 *       etext you indicate that you understand agree to and
 *       #grkpafcoulqemjwhsibxtvnzdy,aznqi/az(5)#
 * 
 * To compile: 
 *   javac -classpath ../../classes:. MakeAlbertiFiles.java
 * 
 * To run:
 *   java -classpath ../../classes:. MakeAlbertiFiles <command line arguments>
 * Example:
 *   java -classpath ../../classes:. MakeAlbertiFiles bookfile filenameroot 25 25 125 20 5 hard");
 *
 * <command-line arguments> : <bookfile filenameroot low incr high number shiftlen [hard]>
 *   where
 *    bookfile     -- is the name of a text file containing some book
 *    filenameroot -- the root to use as the name of the generated files and directory
 *    low          -- the minimum message size
 *    incr         -- the amount by which to vary the message size
 *    high         -- the maximum message size
 *    number       -- the number of each size message
 *    shiftlen     -- the length of the Alberti shift key
 *    hard         -- optional argument to force hard Alberti (Type II); the default is easy (Type I)
 *
 ************************************************************************************************** */

import hcrypto.cipher.*;
import hcrypto.provider.*;

import java.io.*;

public class MakeAlbertiFiles {

    private String shiftKeyword = "";
    private String keyspec;
    private int shiftKeyLen;
    private boolean hardAlberti = false;

    /**
     * randomAlberti() creates a random Alberti message (either Type I or Type II) of
     *  the input text.
     * @param inText -- a String giving the input text.
     * @param substitution -- an array giving the standard alphabet
     * @param maxChanged -- a count of the number of swaps to perform on the
     *  standard alphabet to create the permuted primary alphabet
     */
    public String randomAlberti(String inText, int[] substitution, int maxChanged){
	int temp;  //used for loops and swap

	for (int j = 0; j < maxChanged; j++){       // Randomize permutation alphabet
	    int k = (int)(Math.random()*maxChanged);
	    temp = substitution[j];
	    substitution[j] = substitution[k];
	    substitution[k] = temp;
	}//for

	// Use the randomized alphabet as the permuted primary key

	StringBuffer permKeyBuffer = new StringBuffer();
	for (int k = 0; k < 26; k++) {      // Alberti uses 26 letter alphabet
            permKeyBuffer.append((char)('a' + substitution[k]));
	}
	String permKey = permKeyBuffer.toString();

	// Create a random shift keyword of length shiftKeyLen

 	StringBuffer shiftKeyBuffer = new StringBuffer();
	for (int k = 0; k < shiftKeyLen; k++) {
	    shiftKeyBuffer.append( (char)('a' + (int)(Math.random() * 26)));
	}
	String shiftKey = shiftKeyBuffer.toString();    // Create the shift key
	shiftKeyword = shiftKey +"(" + shiftKeyLen +")";

	// Create a keyspec to be used for encryption

	try {
	    char [] azRange ={'a', 'z'};
	    Alphabet alph = new Alphabet(azRange);
	    keyspec = alph.eAlbNormedPermString(permKey, shiftKey) + "," + alph.normedShiftString(shiftKey) + "/az";
	} catch (Exception e) {
	    System.out.println("Problem with reducing permutation or shift keys");
	}
	System.out.println("keyspec = " + keyspec + "(" + shiftKeyLen + ")");

	// Encrypt the message using an Alberti Cipher Engine

        Provider.addProvider(new DefaultProvider("Default"));    // Create and register the provider classes
        Provider.addProvider(new RamProvider("Ram")); 
	String encryptStr="";
	try {
	    Cipher cipher = Cipher.getInstance("Alberti");   // Create an instance of the cipher and its corresponding key
	    HistoricalKey key = HistoricalKey.getInstance(cipher.getAlgorithm(), cipher.getProvider());
	    key.init(keyspec);        // Initialize the key using the keyspec
	    cipher.init(key);          // Get and use individual ciphers
	    if (hardAlberti) {
		System.out.print("Hard Alberti\t");
		encryptStr = cipher.decrypt(inText);
	    } else {
		System.out.print("Easy Alberti\t");
		encryptStr = cipher.encrypt(inText);
	    }
	} catch (Exception e) {
	}
        return encryptStr;
   } // randomAlberti

    /**
     * singleSpaces() converts its parameter into a single-spaced text.
     */
   private String singleSpaces(String inText){
        char thisChar, lastChar;
        StringBuffer outText = new StringBuffer();
        lastChar = inText.charAt(0);
        outText.append(lastChar);
        for ( int j = 1; j < inText.length()-1; j++){
            thisChar = inText.charAt(j);
            if (!((thisChar == ' ') && (lastChar == ' ')) ){
              outText.append(thisChar);
              lastChar = thisChar;
            }//if
        }//for
        return outText.toString();
   } // singleSpaces()

    /**
     *  writeCiphertextFiles() creates a collection of files in a directory named
     *   outRootName+"dir". There are numEachSize different files for each of a
     *   collection of different sizes. The different sizes start at minMessSize and
     *   the size is increased by gapMessSize repeatedly up to the largest such
     *   size which is less than or equal to maxMessSize. The size of the file is
     *   is determined by counting the non-space characters.  Additional characters
     *   are added to complete the last word. Each file consists of a cryptogram on
     *   one line, "$$$" on the next line and the plaintext on a third line.
     * @param sourceFile -- The book from which plaintext messages are selected.
     * @param outRootName -- a string giving the name of the root directory where
     *   the created files are stored.
     * @param minMessSize -- minimum message size
     * @param gapMessSize -- amount by 
     * @param maxMessSize -- maximimum message size
     * @param numEachSize -- the number of each sized message
     * @param shiftlen -- the length of the shift keyword for each message
     * @param alph -- the Alphabet to be used in the encryption
     * @param hard -- a boolean that that determines whether easy or hard Alberti
     *  (Type I or TYpe II) encryption is used.
     */
    public void writeCiphertextFiles(String sourceFile, String outRootName, int minMessSize, int gapMessSize, 
				     int maxMessSize, int numEachSize, int shiftlen, Alphabet alph, boolean hard) {
	shiftKeyLen = shiftlen;
	hardAlberti = hard;

	try{
	    StringBuffer pText = new StringBuffer(); // The palintext message
	    String pString;       // Will store the cleaned up version of pText
	    String cText;         // the ciphertext message
	    String line = null;
	    int len = 0;          // will store the length of line.
	    int pos = 0;          // position of char in line
	    char ch;              // Will point to char read from file.
	    int chNum = 0;        // Will count the number of acceptable chars.

	    boolean done = false;
	    boolean messDone = false;
	    FileWriter outStream;
	    File outDir = new File(outRootName + "dir");

	    // Create the output directory

	    if (outDir.mkdir())  
		System.out.println("Directory " + outRootName + "dir created");
	    else 
		System.out.println("Directory " + outRootName + "dir either already exists or failed");

	    File outFile;          // Used to reference files created.
	    BufferedReader inStream = new BufferedReader(new FileReader(sourceFile));
	    System.out.println("sourceFile= " + sourceFile + " opened");
	    int curMessSize =  minMessSize;
	    int curMessNum = 0;
	    int lineCount = (int)(Math.random()*50);
	    int[] substitution = new int[alph.getSize()];
	    for (int j = 0; j < alph.getSize(); j++){
		substitution[j] = j;
	    } // for

	    // Read a line from the source text and encrypt it

	    while (!done){
		line = inStream.readLine();
		if(line == null){   // If eof then close and reopen
		    inStream.close();
		    inStream = new BufferedReader(new FileReader(sourceFile));
		    System.out.println("sourceFile= " + sourceFile + " opened");
		    line = inStream.readLine();
		    // if there still is no line throw exception
		    if (line == null) {
			throw new Exception("Bad source file in TextUtilities.writeCiphertextFiles()");
		    } // if
		} // if
		lineCount--;
		len = line.length();
		if ((len > 0) && (lineCount < 0)){
		    line = line.toLowerCase();
		    pos = 0;
		    while ((pos < len) && !messDone){
			ch  = line.charAt(pos);
			if (alph.isInAlphabet(ch)){
                            chNum++;
                            pText.append(ch);
			} // if  ch is in alphabet
			else{ // if not in the alphabet it still might separate words
			    if (ch != '\'') pText.append(' ');
			} // else
			if ( (chNum >= curMessSize) && ((ch ==' ') || (ch == '\n')) ){
			    messDone = true;
			    outFile = new File(outDir, outRootName+curMessSize+"_"+curMessNum+".txt");
			    outStream = new FileWriter(outFile);
			    pString = singleSpaces(pText.toString());
			    cText = randomAlberti(pString,substitution,26);
			    outStream.write(cText);
			    outStream.write("\n$$$\n");
			    outStream.write(pString);
			    outStream.write("\n#"+keyspec+"(" + shiftKeyLen + ")"+"#\n");
			    outStream.close();
			} //if message is long enough
			pos++;
			if ((pos == len) && (messDone == false))  pText.append(' ');
		    } //while pos in line
		} // if a useable line

		// Reset variables before reading next line

		if (messDone) { 
		    messDone = false;
		    pText = new StringBuffer();
		    chNum = 0;
		    lineCount = (int)(Math.random()*50);
		    curMessNum++;
		    if (curMessNum >= numEachSize){
			curMessNum = 0;
			curMessSize += gapMessSize;
		    } //if
		    if (curMessSize > maxMessSize){
			done = true;
		    } //if
		} //if message wasdone
		line = inStream.readLine();
	    } // while not done
	    // textSize  = chNum;
	    System.out.println("Finished writing Files." );
            inStream.close();
        } // try
        catch(Exception exc){
            System.out.println("In TextUtilities.writeCiphertextFiles() - " +exc.toString());
	    exc.printStackTrace();
        } //catch
    } //writeCiphertextFiles

    /**
     * main() reads the command line and parses it. If correct, it then creates an instance of this
     *  class and uses that to generate a collection of randomly encrypted messages of a given cipher
     *  type.
     */
    public static void main(String[] args){
	if (args.length < 7) {
	    System.out.println("Usage: java MakeAlbertiFiles bookfile filenameroot low incr high number shiftlen [hard]");
	    System.out.println("Default is easy Alberti");
	    System.out.println("E.G: java MakeAlbertiFiles bookfile filenameroot 25 25 125 20 5 hard");
	    return;
	}
	boolean hardAlberti = args.length == 8;
	
	MakeAlbertiFiles maker = new MakeAlbertiFiles();
	try{
	    char[] arr = {'a','z',' ',' '};
	    Alphabet alph = new Alphabet(arr);
	    //	    maker.writeCiphertextFiles("oliver.txt","oliver",25,25,125,20,alph);
	    maker.writeCiphertextFiles(args[0],args[1],Integer.parseInt(args[2]),
				       Integer.parseInt(args[3]),
				       Integer.parseInt(args[4]),
				       Integer.parseInt(args[5]), Integer.parseInt(args[6]), alph, hardAlberti);
	}  //try
	catch(Exception exc){
	    System.out.println(exc.toString());
	} //catch
    }//main()
} // CryptoFileMaker
