/**
 * ExpertAnalyzer.java reads an encrypted file and runs a number of analyses
 *  one it. It first counts the letters in the file. It then computes the
 *  index of coincidence (IC). Based on the IC it performs either a 
 *  Vigenere analysis (if the file was encrypted using a polyalphabetic
 *  cipher) or simple substitution analysis (Caesar, Affine, or Digram analysis).

 *  Results: In either case, it displays the results of its analyses. Some of
 *   the results are simply printed to the console and others are displayed
 *   in separate windows.
 *
 * To compile and run this program using the hcryptoj classes with either
 *  an archive file or the classes directory itself, use one of the following
 *  commands, which assume that the classes or archive are located two 
 *  directories up:
 *
 *  To compile: javac -classpath ../../classes:. ExpertAnalyzer.java
 *  To compile: javac -classpath ../../hcrypto.jar:. ExpertAnalyzer.java
 *  To run:     java -classpath ../../classes:. ExpertAnalyzer cryptotext
 *  To run:     java -classpath ../../hcrypto.jar:. ExpertAnalyzer cryptotext
 */

import hcrypto.analyzer.*;
import hcrypto.cipher.*;
import hcrypto.provider.*;
import hcrypto.engines.*;

import analyzers.*;   // Plugin analyzers

import java.io.*;

public class ExpertAnalyzer {
   
    public static void main (String args[]) throws Exception {
        if (args.length < 1) {
            System.out.println("Usage: java ExpertAnalyzer infile [encoding]");
            return;
        }

       	// ----------- Read the file. ----------------------------

        File file = new File(args[0]);
        InputStreamReader inStream;
        if (args.length==2)
   	    inStream = new InputStreamReader(new FileInputStream(file), args[1]);
        else
   	    inStream = new InputStreamReader(new FileInputStream(file));
	int length = (int)file.length();            // Read the data
	char[] input = new char[length];
	inStream.read(input);
	inStream.close();

       // ---------- Analyze the letter frequencies --------------

        String text = new String(input);           // Get the text from the file
        TextStatistics stats = new TextStatistics(text, true);

        java.text.NumberFormat nf = java.text.NumberFormat.getNumberInstance();
        nf.setMaximumFractionDigits(3);         
        System.out.println("Cryptanalyzer Report");
        System.out.println(stats.getReport());
        System.out.println(stats.getFrequencyReport());

	// --------- Calculate the index of coincidence ------------

        Analyzer analyzer = new IndexOfCoincidence();
        analyzer.setup(text);
        analyzer.run();
        System.out.println(analyzer.getReport());

	double ci = ((IndexOfCoincidence)analyzer).getIOC();
	System.out.println("The index of coincidence = " + nf.format(ci));

	// -------- Test for Vigenere, Caesar, Affine, and Simple Substition --

        if (  ci < 0.056 ) {
             System.out.println("The index of coincidence suggests this is polyalphabetic.");
             System.out.println("Doing Vigenere Analysis");
             analyzer = new VigenereAnalyzer(stats);
             analyzer.setup(text);
             analyzer.run();
             System.out.println(analyzer.getReport());
        }
        else { // if (ci > 0.06) {
             System.out.println("The index of coincidence suggests this is monoalphabetic.");
             System.out.println("Doing Caesar Shift Analysis");
             analyzer = new CaesarAnalyzer(stats);
             analyzer.setup(text);
             analyzer.run();
             System.out.println(analyzer.getReport());

             System.out.println("\nDoing Affine Analysis");
             analyzer = new AffineAnalyzer(stats);
             analyzer.setup(text);
             analyzer.run();
             System.out.println(analyzer.getReport());

             System.out.println("\nDoing Digram Analysis");
             analyzer = new DigramAnalyzer(stats);
             analyzer.setup(text);
             analyzer.run();
             System.out.println(analyzer.getReport());
        }
    }

    /**
     * A method to read the named file.
     * @param file -- the file to be read.
     */
    private static String read(File file) throws IOException {
       FileInputStream inStream = new FileInputStream(file);
       int length = (int)file.length();            // Read the data
       byte[] input = new byte[length];
       inStream.read(input);
       inStream.close();
       return new String(input);
    }



}

