/* **********************************************************
 * File: TestEvaluationMeasure.java 
 * Author: R. Morelli
 * Description: This program runs experiments to test 
 *  various evaluation measures, such as IC and Ngram 
 *  frequencies.
 * 
 * To Compile:  javac -classpath ../../classes/:. TestEvaluationMeasure.java
 * To Run: java -classpath ../../classes/:. TestEvaluationMeasure <CommandLineArgs>
 * 
 *  <CommandLineArgs> consists of the following items:
 *    cipher -- The name of the cipher, e.g. Alberti
 *    keyspec -- The correct keyspec for the file, e.g., kcrmpngoidztujsafqexbyvlhw,avfjxrypgb/az
 *    textfile -- The file containing the encrypted message and its decryption
 *    book -- The book used to initialize the NgramAnalyzer, e.g., tom.txt (Tom Sawyer)
 *    NN -- The number of characters to use in the NGramAnalyzer (NN = 2, 3 or 4)
 *    skip -- How many letters to skip in the text (skip  = 1 means every letter, 2 means every other letter)
 *
 * Example: java -classpath ../../classes/:. TestEvaluationMeasure Alberti  \ 
 *        kcrmpngoidztujsafqexbyvlhw,avfjxrypgb/az experiments_TypeII/hard10shiftdir/hard10shift500_0.txt \
 *        book.txt 4 1 
 */

import hcrypto.analyzer.*;
import hcrypto.cipher.*;
import hcrypto.engines.*;
import hcrypto.provider.*;
import java.io.*;
import java.text.*;

public class TestEvaluationMeasure {

    public static NumberFormat num;

    public static void main (String args[]) {
        if (args.length < 6) {
	    System.out.println("Usage: java -classpath ../../classes:. TestEvaluationMeasure cipher keyspec textfile book NN skip");
            return;
	}
	num = NumberFormat.getInstance();
	num.setMaximumFractionDigits(3);

	// Read the file and convert to a string
	String text = "", ciphertext = "";
	String solution = "";
	try {
	    File oFile = new File(args[2]);
	    InputStreamReader iStream = new InputStreamReader(new FileInputStream(oFile));
	    int length = (int)oFile.length();            // Read the data
	    char[] input = new char[length];
	    iStream.read(input);
	    iStream.close();
	    text = new String(input);

	    int indx = text.indexOf("$$$");
	    if (indx != -1) {
		ciphertext = text.substring(0, indx);
		solution = text.substring(indx+4);
		solution = solution.substring(0,30);
	    }
	    System.out.println("Text= " + ciphertext.substring(0,30) + " Solution= " + solution);

	    // Create and register the provider classes
	    Provider.addProvider(new DefaultProvider("Default")); 
	    Provider.addProvider(new RamProvider("Ram")); 

	    // Get the command line arguments and construct the key name
	    
	    String cipherName = args[0];
	    String keyspec = args[1];
	    String book = args[3];
	    int NN = Integer.parseInt(args[4]);
	    int skip = Integer.parseInt(args[5]);
 
	    // Create an instance of the cipher and its corresponding key

	    Cipher cipher = Cipher.getInstance(cipherName);
	    HistoricalKey key = HistoricalKey.getInstance(cipher.getAlgorithm(), cipher.getProvider());

	    // Initialize the key using the keyspec
	    key.init(keyspec);

	    // Get and use individual ciphers
	    cipher.init(key);
	    System.out.println("Algorithm= " + cipher.getAlgorithm());
	    System.out.println("Provider= " + cipher.getProvider());

	    keyExperiment(ciphertext, solution, cipher, key, keyspec, book, true, NN, skip); // True means permutation key, false is shift
	    keyExperiment(ciphertext, solution, cipher, key, keyspec, book, false, NN, skip); // True means permutation key, false is shift

	} catch(Exception e)  {}
    }

    private static void keyExperiment(String text, String solution,  Cipher cipher, HistoricalKey key, String keyspec, String book, boolean permutetest, int NN, int skip) {
	IndexOfCoincidence ic = null;
	String decrypt = null;
	String permKey = keyspec.substring(0, keyspec.indexOf(","));
	String shiftKey = keyspec.substring(keyspec.lastIndexOf(",")+1, keyspec.indexOf("/"));
        System.out.println("Permkey= " + permKey  + " Shiftkey= " + shiftKey);

	NgramArray ngramArr = null;
	try {
	    ngramArr = new NgramArray(NN, book, AlphabetFactory.getInstance(AlphabetFactory.ALPH_az), skip);
	} catch (Exception e) {
	    System.out.println("Problem with NgramArray: " + e.getMessage());
	}
	System.out.println("NN= " + ngramArr.getNN() + "\tStep= " + ngramArr.getStepSize());

	String initPermKey = new String(permKey);
	String initShiftKey = new String(shiftKey);
	
	double solErrsBySwap[] = new double[27];
	double keyErrsBySwap[] = new double[27];
	double shiftErrsBySwap[] = new double[27];
	double solutionErrs[] = new double[31];
	double solErrsIC[] = new double[31];
	double solErrsNGA[] = new double[31];
	double sumIC[] = new double[25];
	double sumNGA[] = new double[25];
	double ngramByWrongs[] = new double[27];
	double icByWrongs[] = new double[27];
	int nwrong[] = new int[27];   

	double rdist = 0;

	if (permutetest) {
	    System.out.println("Permutation keyword is varied, k = number of swaps");
	}
	else {
	    System.out.println("Shift keyword is varied, k = number of letters replaced");
	}

	int bound = 0;
	int ntrials = 40;
	for (int m = 0; m < ntrials; m++) {
	    permKey = new String(initPermKey);
	    shiftKey = new String(initShiftKey);
	    String newkeyspec="";

	    if (permutetest)
		bound = 25;
	    else
		bound = shiftKey.length()+1;

	    for (int k = 0; k < bound; k++) {
		if (cipher.getAlgorithm().equals("TypeIV"))
		    newkeyspec = permKey + "," + permKey + "," + shiftKey + "/az";
		else
		    newkeyspec = permKey + "," + shiftKey + "/az";
		try {
		    //		    System.out.println("Keyspec = " + newkeyspec);
		    key.init(newkeyspec);
		    cipher.init(key);
		    if (cipher.getAlgorithm().equals("Alberti"))
			decrypt = cipher.encrypt(text);
		    else
			decrypt = cipher.decrypt(text);
		    //		    System.out.println("Decr= " + decrypt.substring(0,30) + "\nSolu= " + solution.substring(0,30));
		    ic = new IndexOfCoincidence(decrypt);
		    rdist = ngramArr.recipDistSkip(decrypt);
		} catch (Exception e) {}
		
		double icVal = ic.getIOC();

		int solWrongs = countWrongs(decrypt.substring(0,30), solution.substring(0,30));
		solErrsBySwap[k] += solWrongs;

		++solutionErrs[solWrongs];
		solErrsIC[solWrongs] += icVal;
		solErrsNGA[solWrongs] += rdist;

		int wrongs = countWrongs(permKey, initPermKey);
		keyErrsBySwap[k] += wrongs;
		shiftErrsBySwap[k] += countWrongs(shiftKey, initShiftKey);
		++nwrong[wrongs];
		ngramByWrongs[wrongs] += rdist;
		icByWrongs[wrongs] += icVal;
		sumIC[k] += icVal;
		sumNGA[k] += rdist;
		if (permutetest)
		    permKey = swap(permKey, (int)(Math.random() * 26), (int)(Math.random() * 26));
		else
		    shiftKey = replace(shiftKey, (int)(Math.random() * shiftKey.length()), (int)(Math.random() * 26));
	    }
	}

	System.out.println("Averages of " + ntrials + " trials. Solution errors out of 30 characters.");
	System.out.println("k " + "\tic " + "\tnga " + "\tavgSolErrs " + "\tavgKeyErrs " + "\tavgShiftErrs");
	for (int k = 0; k < bound; k++) {
	    //	for (int k = 0; k < shiftKeywd.length(); k++) {
	    System.out.println(k +  "\t" + num.format(sumIC[k]/ntrials) + "\t" + num.format(sumNGA[k]/ntrials) 
			       + "\t" + num.format(solErrsBySwap[k]/ntrials) + "\t" + num.format(keyErrsBySwap[k]/ntrials) 
			       + "\t" + num.format(shiftErrsBySwap[k]/ntrials));
	}
	System.out.println("Breakdown by number of wrongs in key");
	System.out.println("k " + "\tnWrong " + "\tnga " + "\tic");
	for (int k = 0; k < nwrong.length; k++) {
	    if (nwrong[k] != 0)
		System.out.println(k + "\t" + nwrong[k] + "\t" + num.format(ngramByWrongs[k]/nwrong[k]) + "\t" + num.format(icByWrongs[k]/nwrong[k]));
	}
	System.out.println("Breakdown by number of wrongs in solution");
	System.out.println("k " + "\tnWrong " + "\tnga " + "\tic");
	for (int k = 0; k < solutionErrs.length; k++) {
	    if (solutionErrs[k] != 0)
		System.out.println(k + "\t" + solutionErrs[k] + "\t" + num.format(solErrsNGA[k]/solutionErrs[k]) + "\t" + num.format(solErrsIC[k]/solutionErrs[k]));
	}

    }

    private static String swap(String s, int j, int k) {
        StringBuffer sb = new StringBuffer(s);
	if (j == k) {
	    if (j != 0)
		j--;
	    else
		j++;
	}
	
	    
	char ch = sb.charAt(j);
	sb.setCharAt(j, sb.charAt(k));
	sb.setCharAt(k, ch);
	return sb.toString();
    }

    private static String replace(String s, int j, int k) {
        StringBuffer sb = new StringBuffer(s);
	char ch = (char)('a' + ((sb.charAt(j) - 'a' + k) % 26));
	sb.setCharAt(j, ch);
	return sb.toString();
    }

    private static int countWrongs(String s1, String s2) {
	if (s1.length() != s2.length())
	    return Math.max(s1.length(), s2.length());
	int sum = 0;
	for (int k = 0; k < s1.length(); k++)
	    if (s1.charAt(k) != s2.charAt(k))
		++sum;
	return sum;
    }

} // TestEvaluationMeasure

