/**
 * TestGaAnalyzer.java runs any GaAnalyzer class given the fully
 *  qualified class name (package.name) as its first command-line
 *  parameter and the name of a file containing sets of parameter
 *  settings as the second command-line argument. Each line of the
 *  parameters file represents parameter settings for a single
 *  experiment.
 *  
 * To compile and run with a jarfile containing the hcrypto classes:
 *
 * To compile: javac -classpath jarfile.jar:. TestGaAnalyzer.java
 * To run:     java -classpath jarfile.jar:. TestGaAnalyzer analyzers.GaClassName ParamFile
 *
 * To compile and run from the classes directory:
 *
 * Compile: javac -classpath ../../classes:. TestGaAnalyzer.java
 * Run:     java -classpath  ../../classes:. TestGaAnalyzer analyzers.GaClassName ParamFile

 *  A sample parameter file:
 *   file=../../cgrams/tobe.txt     crossrate=0.75 mutaterate=0.1 generations=100
 *   file=../../cgrams/appendix.txt crossrate=0.75 mutaterate=0.1 generations=100 */

import hcrypto.analyzer.*;
import hcrypto.cipher.*;
import hcrypto.provider.*;
import hcrypto.engines.*;

import analyzers.*;   // Plugin analyzers

import java.io.*;
import java.util.*;

public class TestGaAnalyzer {
   
    public static void main (String args[]) throws Exception {
        if (args.length < 2) {
            System.out.println("Usage: java TestGaAnalyzer analyzers.GaClassName ParamFile");
            System.out.println("E.G: TestGaAnalyzer analyzers.WordBasedGaAnalyzer params.txt");
            System.out.println("See: classes/analyzers for available GA analyzers.");
            System.out.println("See: params.txt for an example of a parameter file.");
            return;
        }
	readAndProcess(args[0], args[1]);   // Read a list of parameter values and run each experiment
    }

    /**
     * readAndProcess() takes a file containing multiple lines of parameter settings
     *  and conducts one GA experiment for each line.
     * @param filename -- the name of the parameter file
     */
    private static void readAndProcess(String className, String filename) throws IOException {
	BufferedReader instream = new BufferedReader(new FileReader(filename));
	StringTokenizer st = null;
	String line = instream.readLine();
	while (line != null) {
	    //	    System.out.println(line);
	    if (line.charAt(0) != '#') {
		doExperiment(className, new GaParameters(line), line);
	    }
	    line = instream.readLine();
	}
	instream.close();
    }

    /**
     * doExperiment() takes a space-delimited String representing a set of 
     *  parameter settings and conducts an GA experiment using those settings.
     *  When the experiment is done, a report is issued.
     * @param params --  a string of the form param1=value1 param2=value2
     */
    private static void doExperiment(String className, GaParameters params, String line) throws IOException {
	Analyzer analyzer = null;
        System.out.println("GaANALYZER: " + className);
	System.out.println("PARAMETERS: " + params.toString() );
        String text = read(new File(params.filename));

	// The following try block creates an Analyzer object given its fully qualified classname
	//  and calls its Analyzer(GaParameters) constructor with the name of a GaParameters object.

	try {
	    Class paramtypes[] = { Class.forName("hcrypto.analyzer.GaParameters") };
	    java.lang.reflect.Constructor constr = Class.forName(className).getConstructor(paramtypes);
	    Object constargs[] = { params };
	    analyzer = (Analyzer)constr.newInstance(constargs);
	} catch (ClassNotFoundException e) {
	    System.out.println("Class not found " + className);
	} catch (InstantiationException e) {
	    System.out.println("Unable to instantiate class " + className);
	} catch (IllegalAccessException e) {
	    System.out.println("Illegal Access " + className);
	} catch (Exception e) {
	    System.out.println("Exception " + className);
	}
	    
	// Run the experiment and print the result.

        analyzer.setup(text);
        analyzer.run();
        System.out.println(analyzer.getReport());
    }

    /**
     * readFile() reads the file supplied as its parameter and returns its
     *  contents as a String.
     * @param file --  a text file containing a cryptogram
     * @return a string of the file's text
     */
    private static String read(File file) throws IOException {
       FileInputStream inStream = new FileInputStream(file);
       int length = (int)file.length();                       
       byte[] input = new byte[length];
       inStream.read(input);              // Read the data
       inStream.close();
       return new String(input);          // Convert data to string
    }
}

