/*
 * File: AlbertiAnalyzer.java
 * @author R. Morelli <ralph.morelli@trincoll.edu> 
 * 
 * Description: This class assumes that the text is
 *  encrypted with a polyalphabetic Alberti cipher. It uses the
 *  Index of Coincidence (IC) to cryptanalyze it. This version
 *  attacks the "easy" Alberti---that is, it assumes that each
 *  plaintext characters is first substituted from the permutation
 *  alphabet and then shifted.  In the easy version, the IC can
 *  be used to determine when the correct set of shifts have been
 *  found, as outlined in the algorithm below.
 *
 * The "hard" version of Alberti is really the decrypt function, which
 *  first shifts each plaintext character and then substitutes
 *  for it from the permutation alphabet.  In the hard case,
 *  each plaintext character is encrypted by a composition 
 *  consisting of a substitution and a shift. This is equivalent
 *  to a double substitution. It is not possible to use IC
 *  to identify the correct set of shifts. 
 * 
 * Algorithm: A three step algorithm is used:
 *  1. Find the length of the shift keyword, k, using the IC
 *  2. Break the text into k simple substitution subtexts and use IC 
 *     to determine the shift of each subtext. These correspond to
 *     the letters in the shift keyword.
 *  3. Shift each subtext and merge them into a single message that will
 *     now be a simple subsitution cipher. 
 *  4. Solve the simple substitution cipher using NgramAnalyzer or some
 *     other analyzer.
 *
 *  To compile and run from the TestCryptAnalyzer application:
 *
 *  cd ~crypto/hcryptoj/1.4/applications/testanalyzer
 *  javac -classpath ../../classes -d ../../classes AlbertiAnalyzer.java
 *  java -classpath ../../classes:. TestCryptAnalyzer analyzers.AlbertiAnalyzer ga_paramfiles/albertiparam.txt
 */

package analyzers;

import hcrypto.analyzer.*;
import hcrypto.cipher.*;
import hcrypto.engines.*;
import hcrypto.provider.*;

import java.util.*;
import java.io.*;

public class AlbertiAnalyzer extends CryptoAnalyzer implements Analyzer {
    private final int KEYWD_MAX = 34;
    private final int MAX_TRIES = 7;            // Step 2   Magic numbers determined experimentally
    private final double IC_EPSILON = 0.008;    // Step 2
    private final double BACKUP_EPSILON = 0.004;  // Step 2 How we decide which ICs to save
    private final int MAX_RETRIES = 15;         // Step 3
    //    private final double EVAL_EPSILON = 0.42;   // Step 3  How we know when text is decrypted
    private final double EVAL_EPSILON = 0.25; // 0.35;   // Step 3  How we know when text is decrypted
    private final double EVAL_DELTA = 0.0018;   // Step 3  How we know if a retry has improved decryption

    private IndexOfCoincidence IC;
    java.text.NumberFormat nf = java.text.NumberFormat.getNumberInstance();

    private String actualKeywd = "";  // The actual secret shift keyword 
    private int  actualKeywdLen;
    private String cleantext = "";
    private int shiftLen = 0;
    private int keyCount = 0;

    private TreeMap icBackupTree;  // Keeps a sorted collection of ICs 

    /**
     * AlbertiAnalyzer() -- Default constructor
     */
    public AlbertiAnalyzer() {
	super();
    }

    /**
     * AlbertiAnalyzer() -- this constructor is given an object containing parameter settings
     * @param params -- an object containing param1=val1 param2=val2 ...
     */
    public AlbertiAnalyzer(GaParameters params) {
	super(params);
    }

    /**
     * setup() sets up the text and the parameters.. 
     */
    public void setup(String text) {
	super.setup(text);
	if (params == null) {
	    params = new GaParameters(); // Default GaParameters
	}
    }

    /**
     * findShiftLength() determines the length of the Alberti shift key through
     *  an exhaustive search. For each possible length, up to KEYWD_MAX, we break
     *  the text into columns and compute the average IC of the columns.  The
     *  length with the greatest IC is returned.  As an optimization, we break
     *  out of the search when we find an IC within 0.008 of 0.066. This works well
     *  for shifts that are not prime numbers, whose multiples would frequently
     *  show a higher IC than the correct lower value. 
     */
    private int findShiftLength(String text) {
        double ioc[] = new double[KEYWD_MAX];   // Index of Coincidence of shifts of 1..10
	boolean done = false;
	int best = 1;
        double bestval = ioc[0];
        for (int k = 0; k < KEYWD_MAX; k++) {
	    double iclist[] = getICs(getKColumns(text,k+1));
	    ioc[k] = mean(iclist);
	    //            System.out.println("SUBTEXT["+(k+1)+"]\n " + subtext);
	    //	    if (params.verbose) 
		System.out.println("IC["+(k+1)+"] =" + ioc[k] + " Variance= " + variance(iclist,ioc[k]) + " Diff= " + Math.abs(ioc[k]-0.066));  // + "\t" + subtext);
            if (Math.abs(ioc[k]-0.066)  <  IC_EPSILON || ioc[k] > 0.066) {
		best = k+1;
		bestval = ioc[k];
		done = true;
		break;
	    }
	}
        for (int k = 1; !done && k < KEYWD_MAX; k++) { 	// If best value not already found, use the highest value
            if (ioc[k] >  bestval) {
                best = k+1;
                bestval = ioc[k];
	    }
	}
        return best;
    }

    /**
     * returns an array of the ICs for each string in the
     *  array, s.
     */
    private double[] getICs(String[] s) {
	double ic[] = new double[s.length];
	for (int k = 0; k < s.length; k++) {
	    IndexOfCoincidence IC = new IndexOfCoincidence(s[k]);
	    ic[k] = IC.getIOC();
	}
	return ic;
    }
    /**
     * returns the mean of an array 
     */
    private double mean(double[] a) {
	double sum = 0;
	for (int k = 0; k < a.length; k++)
	    sum += a[k];
	return sum/a.length;
    }
    /**
     * returns the variance of an array
     */
    private double variance(double[] a, double mean) {
	double v = 0;
	for (int k = 0; k < a.length; k++)
	    v += (a[k]-mean) * (a[k]-mean);
	return v/a.length;
    }


    /** 
     * getSumIC() concatenates the columns and returns their IC
     */
    private double getSumIC(String[] s) {
	String concat = "";	
	for (int k = 0; k < s.length; k++) 
	    concat += s[k];
	IndexOfCoincidence IC = new IndexOfCoincidence(concat);
        return IC.getIOC();
    }

    /**
     * getKColumns() converts the text, s, into an array of k columns. If k
     *  corresponds to the period of the Alberti cipher, then each column
     *  is a simple substitution cipher.
     * @param s -- the ciphertext
     * @param k -- the period
     */
    private String[] getKColumns(String s, int k) {
        String columns[] = new String[k];
        if (k <= 0) 
            throw new IllegalArgumentException("Skip size must be positive");
	for (int j = 0; j < k; j++) {
            StringBuffer sb = new StringBuffer();
            for (int i = j; i < s.length(); i+=k) 
                sb.append(s.charAt(i));
            columns[j] = sb.toString();
	    //            System.out.println("Column " + j + ":" + columns[j]);
	}
        return columns;
    }

    /**
     * mergeColumns() merges the columns into a single string of text.
     *  This method is used to reconstruct the original message
     * @param columns -- an array of the k simple substitution subtexts
     *  of the original message.
     */
    private String mergeColumns(String columns[]) {
	int d = Math.min(columns[0].length(), columns[columns.length-1].length());
        StringBuffer sb = new StringBuffer();
        for (int k = 0; k < d; k++) 
            for (int j = 0; j < columns.length; j++) 
                sb.append(columns[j].charAt(k));
	for (int j = 0; j < columns.length; j++)
	    if (columns[j].length() > d)
		sb.append(columns[j].charAt(d));
        return sb.toString();
    }

    /**
     * findShiftKeyword() uses brute force search to analyze each column in the array to determine
     *   how much it is shifted from a simple substitution histogram. Every possible shift is
     *   tried and the column is re-merged into the text and the text's IOC is computed. The shift
     *   that gives the highest IC is likely the correct shift for that column.
     * @param columns -- an array of the k simple substitution subtexts, shifted as a SIDE EFFECT
     * @return -- returns the encryption keyword, the complement of the shiftword found in the search
     */
    private String findShiftKeyword(String columns[]) {
        String text;
        StringBuffer  shiftword = new StringBuffer();     // Initialize shiftword to "aaaaa..."
	for (int k = 1; k <= columns.length; k++) 
	    shiftword.append('a');

	//	FastIC fastIC = new FastIC(text, shiftLen);
	FastIC fastIC = new FastIC(mergeColumns(columns), shiftLen);
	return searchForShiftKeywords(columns, shiftword, fastIC,  0);
    }

    /**
     * complementWord() computes the 26-ch complement of a word
     */
    private String complementWord(String s) {
	StringBuffer result = new StringBuffer();             
	for (int k = 0; k < s.length(); k++) 
	    result.append((char)('a' + ((26 - s.charAt(k) + 'a') % 26) )); // Complement of s
	return result.toString();
    }

    /**
     * printBackupICs is used to test that the icBackupTree is working properly. When called
     *  it should print the entries in the tree sorted in increasing order by IC.
     */
    private void printBackupICs(TreeMap tree, String keyword) {
	Vector v = new Vector(tree.values());
	int k = 0;
	while (k < v.size()) {
	    AlbertiAnalyzerBackupIC backup = (AlbertiAnalyzerBackupIC)v.get(k);
	    System.out.print("backup " + k + ":" +  backup.toString());
	    if (backup.shiftword.equals(keyword))
		System.out.println(" CORRECT");
	    else
		System.out.println();
	    ++k;
	}
    }

    /**
     * Repairs s1, the shiftword we found. This is for development use.
     */
    private String repairShiftword(String s1, String s2) {
	StringBuffer sb = new StringBuffer(s1);
	int diffcount[] = new int[26];        // Each possible difference
	int diffs[] = new int[s1.length()];   // Each letter-pair

	int maxdiffcount = 0;
	int maxdiff = 0;
	for (int k = 0; k < s1.length(); k++) {
	    int diff = (26 + (s1.charAt(k)-'a') - (s2.charAt(k)-'a')) % 26;
	    //	    System.out.println(s1.charAt(k) + " - " + s2.charAt(k) + " = " + diff);
	    diffs[k] = diff;    // Store the difference for this letter pair
	    ++diffcount[diff];  // And how many times this occurs
	    if (diffcount[diff] > maxdiffcount) {
		maxdiffcount = diffcount[diff];
		maxdiff = diff;
	    }
	}
	for (int j = 0; j < diffs.length; j++)
	    if (diffs[j] != maxdiff) {
		char ch = (char)('a' + ((sb.charAt(j) -'a' + 26 - diffs[j] + maxdiff)%26));   // Repair characters
		char ch_decrypt = (char)('a' + (26-ch+'a'));
		System.out.print("Error at k=" + j + " ch='" + ch_decrypt + "'. In encryption keyword '" + s1 + "', difference should be " + maxdiff + " instead of " + diffs[j]);
		System.out.print(", '" +  s1.charAt(j) + "' should be '" + ch);
		sb.setCharAt(j,(char)('a' + ((sb.charAt(j) -'a' + 26 - diffs[j] + maxdiff) % 26)));
		System.out.println("' giving= '" + sb.toString() +"'");  
	    }
	return sb.toString();
    }

    /**
     * substitute() applies the permutation alphabet to a cryptotext
     */
    private String substitute(String s, String alpha) {
	StringBuffer sb = new StringBuffer();
	for (int k = 0; k < s.length(); k++)
	    sb.append(alpha.charAt(s.charAt(k)-'a'));
	return new String(sb.toString());
    }

    /**
     * shiftText() breaks text into keywd.length() columns,
     *  shifts each columns by the keywd[k], and returns the merged
     *  columns.
     */
    private String shiftText(String text, String keywd) {
	int period = keywd.length();
	String columns[] = new String[period];
	columns = getKColumns(text, period);
	for (int k = 0; k < period; k++) 
	    columns[k] = shift(columns[k], keywd.charAt(k)-'a');
	return mergeColumns(columns);
    }
    
    /**
     * shift() shifts each character in the string s by n, assuming a 26
     *  letter alphabet
     * @param s, a text
     * @param n -- the amount to shift each letter
     */
    private String shift (String s, int n) {
        StringBuffer sb = new StringBuffer();
        for (int k = 0; k < s.length(); k++) {
            char ch = s.charAt(k);
            sb.append((char)('a' + ((ch - 'a') + n) % 26));
	}
        return sb.toString();
    }    

    /**
     * getWords() returns the space-delimited string of words formed
     *  by shifting the string s, 26 times.
     */
    private String getWords(String s) {
        StringBuffer sb = new StringBuffer();
	sb.append(" ");
        for (int k = 0; k < 26; k++) { // For each shift
            sb.append(shift(s,k));
	    //	    sb.append("\n");
	    sb.append(" ");
        }
        return sb.toString();
    } 

    private int countWrongs(char a1[], char a2[]) {
	int sum = 0;
	for (int k = 0; k < a1.length; k++)
	    if (a1[k] != a2[k])
		++sum;
	return sum;
    }

    /**
     * countWrongChars() compares s1 and s2 and returns the number
     *  of characters that are not identical
     */
    private int countWrongChars(String s1, String s2) {
	int n = Math.min(s1.length(),s2.length());
	int count = 0;
	for (int k = 0; k < n; k++)
	    if (s1.charAt(k) != s2.charAt(k))
		++count;
	return count;
    }

    /**
     * getMinKeywdMatch() returns the minimum number of characters by which 
     *   keywd differs from one of words on keywdlist.  If keywd exactly
     *   matches a word on keywdlist, 0 is returned.
     */
     private int getMinKeywdMatch(String keywd, String keywdlist) {
	 java.util.StringTokenizer st = new java.util.StringTokenizer(keywdlist);
	 int min = 100;
	 while (st.hasMoreTokens()) {
             int diff = countWrongChars(keywd, st.nextToken());
	     if (diff < min)
		 min = diff;
	 }
	 return min;
     }


    /**
     * run() conducts the cryptanalysis and reports the result
     */
    public void run() {
	nf.setMaximumFractionDigits(3);

	/********** Clean up the text and initialize things. *************/
        cleantext = TextUtilities.cleanString(text.toLowerCase());
        cleantext = TextUtilities.removeWhiteSpace(cleantext);
	String secretshiftkeyword = this.solution.substring(this.solution.indexOf("#"),this.solution.length()-1);
	actualKeywd = secretshiftkeyword.substring(secretshiftkeyword.indexOf("#")+1,secretshiftkeyword.indexOf("("));
        actualKeywdLen = Integer.parseInt(secretshiftkeyword.substring(secretshiftkeyword.indexOf("(")+1,secretshiftkeyword.indexOf(")")));
	//	System.out.println("Secret: " + actualKeywd + " " + actualKeywdLen + "\tsolution="+ solution);
	
	this.solution = TextUtilities.cleanString(this.solution.toLowerCase());
        this.solution = TextUtilities.removeWhiteSpace(this.solution);

	System.out.print("\tN= " + cleantext.length());
	//        System.out.println("CRYPTOGRAM begins: " + cleantext.substring(0,27) + "...");
        IC = new IndexOfCoincidence(cleantext);
        System.out.print("\tIC= " + nf.format(IC.getIOC()));
   
        /****** STEP 1: Find the length of the Alberti shift **************/
	shiftLen = findShiftLength(cleantext);
        System.out.print("\tP= " + actualKeywdLen + "\tN/P= " + nf.format(1.0*cleantext.length()/actualKeywdLen));
        if (shiftLen == actualKeywdLen)
	    System.out.print("\tOK");
	else if (actualKeywdLen % shiftLen  == 0)  // The period is a multiple of the actual shift
	    System.out.print("\tOK(" + shiftLen +")");
	else
	    System.out.print("\tNO(" + shiftLen +")");

	/********** STEP 2: Find the shift keyword ***********/
	String columns[] = new String[shiftLen];          // Break text into columns
	columns = getKColumns(cleantext, shiftLen);   

	//        String shiftKeyword = complementWord(findShiftKeyword(columns));   // This shifts the columns
	String shiftWords = findShiftKeyword(columns);   // This shifts the columns

	//	System.out.println(" Shiftkeyword= " + shiftKeyword + " Actual= " + actualKeywd);
	//	System.out.println(getWords(shiftKeyword));
	//	System.out.println(getWords(actualKeywd));

        String shiftedText = mergeColumns(columns);              
	FastIC ic = new FastIC(shiftedText);   // Check IC of shifted columns
	if (ic.getIC() < 0.062) {
	    System.out.println("*************** Step 2: Hard Alberti IC= " + ic.getIC() + " ***************** \n");
	    GaPopulation population = new AlbertiGaPopulation(cleantext, this.solution, params, shiftLen,"nopermutationalphabetgiven," + actualKeywd);
	//	    GaPopulation population = new AlbertiGaPopulation(cleantext, this.solution, params, shiftLen);
	    population.run();

	    String permKey = ((AlbertiGaPopulation)population).getBestPermutationKey();
	    String shiftKeys = findBestShiftKeywords(permKey, ((AlbertiGaPopulation)population).getBestShiftKey());
	    System.out.println("Phase 3: The best shift keywords are: " + shiftKeys);
	    ((AlbertiGaPopulation)population).runPhaseFour(shiftKeys);
	    return;
	} else {
	    System.out.println("*************** Phase 3: Easy Alberti IC = " + ic.getIC() + " ****************** \n"); //  + shiftWords);
	    StringTokenizer st = new StringTokenizer(shiftWords);
	    String shiftKeyword = st.nextToken();
	    //	    shiftedText = shiftText(cleantext, complementWord(shiftKeyword));
	    shiftedText = shiftText(cleantext, shiftKeyword);
	    String analysis = analyzeSimpleSubstitution(shiftedText);
	    System.out.println(analysis);
	    double rawEval = Double.parseDouble(analysis.substring(analysis.indexOf("Value=") + 6, analysis.indexOf("DECRYPT")));
	    double baseEval = cleantext.length()/shiftLen/rawEval;

	    if (baseEval > EVAL_EPSILON)
		System.out.println(" Done\n" + analysis + "\tRatio=" + baseEval);
	    else {
		System.out.println(" Retrying\n" + analysis + "\tRatio=" + baseEval);
		//		analyzeAndRetry(columns, shiftKeyword, baseEval);
		analyzeAndRetry(columns, complementWord(shiftKeyword), baseEval);
	    }
	    /*****************

	    while (baseEval < EVAL_EPSILON && st.hasMoreTokens()) {
		shiftedText = shiftText(cleantext, st.nextToken());
		//		System.out.println(analyzeSimpleSubstitution(shiftedText));
		analysis = analyzeSimpleSubstitution(shiftedText);
		System.out.println(analysis);
		rawEval = Double.parseDouble(analysis.substring(analysis.indexOf("Value=") + 6, analysis.indexOf("DECRYPT")));
		baseEval = cleantext.length()/shiftLen/rawEval;
	    }
	    System.out.println(analysis);
	    **********/
	    return;
	}
    }



    /**
     * Phase 3:  Find a collection of the best shift keywords given the partially correct
     *  permutation key and partially correct shift key.
     * @result return the collection as a space delimited String
     */
    private String findBestShiftKeywords(String permKey, String shiftKey) {
	String text = substitute(cleantext, permKey);
	StringBuffer shiftword = new StringBuffer(shiftKey.substring(0,3));  // Initialize shiftKey
	for (int k = 3; k < shiftKey.length(); k++) 
	    shiftword.append('a');
	String columns[] = getKColumns(text, shiftLen);
	FastIC fastIC = new FastIC(text, shiftLen);
	for (int k = 0; k < 3; k++) {                       // Shift the first 3 columns
	    fastIC.subtract(columns[k]);
	    columns[k] = shift(columns[k], shiftword.charAt(k)-'a');
	    fastIC.add(columns[k]);
	}
	return searchForShiftKeywords(columns, shiftword, fastIC,  3);
    }

    /*****
    private String complement(String words) {
	StringTokenizer st = new StringTokenizer(words);
	StringBuffer sb = new StringBuffer();
	while (st.hasMoreTokens()) {
	    sb.append(complementWord(st.nextToken()) + " ");
	}
	return sb.toString();
    }
    *********/

    private String searchForShiftKeywords(String columns[], StringBuffer shiftword, FastIC fastIC, int start) {
	boolean noimprovement = false;  // Variables used to control the search
	boolean lastlap = false;
	double prevmaxIC = 0;
	double thismaxIC = 0;
	int nbackups = 0;
	int m = 1;                      // Number of tries
	icBackupTree = new TreeMap(new AlbertiAnalyzerBackupIC());  // Stores backup shifts by column

	while (!noimprovement && m <= MAX_TRIES) {           // Try this repeatedly for best results
	    if (lastlap) {
		noimprovement = true;
		//		System.out.println("No improvement " + prevmaxIC);
	    }
	    thismaxIC = 0;
	    for (int k = start; k < shiftword.length(); k++) {                // For each subtext column
		double maxioc = 0;                                    // Keep track of best IC
		int bestshift = 0;
		String beststring = null;
		for (int j = 1; j <= 26; j++) {                       // For each possible shift
		    StringBuffer sb = new StringBuffer();
		    for (int c=0; c < columns[k].length(); c++) {     // For each letter in subtext
			char ch = columns[k].charAt(c);
			sb.append((char)('a' +  ((ch-'a') + 1) % 26));  // Shift it by 1
		    }
		    fastIC.add(sb.toString());
		    fastIC.subtract(columns[k]);
		    columns[k] = sb.toString();              // Replace with the shifted column
		    double ioc = fastIC.getIC();          // Calculate the text's new IC
		    if (ioc > maxioc) {                      // Remember the good ones
			//			char ch1 = actualKeywd.charAt(k);    // Used in development
			//			char ch2 = (char)('a' + ((shiftword.charAt(k) - 'a' + j) % 26));
			//			System.out.println("m=" + m + " k= " + k + "\tj= " + j + "\tch= " + ch2 +"\tsum= " + (((ch1-'a')+(ch2-'a'))%26) +  "\tic= " + ioc + "\timpr= " + (ioc-maxioc));
			maxioc = ioc;
			bestshift = j;
			beststring = columns[k];
			if (lastlap && Math.abs(ioc-prevmaxIC) < BACKUP_EPSILON && ioc != prevmaxIC) {
			    char ch = (char)('a' + ((shiftword.charAt(k) - 'a' + j) % 26));
			    StringBuffer shiftwrd = new StringBuffer(shiftword.toString());
			    shiftwrd.setCharAt(k, (char)('a' + ((shiftwrd.charAt(k) -'a' + j) % 26)));    // Copy and update the keyword
			    //			    AlbertiAnalyzerBackupIC backup = new AlbertiAnalyzerBackupIC(ioc,k,(int)(ch-'a'),complementWord(shiftwrd.toString()));
			    AlbertiAnalyzerBackupIC backup = new AlbertiAnalyzerBackupIC(ioc,k,(int)(ch-'a'),shiftwrd.toString());
			    icBackupTree.put (backup,backup);
			    //			    System.out.println("BACKUP CANDIDATE " + (++nbackups) + " = "  + ioc + "\tk=" + k + "\tch='" + ch  +"'");
			}
		    }
		} // j loop for each shift
		fastIC.subtract(columns[k]);
		fastIC.add(beststring);
		columns[k] = beststring;                                                                // Replace column k with best
		shiftword.setCharAt(k, (char)('a' + ((shiftword.charAt(k) -'a' + bestshift) % 26)));    // Update the shift keyword
		//		System.out.println("Shiftword= " + shiftword + " keywd= " + actualKeywd);
		if (maxioc > thismaxIC) 
		    thismaxIC = maxioc;
	    } // k loop for each column
	    if (thismaxIC > prevmaxIC) 
		prevmaxIC = thismaxIC;
	    else
		lastlap = true;
	    ++m;
	} // m loop
	//	if (m > MAX_TRIES)                            // Development use
	//	    System.out.print(" --MAX TRIES-- ");
	//	printBackupICs(icBackupTree);
	String keywds = shiftword.toString() + getBackupKeywords(icBackupTree);
	return keywds;
    }

    /**
     * getBackupKeywords returns of list of the keywords on the backup list.
     */
    private String getBackupKeywords(TreeMap tree) {
	String list = " ";
	Vector v = new Vector(tree.values());
	int k = 0;
	while (k < v.size()) {
	    AlbertiAnalyzerBackupIC backup = (AlbertiAnalyzerBackupIC)v.get(k);
	    list = list + backup.shiftword + " ";
	    ++k;
	}
	return list;
    }

    /**
     * analyzeSimpleSubstitution() creates an NgramAnalyzer and uses it to analyze the text.
     * @param text -- the text to be analyzed.
     * @result -- a report giving the NgramAnalyzer's evaluation value and part of the decryption.
     */
    private String analyzeSimpleSubstitution(String text) {
	String result="";
	try {
            Analyzer analyzer;
            analyzer = new NgramAnalyzer(params);
	    Alphabet alpha = AlphabetFactory.getInstance(AlphabetFactory.ALPH_az);
	    //	    System.out.println("solution= " + solution);
	    ((NgramAnalyzer)analyzer).setup(text+"$$$"+solution, params.book);
	    ((NgramAnalyzer)analyzer).setNgramAnalyzer(4,params.book,alpha,26,NgramAnalyzer.SIMPLESUB,100000);

	    analyzer.run();
	    result =  analyzer.getReport(); 

	} catch (Exception e) {
	    System.out.println("Problem in analyzeSimpleSubstitution: " + e.toString());
	    e.printStackTrace();
	}
	return result;
    }

    /**
     * analyzeAndRetry() repeatedly calls NgramAnalyzer until the message is
     *   decrypted or it runs out of alternative shifts.  The alternative shifts
     *   are stored in icBackupTree.  Each alternative indicates a column in the
     *   keyword and a alternative shift for that column.  
     * @param columns -- an array storing the cryptogram broken into P columns,
     *   where P is the period.
     * @param shiftKeyword -- a String of letters of length P that holds 
     *  the encryption keyword, the (26-k) converse of which gives the shifts
     *  that produced the current columns
     * @param baseVal -- the initial evaluation returned by NgramAnalyzer.
     *   baseVal = Size/P/RawNgramValue
     *   
     */
    private void analyzeAndRetry(String[] columns, String shiftKeyword, double baseVal) {
	AlbertiAnalyzerBackupIC key =  null, 
	    backup = null;
	StringBuffer newShiftKeyword = new StringBuffer(shiftKeyword);
	String oldShiftKeyword = shiftKeyword;                   // Save the shiftKeyword
	double diff = 0, rawEval = 0, eval= 0, 
	    baseEval = baseVal;

	for (int k = 1; k <= MAX_RETRIES && icBackupTree.size() != 0; k++) {
	    
	    /**** Use the backup (column,shift) to modify the encryption keyword ****/
	    key = (AlbertiAnalyzerBackupIC)icBackupTree.firstKey();
	    backup = (AlbertiAnalyzerBackupIC)icBackupTree.remove(key);
	    shiftKeyword = oldShiftKeyword;       
	    newShiftKeyword = new StringBuffer(shiftKeyword);
	    newShiftKeyword.setCharAt(backup.column, (char)('a' + (26-backup.shift)));  // Modify shift keyword 
	    char oldCh = shiftKeyword.charAt(backup.column);                            // Calculate the shift
	    char newCh = newShiftKeyword.charAt(backup.column);
	    //	    System.out.println("Col= " + backup.column + " j= " + backup.shift + " old= '" + shiftKeyword.charAt(backup.column) + "' new= '" + newShiftKeyword.charAt(backup.column) + "'");
	    shiftKeyword = newShiftKeyword.toString();                                  

	    /**** Shift columns[column] by shift, giving a new simple substitution cryptogram.  ****/
	    String columnK = new String(columns[backup.column]);     // Preserve the column being shifted
	    int shift = 0;                                           // Calculate the shift
	    if (oldCh > newCh)
		shift = oldCh - newCh;
	    else
		shift = 26 + (oldCh - newCh);
	    //	    System.out.println("Shifting column " + backup.column + " by " + shift);
	    columns[backup.column] = shift(columns[backup.column], shift);              // Rearrange the text

	    /***** Call NgramAnalyzer to analyze the text. *****/
	    String analysisReport = analyzeSimpleSubstitution(mergeColumns(columns));   // Retry the analysis
	    rawEval = Double.parseDouble(analysisReport.substring(analysisReport.indexOf("Value=") + 6, analysisReport.indexOf("DECRYPT")));
	    eval = (cleantext.length()/shiftLen)/rawEval;
	    diff = eval - baseEval;

	    if (eval >= EVAL_EPSILON) {             // Success
		System.out.println(analysisReport + "\t+"   + "\tRatio=" + eval);
		break;
	    } else if (diff < EVAL_DELTA) {        // No improvement
		columns[backup.column] = new String(columnK);  // Restore the column 
		shiftKeyword = oldShiftKeyword;    // Restore the keyword (Development)
		System.out.println(analysisReport + "\t-"  + "\tRatio=" + eval);
	    } else {                              // Improvement
		baseEval = eval;                  // Update the eval
		oldShiftKeyword = shiftKeyword;   // Forget the old shiftword (Development)
		System.out.println(analysisReport + "\t+"  + "\tRatio=" + eval);
	    }
	} // for max tries
    } // analyzeAndRetry()

    /**
     * ************AlbertiAnalyzerBackupIC ***************************
     * Inner class stores a record that can be used to repair the
     *  shift keyword. Gives the column and the shift that would
     *  produce the ic. 
     */
    class AlbertiAnalyzerBackupIC implements Comparator {
	public double ic;       // The Index of Coincidence Value
	public int column;      // The column number in the keyword
	public int shift;       // The shift value, 0..25
	public String shiftword; 

	public AlbertiAnalyzerBackupIC() { }

	public AlbertiAnalyzerBackupIC (double ic, int col, int shift, String s) {
	    this.ic = ic;
	    this.column = col;
	    this.shift = shift;
	    this.shiftword = new String(s);
	}

	/**
	 * Used to sort in increasing order. If two records have the same IC,
	 *  the column is used to break the tie.
	 */
	public int compare (Object o1, Object o2) {
	    if (((AlbertiAnalyzerBackupIC)o1).equals((AlbertiAnalyzerBackupIC)o2))
		return 0;
	    if ( ((AlbertiAnalyzerBackupIC)o1).ic > ((AlbertiAnalyzerBackupIC)o2).ic )
		return -1;
	    if ( ((AlbertiAnalyzerBackupIC)o1).ic == ((AlbertiAnalyzerBackupIC)o2).ic  && 
		 ((AlbertiAnalyzerBackupIC)o1).column < ((AlbertiAnalyzerBackupIC)o2).column )
		return -1;
	    return 1;
	}

	/**
	 * Two records are equal when their ICs and columns are equal.
	 */
	public boolean equals (Object o) {
	    return ((AlbertiAnalyzerBackupIC)o).ic == ic && ((AlbertiAnalyzerBackupIC)o).column == column;
	}

	public String toString() {
	    return "col= " + column + " shift= " + (char)('a' + shift) + " ic= " + ic + " " + shiftword;
	}
    }
} // AlbertiAnalyzer

