/*
 * File: EasyAlbertiAnalyzer.java
 * @author R. Morelli <ralph.morelli@trincoll.edu> 
 * 
 * Description: This class assumes that the text is
 *  encrypted with a polyalphabetic Alberti cipher. It uses the
 *  Index of Coincidence (IC) to cryptanalyze it.
 * 
 * Algorithm: A three step algorithm is used:
 *  1. Find the length of the shift keyword, k, using the IC
 *  2. Break the text into k simple substitution subtexts and use IC 
 *     to determine the shift of each subtext. These correspond to
 *     the letters in the shift keyword.
 *  3. Merge each of the k subtexts into the plaintext message.
 *
 *  To compile and run from the TestCryptAnalyzer application:
 *
 *  cd ~crypto/hcryptoj/1.4/applications/testanalyzer
 *  javac -classpath ../../classes -d ../../classes EasyAlbertiAnalyzer.java
 *  java -classpath ../../classes:. TestCryptAnalyzer analyzers.EasyAlbertiAnalyzer ga_paramfiles/albertiparam.txt
 */

package analyzers;

import hcrypto.analyzer.*;
import hcrypto.cipher.*;
import hcrypto.engines.*;
import hcrypto.provider.*;

import java.util.*;
import java.io.*;

public class EasyAlbertiAnalyzer extends CryptoAnalyzer implements Analyzer {
    private final int KEYWD_MAX = 26;
    private final int MAX_TRIES = 7;            // Step 2
    private final double IC_EPSILON = 0.005;    // Step 1 finding period
    //    private final double IC_EPSILON = 0.002;    // Step 2
    private final double BACKUP_EPSILON = 0.004;  // Step 2
    private final int MAX_RETRIES = 15;         // Step 3
    private final double EVAL_EPSILON = 0.42;   // Step 3
    private final double EVAL_DELTA = 0.0018;   // Step 3

    private IndexOfCoincidence IC;
    java.text.NumberFormat nf = java.text.NumberFormat.getInstance();

    private String actualKeywd = "";  // The actual secret shift keyword 
    private int  actualKeywdLen;
    private String cleantext = "";
    private int shiftLen = 0;

    private TreeMap icBackupTree;  // Keeps a sorted collection of ICs 


    /**
     * EasyAlbertiAnalyzer() -- Default constructor
     */
    public EasyAlbertiAnalyzer() {
	super();
    }

    /**
     * EasyAlbertiAnalyzer() -- this constructor is given an object containing parameter settings
     * @param params -- an object containing param1=val1 param2=val2 ...
     */
    public EasyAlbertiAnalyzer(GaParameters params) {
	super(params);
    }

    /**
     * setup() sets up the text and the parameters.. 
     */
    public void setup(String text) {
	super.setup(text);
	if (params == null) {
	    params = new GaParameters(); // Default GaParameters
	}
    }

    /**
     * findShiftLength() determines the length of the Alberti shift key through
     *  an exhaustive search. For each possible length, up to KEYWD_MAX, we break
     *  the text into columns and compute the average IC of the columns.  The
     *  length with the greatest IC is returned.  As an optimization, we break
     *  out of the search when we find an IC with 0.008 or 0.066. This works well
     *  for shifts that are not prime numbers, whose multiples would frequently
     *  show a higher IC than the correct lower value. 
     */
    private int findShiftLength(String text) {
        double ioc[] = new double[KEYWD_MAX];   // Index of Coincidence of shifts of 1..10
	boolean done = false;
	int best = 1;
        double bestval = ioc[0];
        for (int k = 0; k < KEYWD_MAX; k++) {
	    double iclist[] = getICs(getKColumns(text,k+1));
	    ioc[k] = mean(iclist);
	    //            System.out.println("SUBTEXT["+(k+1)+"]\n " + subtext);
	    if (params.verbose) 
		System.out.println("IC["+(k+1)+"] =" + ioc[k] + " Variance= " + variance(iclist,ioc[k]) + " Diff= " + Math.abs(ioc[k]-0.066));  // + "\t" + subtext);
            if (Math.abs(ioc[k]-0.066)  <  IC_EPSILON) {
		best = k+1;
		bestval = ioc[k];
		done = true;
		break;
	    }
	}
        for (int k = 1; !done && k < KEYWD_MAX; k++) { 	// If best value not already found, use the highest value
            if (ioc[k] >  bestval) {
                best = k+1;
                bestval = ioc[k];
	    }
	}
        return best;
    }


    /**
     * findPrimeShiftLength() determines the length of the Alberti shift key, searching
     *  only prime shift sizes. NO LONGER USED.
    private int findPrimeShiftLength(String text) {
	int primes[] = {1,2,3,5,7,11,13,17,19,23};   // First 10 primes
        double ioc[] = new double[primes.length];   // Index of Coincidence of shifts for first 9 primes
        for (int k = 0; k < primes.length; k++) {
	    int n = primes[k];
	    double iclist[] = getICs(getKColumns(text,k+1));
	    ioc[k] = mean(iclist);
	    if (params.verbose) 
		System.out.println("IC["+(n)+"] =" + ioc[k]);  // + "\t" + subtext);
	}
        int best = 1;
        double bestval = ioc[0];
        for (int k = 1; k < primes.length; k++) 
            if (ioc[k] >  bestval) {
                best = primes[k]; /// k+1;
                bestval = ioc[k];
	    }
        return best;
    }
    ************************************** */

    /**
     * returns an array of the ICs for each string in the
     *  array, s.
     */
    private double[] getICs(String[] s) {
	double ic[] = new double[s.length];
	for (int k = 0; k < s.length; k++) {
	    IndexOfCoincidence IC = new IndexOfCoincidence(s[k]);
	    ic[k] = IC.getIOC();
	}
	return ic;
    }
    /**
     * returns the mean of an array 
     */
    private double mean(double[] a) {
	double sum = 0;
	for (int k = 0; k < a.length; k++)
	    sum += a[k];
	return sum/a.length;
    }
    /**
     * returns the variance of an array
     */
    private double variance(double[] a, double mean) {
	double v = 0;
	for (int k = 0; k < a.length; k++)
	    v += (a[k]-mean) * (a[k]-mean);
	return v/a.length;
    }


    /** 
     * getSumIC() concatenates the columns and returns their IC
     */
    private double getSumIC(String[] s) {
	String concat = "";	
	for (int k = 0; k < s.length; k++) 
	    concat += s[k];
	IndexOfCoincidence IC = new IndexOfCoincidence(concat);
        return IC.getIOC();
    }

    /**
     * getKColumns() converts the text, s, into an array of k columns. If k
     *  corresponds to the period of the Alberti cipher, then each column
     *  is a simple substitution cipher.
     * @param s -- the ciphertext
     * @param k -- the period
     */
    private String[] getKColumns(String s, int k) {
        String columns[] = new String[k];
        if (k <= 0) 
            throw new IllegalArgumentException("Skip size must be positive");
	for (int j = 0; j < k; j++) {
            StringBuffer sb = new StringBuffer();
            for (int i = j; i < s.length(); i+=k) 
                sb.append(s.charAt(i));
            columns[j] = sb.toString();
	    //            System.out.println("Column " + j + ":" + columns[j]);
	}
        return columns;
    }

    /**
     * mergeColumns() merges the columns into a single string of text.
     *  This method is used to reconstruct the original message
     * @param columns -- an array of the k simple substitution subtexts
     *  of the original message.
     */
    private String mergeColumns(String columns[]) {
        StringBuffer sb = new StringBuffer();
        for (int k = 0; k < columns[columns.length-1].length(); k++) 
            for (int j = 0; j < columns.length; j++) 
                sb.append(columns[j].charAt(k));
        return sb.toString();
    }

    /**
     * findShiftKeyword() uses brute force search to analyze each column in the array to determine
     *   how much it is shifted from a simple substitution histogram. Every possible shift is
     *   tried and the column is re-merged into the text and the text's IOC is computed. The shift
     *   that gives the highest IC is likely the correct shift for that column.
     * @param columns -- an array of the k simple substitution subtexts
     * @return -- returns the encryption keyword, the complement of the shiftword found in the search
     */
    private String findShiftKeyword(String columns[]) {
        String text;
        StringBuffer  shiftword = new StringBuffer();     // Initialize shiftword to "aaaaa..."
	for (int k = 1; k <= columns.length; k++) 
	    shiftword.append('a');
	boolean noimprovement = false;  // Variables used to control the search
	boolean lastlap = false;
	double prevmaxIC = 0;
	double thismaxIC = 0;
	int nbackups = 0;
	int m = 1;                      // Number of tries

	FastIC fastIC = new FastIC(mergeColumns(columns), columns.length);
	icBackupTree = new TreeMap(new AlbertiAnalyzerBackupIC());  // Stores backup shifts by column

	while (!noimprovement && m <= MAX_TRIES) {           // Try this repeatedly for best results
	    if (lastlap) {
		noimprovement = true;
		//		System.out.println("No improvement " + prevmaxIC);
	    }
	    thismaxIC = 0;
	    for (int k = 0; k < columns.length; k++) {                // For each subtext column
		double maxioc = 0;                                    // Keep track of best IC
		int bestshift = 0;
		String beststring = null;
		for (int j = 1; j <= 26; j++) {                       // For each possible shift
		    StringBuffer sb = new StringBuffer();
		    for (int c=0; c < columns[k].length(); c++) {     // For each letter in subtext
			char ch = columns[k].charAt(c);
			sb.append((char)('a' +  ((ch-'a') + 1) % 26));  // Shift it by 1
		    }
		    fastIC.add(sb.toString());
		    fastIC.subtract(columns[k]);
		    columns[k] = sb.toString();              // Replace with the shifted column
		    double ioc = fastIC.getIC();
		    //		    System.out.println("k= " + k + " j= " + j + " ioc= " + ioc + "\t" + 
		    //				       (char)('a' + ((shiftword.charAt(k) -'a' + j) % 26)) + " " + actualKeywd.charAt(k));
		    //		    System.out.println("k= " + k + " j= " + j + " ioc= " + ioc + (char)('a' + j % 26) + " " + actualKeywd.charAt(k));
		    //		    double ioc = getSumIC(columns);          // Calculate the text's new IC
		    if (ioc > maxioc) {                      // Remember the good ones
			//			System.out.println("******** ");
			//			char ch1 = actualKeywd.charAt(k);    // Used in development
			//			char ch2 = (char)('a' + ((shiftword.charAt(k) - 'a' + j) % 26));
			//			System.out.println("m=" + m + " k= " + k + "\tj= " + j + "\tch= " + ch2 +"\tsum= " + (((ch1-'a')+(ch2-'a'))%26) +  "\tic= " + ioc + "\timpr= " + (ioc-maxioc));
			maxioc = ioc;
			bestshift = j;
			beststring = columns[k];
			if (lastlap && Math.abs(ioc-prevmaxIC) < BACKUP_EPSILON && ioc != prevmaxIC) {
			    char ch = (char)('a' + ((shiftword.charAt(k) - 'a' + j) % 26));
			    StringBuffer shiftwrd = new StringBuffer(shiftword.toString());
			    shiftwrd.setCharAt(k, (char)('a' + ((shiftwrd.charAt(k) -'a' + j) % 26)));    // Copy and update the keyword
			    AlbertiAnalyzerBackupIC backup = new AlbertiAnalyzerBackupIC(ioc,k,(int)(ch-'a'),complementWord(shiftwrd.toString()));
			    icBackupTree.put (backup,backup);
			    //			    System.out.println("BACKUP CANDIDATE " + (++nbackups) + " = "  + ioc + "\tk=" + k + "\tch='" + ch  +"'");
			}
		    }
		} // j loop for each shift
		fastIC.subtract(columns[k]);
		fastIC.add(beststring);
		columns[k] = beststring;                                                                // Replace column k with best
		shiftword.setCharAt(k, (char)('a' + ((shiftword.charAt(k) -'a' + bestshift) % 26)));    // Update the shift keyword
		//		System.out.println("Shiftword= " + shiftword + " keywd= " + actualKeywd);
		if (maxioc > thismaxIC) 
		    thismaxIC = maxioc;
	    } // k loop for each column
	    //	    System.out.println(">>>>>>>>>>>>>>>>>>>>>>>>>> ReducedKeywd= "+ reduce(shiftword.toString())
	    //			       + "\tReducedComplementActual= " + reduce(complementWord(actualKeywd))
	    //			       + "\tnWrong= "  + countWrongChars(reduce(shiftword.toString()),reduce(complementWord(actualKeywd))));
	    if (thismaxIC > prevmaxIC) 
		prevmaxIC = thismaxIC;
	    else
		lastlap = true;
	    ++m;
	} // m loop
	//	if (m > MAX_TRIES)                            // Development use
	//	    System.out.print(" --MAX TRIES-- ");

	//	printBackupICs(icBackupTree,actualKeywd);
	//	String repairedKeyword = repairShiftword(complementWord(shiftword.toString()), actualKeywd);   // Development
	//        return result.toString();
	return complementWord(shiftword.toString());  // Return the encryption keyword
    }

    /**
     * complementWord() computes the 26-ch complement of a word
     */
    private String complementWord(String s) {
	StringBuffer result = new StringBuffer();             
	for (int k = 0; k < s.length(); k++) 
	    result.append((char)('a' + ((26 - s.charAt(k) + 'a') % 26) )); // Complement of s
	return result.toString();
    }

    /**
     * printBackupICs is used to test that the icBackupTree is working properly. When called
     *  it should print the entries in the tree sorted in increasing order by IC.
     */
    private void printBackupICs(TreeMap tree, String keyword) {
	Vector v = new Vector(tree.values());
	int k = 0;
	while (k < v.size()) {
	    AlbertiAnalyzerBackupIC backup = (AlbertiAnalyzerBackupIC)v.get(k);
	    System.out.print("backup " + k + ":" +  backup.toString());
	    if (backup.shiftword.equals(keyword))
		System.out.println(" CORRECT");
	    else
		System.out.println();
	    ++k;
	}
    }

    /**
     * Repairs s1, the shiftword we found. This is for development use.
     */
    private String repairShiftword(String s1, String s2) {
	StringBuffer sb = new StringBuffer(s1);
	int diffcount[] = new int[26];        // Each possible difference
	int diffs[] = new int[s1.length()];   // Each letter-pair

	int maxdiffcount = 0;
	int maxdiff = 0;
	for (int k = 0; k < s1.length(); k++) {
	    int diff = (26 + (s1.charAt(k)-'a') - (s2.charAt(k)-'a')) % 26;
	    //	    System.out.println(s1.charAt(k) + " - " + s2.charAt(k) + " = " + diff);
	    diffs[k] = diff;    // Store the difference for this letter pair
	    ++diffcount[diff];  // And how many times this occurs
	    if (diffcount[diff] > maxdiffcount) {
		maxdiffcount = diffcount[diff];
		maxdiff = diff;
	    }
	}
	for (int j = 0; j < diffs.length; j++)
	    if (diffs[j] != maxdiff) {
		char ch = (char)('a' + ((sb.charAt(j) -'a' + 26 - diffs[j] + maxdiff)%26));   // Repair characters
		char ch_decrypt = (char)('a' + (26-ch+'a'));
		System.out.print("Error at k=" + j + " ch='" + ch_decrypt + "'. In encryption keyword '" + s1 + "', difference should be " + maxdiff + " instead of " + diffs[j]);
		System.out.print(", '" +  s1.charAt(j) + "' should be '" + ch);
		sb.setCharAt(j,(char)('a' + ((sb.charAt(j) -'a' + 26 - diffs[j] + maxdiff) % 26)));
		System.out.println("' giving= '" + sb.toString() +"'");  
	    }
	return sb.toString();
    }

    /**
     * shift() shifts each character in the string s by n, assuming a 26
     *  letter alphabet
     * @param s, a text
     * @param n -- the amount to shift each letter
     */
    private String shift (String s, int n) {
        StringBuffer sb = new StringBuffer();
        for (int k = 0; k < s.length(); k++) {
            char ch = s.charAt(k);
            sb.append((char)('a' + ((ch - 'a') + n) % 26));
	}
        return sb.toString();
    }    

    /**
     * getWords() returns the space-delimited string of words formed
     *  by shifting the string s, 26 times.
     */
    private String getWords(String s) {
        StringBuffer sb = new StringBuffer();
	sb.append(" ");
        for (int k = 0; k < 26; k++) { // For each shift
            sb.append(shift(s,k));
	    //	    sb.append("\n");
	    sb.append(" ");
        }
        return sb.toString();
    } 

    /**
     * getKthSubtext() creates a subtext of the message, s, consisting of
     *   every kth character
     * @param s -- the ciphertext
     * @param k -- the period
    private String getKthSubtext(String s, int k) {
        if (k <= 0) 
            throw new IllegalArgumentException("Skip size must be positive");
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < s.length(); i+=k)
            sb.append(s.charAt(i));
        return sb.toString();
    }
    ******************     */

    /**
     * reduces a shift keyword such as 'ralph' to one that
     *  begins with an 'a', such as 'ajuyq' -- that is with a 0 shift in column 0
     */
    private String reduce(String key) {
	StringBuffer sb = new StringBuffer(key);
	int diff = key.charAt(0) - 'a';
	//	System.out.println("Reduce by " + diff);
	for (int k = 0; k < (26 - diff) % 26 ; k++) 
	    for (int j = 0; j < key.length(); j++)
		sb.setCharAt(j, (char)('a' + ((sb.charAt(j) - 'a' + 1) % 26)));
	return sb.toString();
    }

    /**
     * countWrongChars() compares s1 and s2 and returns the number
     *  of characters that are not identical
     */
    private int countWrongChars(String s1, String s2) {
	int n = Math.min(s1.length(),s2.length());
	int count = 0;
	for (int k = 0; k < n; k++)
	    if (s1.charAt(k) != s2.charAt(k))
		++count;
	return count;
    }

    /**
     * getMinKeywdMatch() returns the minimum number of characters by which 
     *   keywd differs from one of words on keywdlist.  If keywd exactly
     *   matches a word on keywdlist, 0 is returned.
     */
     private int getMinKeywdMatch(String keywd, String keywdlist) {
	 java.util.StringTokenizer st = new java.util.StringTokenizer(keywdlist);
	 int min = 100;
	 while (st.hasMoreTokens()) {
             int diff = countWrongChars(keywd, st.nextToken());
	     if (diff < min)
		 min = diff;
	 }
	 return min;
     }


    /**
     * run() conducts the cryptanalysis and reports the result
     */
    public void run() {
	nf.setMaximumFractionDigits(3);
        cleantext = TextUtilities.cleanString(text.toLowerCase());
        cleantext = TextUtilities.removeWhiteSpace(cleantext);
	//	String keyspec = this.solution.substring(this.solution.indexOf("#")+1,this.solution.indexOf('/'));  // New versions cipher files
	//	actualKeywd  = keyspec.substring(keyspec.indexOf(',')+1);
	String keyspec = this.solution.substring(this.solution.indexOf("#")+1,this.solution.lastIndexOf('#')); // Old version of cipher files
	actualKeywd  = keyspec.substring(0,keyspec.indexOf('('));
        actualKeywdLen = actualKeywd.length();
	//	System.out.println("Secret: " + actualKeywd + " " + actualKeywdLen);
	
	this.solution = TextUtilities.cleanString(this.solution.toLowerCase());
        this.solution = TextUtilities.removeWhiteSpace(this.solution);

	System.out.print("\tN= " + cleantext.length());
	//        System.out.println("CRYPTOGRAM begins: " + cleantext.substring(0,27) + "...");
        IC = new IndexOfCoincidence(cleantext);
        System.out.print("\tIC= " + nf.format(IC.getIOC()));
   
	//        System.out.print("STEP 1: Find the length of the Alberti shift (0-"+KEYWD_MAX+")...");
	shiftLen = findShiftLength(cleantext);
	//       int shiftLen = findPrimeShiftLength(cleantext);
        System.out.print("\tP= " + actualKeywdLen);
        if (shiftLen == actualKeywdLen)
	    System.out.print(" OK");
	//	else if (actualKeywdLen % shiftLen  == 0)  // The period is a multiple of the actual shift
	else if (shiftLen % actualKeywdLen == 0)  // The period is a multiple of the actual shift
	    System.out.print(" OK(" + shiftLen +")");
	else
	    System.out.print(" NO(" + shiftLen +")");

	System.out.print("  \tN/P= " + nf.format(cleantext.length()/actualKeywdLen));
	//	System.out.print("  \tN/P= " + nf.format(cleantext.length()/shiftLen));
        System.out.print("  \tKeywd");
	String columns[] = new String[shiftLen];
	columns = getKColumns(cleantext, shiftLen);
        String shiftKeyword = findShiftKeyword(columns);
	//	System.out.println("Shiftkeyword= " + shiftKeyword + " Actual= " + actualKeywd);

	//	/******* Development Version *******************
	int wrongChars = getMinKeywdMatch(actualKeywd, getWords(shiftKeyword));
	if (wrongChars ==0) { // actualKeywd as a word on list of words
	    System.out.println(" RIGHT " + wrongChars);
	    //	    System.out.println(analyzeSimpleSubstitution(mergeColumns(columns)));
	}
	else {
	    //	    System.out.print(" WRONG("+wrongChars+")");
	    System.out.print("("+wrongChars+")");
	    //	    analyzeAndRetry(columns, shiftKeyword, wrongChars);
	}
        String shiftedText = mergeColumns(columns);              
        IC = new IndexOfCoincidence(shiftedText);
	//        System.out.println("STEP 3: Merge and analyze: SIMPLE SUBSTITUTION MESSAGE: " + shiftedText.substring(0,27) + "..." + 
	//			   " (IC= " + IC.getIOC() + ")");
	//	****************/

	/*****
	String analysis = analyzeSimpleSubstitution(mergeColumns(columns));
	double rawEval = Double.parseDouble(analysis.substring(analysis.indexOf("Value=") + 6, analysis.indexOf("DECRYPT")));
	double baseEval = cleantext.length()/shiftLen/rawEval;
	if (baseEval > EVAL_EPSILON)
	    System.out.println(" Done\n" + analysis);
	else {
	    System.out.println(" Retrying\n" + analysis);
	    analyzeAndRetry(columns, shiftKeyword, baseEval);
	}
	**********/
    }

    //    private void analyzeAndRetry(String[] columns, String shiftKeyword, int wrongChars) {  // DEVELOPMENT VERSION
    private void analyzeAndRetry(String[] columns, String shiftKeyword, double baseVal) {
	AlbertiAnalyzerBackupIC key =  null, backup = null;
	StringBuffer newShiftKeyword = new StringBuffer(shiftKeyword);
	String oldShiftKeyword = shiftKeyword;                              // Save the shiftKeyword
	//	int nWrong = wrongChars;
	//	String analysisReport = analyzeSimpleSubstitution(mergeColumns(columns));      // Get the baseline value
	//	double eval= 0, baseEval = Double.parseDouble(analysisReport.substring(analysisReport.indexOf("Ratio") + 6));
	double diff = 0, rawEval = 0, eval= 0, baseEval = baseVal;

	for (int k = 1; k <= MAX_RETRIES && icBackupTree.size() != 0; k++) {
	    //		System.out.println("nWrong= " + nWrong + " wrongChars= " + wrongChars);
	    //		System.out.println("shiftKeyword= " + shiftKeyword);
	    key = (AlbertiAnalyzerBackupIC)icBackupTree.firstKey();
	    backup = (AlbertiAnalyzerBackupIC)icBackupTree.remove(key);
	    //		System.out.println("retry backup= " + k + " ic = " + backup.ic + " col= " + backup.column + " shift= '" + (char)('a' + backup.shift) + "'");
	    
	    /****  This is developmental ***********	    ******************************/
	    shiftKeyword = oldShiftKeyword;
	    newShiftKeyword = new StringBuffer(shiftKeyword);
	    newShiftKeyword.setCharAt(backup.column, (char)('a' + (26-backup.shift)));  // Modify shift keyword (Development)
	    char oldCh = shiftKeyword.charAt(backup.column);
	    char newCh = newShiftKeyword.charAt(backup.column);
	    //	    System.out.println("Col= " + backup.column + " j= " + backup.shift + " old= '" + shiftKeyword.charAt(backup.column) + "' new= '" + newShiftKeyword.charAt(backup.column) + "'");
	    shiftKeyword = newShiftKeyword.toString();                                  // Development
	    //	    System.out.println("shiftKeywd= " + shiftKeyword + " actualKeywd= " + actualKeywd);

	    /**** This is real ****/
	    String columnK = new String(columns[backup.column]);                 // Save the column
	    //	    String mergeText = mergeColumns(columns);
	    int shift = 0;
	    if (oldCh > newCh)
		shift = oldCh - newCh;
	    else
		shift = 26 + (oldCh - newCh);
	    //	    System.out.println("Shifting column " + backup.column + " by " + shift);
	    columns[backup.column] = shift(columns[backup.column], shift);              // Rearrange the text
	    String analysisReport = analyzeSimpleSubstitution(mergeColumns(columns));   // Retry the analysis
	    rawEval = Double.parseDouble(analysisReport.substring(analysisReport.indexOf("Value=") + 6, analysisReport.indexOf("DECRYPT")));
	    eval = (cleantext.length()/shiftLen)/rawEval;
	    diff = eval - baseEval;

	    //	    /************

	    //	    nWrong = getMinKeywdMatch(actualKeywd, getWords(shiftKeyword));             // Development
	    //	    if (nWrong == 0) {                  // Done
	    if (eval >= EVAL_EPSILON) {                  // Done
		//		System.out.println("+ RIGHT " + nWrong);
		System.out.println(analysisReport + "\t+");
		break;
		//	    } else if (nWrong == wrongChars) {  // No change
	    } else if (diff < EVAL_DELTA) {  // No change
		columns[backup.column] = new String(columnK);  // Restore the column 
		shiftKeyword = oldShiftKeyword;    // Restore the keyword (Development)
		System.out.println(analysisReport + "\t-");
		//		nWrong = wrongChars;
		//		if (mergeText.equals(mergeColumns(columns))) System.out.println("Restored Columns");
		//	    } else if (nWrong > wrongChars) {  // Worse
		//	    } else if (eval - baseEval  > RETRY_EPSILON) {  // Worse
		//		System.out.print("-"); 
		//		shiftKeyword = oldShiftKeyword;    // Restore the keyword  (Development)
		//		nWrong = wrongChars;
		//		columns[backup.column] = new String(columnK);  // Restore the column 
		//		if (mergeText.equals(mergeColumns(columns))) System.out.println("Restored Columns");
		//		else System.out.println(mergeText + "\n" + mergeColumns(columns));
	    } else {                           // Improvement
		baseEval = eval;                                   // Update the eval
		oldShiftKeyword = shiftKeyword;                    // Forget the old shiftword (Development)
		System.out.println(analysisReport + "\t+");
		//		wrongChars = nWrong;              //  and the old wrongChars
	    }
	    //	    **********/
	} // for max tries
    } // analyzeAndRetry()

    private String analyzeSimpleSubstitution(String text) {
	String result="";
	try {
            Analyzer analyzer;
            analyzer = new NgramAnalyzer(params);
	    Alphabet alpha = AlphabetFactory.getInstance(AlphabetFactory.ALPH_az);
	    ((NgramAnalyzer)analyzer).setup(text+"$$$"+solution, params.book);
	    ((NgramAnalyzer)analyzer).setNgramAnalyzer(4,params.book,alpha,26,NgramAnalyzer.SIMPLESUB,100000);

	    analyzer.run();
	    result =  analyzer.getReport(); 

	} catch (Exception e) {
	    System.out.println(e.toString());
	}
	return result;
    }


    /**
     * ************AlbertiAnalyzerBackupIC ***************************
     * Inner class stores a record that can be used to repair the
     *  shift keyword. Gives the column and the shift that would
     *  produce the ic. 
     */
    class AlbertiAnalyzerBackupIC implements Comparator {
	public double ic;       // The Index of Coincidence Value
	public int column;      // The column number in the keyword
	public int shift;       // The shift value, 0..25
	public String shiftword; 

	public AlbertiAnalyzerBackupIC() { }

	public AlbertiAnalyzerBackupIC (double ic, int col, int shift, String s) {
	    this.ic = ic;
	    this.column = col;
	    this.shift = shift;
	    this.shiftword = new String(s);
	}

	/**
	 * Used to sort in increasing order. If two records have the same IC,
	 *  the column is used to break the tie.
	 */
	public int compare (Object o1, Object o2) {
	    if (((AlbertiAnalyzerBackupIC)o1).equals((AlbertiAnalyzerBackupIC)o2))
		return 0;
	    if ( ((AlbertiAnalyzerBackupIC)o1).ic > ((AlbertiAnalyzerBackupIC)o2).ic )
		return -1;
	    if ( ((AlbertiAnalyzerBackupIC)o1).ic == ((AlbertiAnalyzerBackupIC)o2).ic  && 
		 ((AlbertiAnalyzerBackupIC)o1).column < ((AlbertiAnalyzerBackupIC)o2).column )
		return -1;
	    return 1;
	}

	/**
	 * Two records are equal when their ICs and columns are equal.
	 */
	public boolean equals (Object o) {
	    return ((AlbertiAnalyzerBackupIC)o).ic == ic && ((AlbertiAnalyzerBackupIC)o).column == column;
	}

	public String toString() {
	    return "col= " + column + " shift= " + (char)('a' + shift) + " ic= " + ic + " " + shiftword;
	}
    }
} // EasyAlbertiAnalyzer

