/*
 * File: AlbertiGaAnalyzer.java
 * @author R. Morelli <ralph.morelli@trincoll.edu> 
 * 
 * Description: This class assumes that the text is
 *  encrypted with the decryption function of an Alberti cihper.
 *  That is, the text is first shifted and then permuted by substituting
 *  through the permutation alphabet. This is a much harder problem than 
 *  breaking the Alberti encryption function, which first permutes the text 
 *  by substituting through the permutation alphabet and then shifts it.
 *  To cryptanalyze the easier, encryption function, the text can simply
 *  be shifted in reverse, given the period. The IC (Index of Coincidence)
 *  can be used to recognize when the correct shifts have been applied.
 *  The correctly shifted text can then be broken by a simple substitution
 *  analyzer.
 * 
 *  To cryptanalyze the decryption function, our strategy is to 
 *  create one individual for every possible shift of the first two letters
 *  in the shift keyword.  That would be 626 individuals, each with its
 *  own random permutation alphabet. The GA will converge on the correct
 *  permutation key, after which the remaining letters in the shift keyword
 *  can be determined. 
 * 
 *  Its evaluation function uses an NgramArray with frequencies of
 *  N = 2 (bigrams). 
 *  Using the sum of the inverses of the frequencies as an evaluation of how
 *  accurately a decrypt matches the usual N-gram frequencies in a
 *  language is an idea described by Alex Griffing the developer of the
 *  "Automatic Cryptogram Solver".  This program is based on R. Walde's
 *  NgramClimber.java.
 *
 *  To compile and run from the TestAnalyzer application:
 *
 *  cd ~crypto/hcryptoj/1.4/applications/testanalyzer
 *  javac -classpath ../../classes -d ../../classes AlbertiGaAnalyzer.java
 *  java -classpath ../../classes:. TestGaAnalyzer analyzers.AlbertiGaAnalyzer ga_paramfiles/albertiparam.txt
 *
 */

package analyzers;

import hcrypto.analyzer.*;
import hcrypto.cipher.*;
import hcrypto.engines.*;
import hcrypto.provider.*;

import java.util.*;
import java.io.*;

public class AlbertiGaAnalyzer extends CryptoAnalyzer implements Analyzer 
{
    
    private final int KEYWD_MAX = 34;
    private final int MAX_TRIES = 7;
    private final int MAX_BACKUPS = 10;
    private final double IC_EPSILON = 0.008;    
    private final double BACKUP_EPSILON = 0.004;  // Step 2 How we decide which ICs to save
    private final int MAX_RETRIES = 15;         // Step 3
    private final double EVAL_EPSILON =  0.35;   // Step 3  How we know when text is decrypted
    private final double EVAL_DELTA = 0.0018;   // Step 3  How we know if a retry has improved decryption

    private int shiftLen;
    private String cleantext;
    private long nKeys = 0;
    private String actualKeywd;
    
    protected GaPopulation population;     
    private TreeMap icBackupTree;  // Keeps a sorted collection of ICs 

    /**
     * AlbertiGaAnalyzer() -- Default constructor
     */
    public AlbertiGaAnalyzer() 
    {
	super();
    }

    /**
     * AlbertiGaAnalyzer() -- this constructor is given an object containing parameter settings
     * @param params -- an object containing param1=val1 param2=val2 ...
     */
    public AlbertiGaAnalyzer(GaParameters params) 
    {
	super(params);
    }

    /**
     * findShiftLength() determines the length of the Alberti shift key through
     *  an exhaustive search. For each possible length, up to KEYWD_MAX, we break
     *  the text into columns and compute the average IC of the columns.  The
     *  length with the greatest IC is returned.  As an optimization, we break
     *  out of the search when we find an IC with 0.008 or 0.066. This works well
     *  for shifts that are not prime numbers, whose multiples would frequently
     *  show a higher IC than the correct lower value. 
     */
    private int findShiftLength(String text) {
        double ioc[] = new double[KEYWD_MAX];   // Index of Coincidence of shifts of 1..10
	boolean done = false;
	int best = 1;
        double bestval = ioc[0];
        for (int k = 0; k < KEYWD_MAX; k++) {
	    double iclist[] = getICs(getKColumns(text,k+1));
	    ioc[k] = mean(iclist);
	    //            System.out.println("SUBTEXT["+(k+1)+"]\n " + subtext);
	    if (params.verbose) 
		System.out.println("IC["+(k+1)+"] =" + ioc[k] + " Variance= " + variance(iclist,ioc[k]) + " Diff= " + Math.abs(ioc[k]-0.066));  // + "\t" + subtext);
            if (Math.abs(ioc[k]-0.066)  <  IC_EPSILON || ioc[k] > IC_EPSILON + 0.066) {
		best = k+1;
		bestval = ioc[k];
		done = true;
		break;
	    }
	}
        for (int k = 1; !done && k < KEYWD_MAX; k++) { 	// If best value not already found, use the highest value
            if (ioc[k] >  bestval) {
                best = k+1;
                bestval = ioc[k];
	    }
	}
        return best;
    }

    /**
     * returns an array of the ICs for each string in the
     *  array, s.
     */
    private double[] getICs(String[] s) {
	double ic[] = new double[s.length];
	for (int k = 0; k < s.length; k++) {
	    IndexOfCoincidence IC = new IndexOfCoincidence(s[k]);
	    ic[k] = IC.getIOC();
	}
	return ic;
    }
    /**
     * returns the mean of an array 
     */
    private double mean(double[] a) {
	double sum = 0;
	for (int k = 0; k < a.length; k++)
	    sum += a[k];
	return sum/a.length;
    }
    /**
     * returns the variance of an array
     */
    private double variance(double[] a, double mean) {
	double v = 0;
	for (int k = 0; k < a.length; k++)
	    v += (a[k]-mean) * (a[k]-mean);
	return v/a.length;
    }

    private String getKthSubtext(String s, int k) {
        if (k <= 0) 
            throw new IllegalArgumentException("Skip size must be positive");
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < s.length(); i+=k)
            sb.append(s.charAt(i));
        return sb.toString();
    }

    /**
     * getKColumns() converts the text, s, into an array of k columns. If k
     *  corresponds to the period of the Alberti cipher, then each column
     *  is a simple substitution cipher.
     * @param s -- the ciphertext
     * @param k -- the period
     */
    private String[] getKColumns(String s, int k) {
        String columns[] = new String[k];
        if (k <= 0) 
            throw new IllegalArgumentException("Skip size must be positive");
	for (int j = 0; j < k; j++) {
            StringBuffer sb = new StringBuffer();
            for (int i = j; i < s.length(); i+=k) 
                sb.append(s.charAt(i));
            columns[j] = sb.toString();
	    //            System.out.println("Column " + j + ":" + columns[j]);
	}
        return columns;
    }

    /**
     * mergeColumns() merges the columns into a single string of text.
     *  This method is used to reconstruct the original message
     * @param columns -- an array of the k simple substitution subtexts
     *  of the original message.
     */
    private String mergeColumns(String columns[]) {
	int d = Math.min(columns[0].length(), columns[columns.length-1].length());
        StringBuffer sb = new StringBuffer();
        for (int k = 0; k < d; k++) 
            for (int j = 0; j < columns.length; j++) 
                sb.append(columns[j].charAt(k));
	for (int j = 0; j < columns.length; j++)
	    if (columns[j].length() > d)
		sb.append(columns[j].charAt(d));
        return sb.toString();
    }

    /**
     * substitute() applies the permutation alphabet to a cryptotext
     */
    private String substitute(String s, String alpha) {
	StringBuffer sb = new StringBuffer();
	for (int k = 0; k < s.length(); k++)
	    sb.append(alpha.charAt(s.charAt(k)-'a'));
	return new String(sb.toString());
    }


    /**
     * shiftText() breaks text into keywd.length() columns,
     *  shifts each columns by the keywd[k], and returns the merged
     *  columns.
     */
    private String shiftText(String text, String keywd) {
	int period = keywd.length();
	String columns[] = new String[period];
	columns = getKColumns(text, period);
	for (int k = 0; k < period; k++) 
	    columns[k] = shift(columns[k], keywd.charAt(k)-'a');
	return mergeColumns(columns);
    }
    
    /**
     * shift() shifts each character in the string s by n, assuming a 26
     *  letter alphabet
     * @param s, a text
     * @param n -- the amount to shift each letter
     */
    private String shift (String s, int n) {
        StringBuffer sb = new StringBuffer();
        for (int k = 0; k < s.length(); k++) {
            char ch = s.charAt(k);
            sb.append((char)('a' + ((ch - 'a') + n) % 26));
	}
        return sb.toString();
    }    

    /**
     * reduces a shift keyword such as 'ralph' to one that
     *  begins with an 'a', such as 'ajuyq' -- that is with a 0 shift in column 0
     */
    private String reduce(String key) {
	StringBuffer sb = new StringBuffer(key);
	int diff = key.charAt(0) - 'a';
	//	System.out.println("Reduce by " + diff);
	for (int k = 0; k < (26 - diff) % 26 ; k++) 
	    for (int j = 0; j < key.length(); j++)
		sb.setCharAt(j, (char)('a' + ((sb.charAt(j) - 'a' + 1) % 26)));
	return sb.toString();
    }

    /**
     * setup() creates the population for the GA.
     */
    public void setup(String text) 
    {
	super.setup(text);
	if (params == null) {
	    params = new GaParameters();       // Default GaParameters
	}
        cleantext = TextUtilities.cleanString(this.text.toLowerCase());
        cleantext = TextUtilities.removeWhiteSpace(cleantext);

	String keyspec = this.solution.substring(this.solution.indexOf("#")+1,this.solution.indexOf('/'));
	actualKeywd  = keyspec.substring(keyspec.indexOf(',')+1);
        int actualKeywdLen = actualKeywd.length();
	//	System.out.println("\tshiftword= " + actualKeywd + " l= " + actualKeywdLen);

	this.solution = TextUtilities.cleanString(this.solution.toLowerCase());
        this.solution = TextUtilities.removeWhiteSpace(this.solution);

        shiftLen= findShiftLength(cleantext);
        System.out.println("\nCLEANTEXT_LEN= " + cleantext.length()
			   + "\tSOLUTION_LEN= " + this.solution.length()
			   + "\tSHIFT_LEN= " + shiftLen 
			   + "\tCOLUMN_LEN= " + cleantext.length()/shiftLen
			   + "\tkey= " + keyspec
			   );
	//        System.out.println("\nText = \n" + cleantext);
	//        System.out.println("\nSolution = \n" + this.solution);

	population = new AlbertiGaPopulation(cleantext, this.solution, params, shiftLen, keyspec);
	//	population = new AlbertiGaPopulation(cleantext, this.solution, params, shiftLen, reduce(actualKeywd));
	//	population = new AlbertiGaPopulation(cleantext, this.solution, params, shiftLen);
    }

    /**
     * run() conducts the GA run and reports the results.  Phases 1 and 2 are done
     *  by the population GA. Phase 1 searches for the first 2 letters of the shift keyword.
     *  Phase 2 searches for the first 3 letters of the shift keyword and the correct
     *  permutation key.  Phase_3 finds the remaining letters in the shift keyword.
     *  Phase_4 finds the permutation key.
     */
    public void run() {
	population.run();  // Phase_1 (2-letter) and Phase_2 (3-letter) GA finds partial permutation key
	//	System.out.println(shiftText(substitute(cleantext,"gluoraztnmxvekiysbphfjqdwc"),"aswbg"));
	String permKey = ((AlbertiGaPopulation)population).getBestPermutationKey();
	String partShiftKey = ((AlbertiGaPopulation)population).getBestShiftKey();

	//	System.out.println("PHASE_3: bestPermKey=" + permKey + " bestShiftKey= " + partShiftKey);
	String text = substitute(cleantext, permKey);

	String shiftKey = findShiftKeyword(getKColumns(text,shiftLen));
	String keywds = shiftKey + getBackupKeywords(icBackupTree);
	String partialDecrypt = shiftText(text, shiftKey);
	//	System.out.println("SHIFTKEYS = " + getBackupKeywords(icBackupTree));
	//	System.out.println("SHIFTKEY= " + shiftKey);
	double wrongs= 100.0 * countWrongs(partialDecrypt,solution)/cleantext.length();
	java.text.NumberFormat nf = java.text.NumberFormat.getNumberInstance();
	nf.setMaximumFractionDigits(3);

	//	System.out.println("PARTIAL_DECRYPT: (nWrong=  " +  nf.format(wrongs)+ " %)\n" + partialDecrypt.substring(0,30));

	/*****************
        String shiftKeys = findBestShiftKeywords(permKey, ((AlbertiGaPopulation)population).getBestShiftKey());
	String skey= shiftKeys.substring(0,shiftKeys.indexOf(' '));
	System.out.println("SHIFTKEYS= " + shiftKeys);
	System.out.println("SHIFTKEY= " + skey);
	partialDecrypt = shiftText(text, skey);
	System.out.println("PARTIAL DECRYPT: (nWrong=  " + countWrongs(partialDecrypt,solution) + ") " + partialDecrypt.substring(0,30));
	*****************/
	System.out.println("KEYWD_3: nWrongShifts= " + countWrongs(shiftKey,actualKeywd)
			   + "\tnShiftWds= " + (1 + icBackupTree.size())  
			   + "\tPARTIAL_DECRYPT= " + partialDecrypt.substring(0,30) + " " +  nf.format(wrongs) + " %"
			   + "\tkeyCount= " + nKeys );
	//	System.out.println("PHASE_3: shiftKeywds= " + keywds);

	//	String shiftedText = shiftText(substitute(cleantext, permKey), shiftKey);
	//        IndexOfCoincidence IC = new IndexOfCoincidence(shiftedText);
	//	System.out.println("Doing Step 3, \tText=" + shiftedText.substring(0,30) + "\tIC=" + IC.getIOC() + " shiftKeys= " + shiftKeys);

	//	System.out.println(analyzeSimpleSubstitution(partialDecrypt, permKey));  // For easy alberti. Only works on a simple substitution (0.066)
	((AlbertiGaPopulation)population).runPhaseFour(keywds);
    }


    /**
     * findShiftKeyword() uses brute force search to analyze each column in the array to determine
     *   how much it is shifted from a simple substitution histogram. Every possible shift is
     *   tried and the column is re-merged into the text and the text's IOC is computed. The shift
     *   that gives the highest IC is likely the correct shift for that column.
     * @param columns -- an array of the k simple substitution subtexts
     * @return -- returns the encryption keyword, the complement of the shiftword found in the search
     */
    //    private String findShiftKeyword(String columns[], String partShiftKey) {
    private String findShiftKeyword(String columns[]) {
        String text;
	//	System.out.println("INITIAL TEXT= " + mergeColumns(columns).substring(0,40));
        StringBuffer  shiftword = new StringBuffer();     // Initialize shiftword to "aaaaa..."
	for (int k = 1; k <= columns.length; k++) 
	    shiftword.append('a');
	boolean noimprovement = false;  // Variables used to control the search
	boolean lastlap = false;
	double prevmaxIC = 0;
	double thismaxIC = 0;
	int nbackups = 0;
	int m = 1;                      // Number of tries
	nKeys = 0;

	FastIC fastIC = new FastIC(mergeColumns(columns), columns.length);
	icBackupTree = new TreeMap(new AlbertiAnalyzerBackupIC());  // Stores backup shifts by column

	while (!noimprovement && m <= MAX_TRIES) {           // Try this repeatedly for best results
	    if (lastlap) {
		noimprovement = true;
		//		System.out.println("No improvement " + prevmaxIC);
	    }
	    thismaxIC = 0;
	    for (int k = 0; k < columns.length; k++) {                // For each subtext column
		double maxioc = 0;                                    // Keep track of best IC
		int bestshift = 0;
		String beststring = null;
		for (int j = 1; j <= 26; j++) {                       // For each possible shift
		    ++nKeys;
		    StringBuffer sb = new StringBuffer();
		    for (int c=0; c < columns[k].length(); c++) {     // For each letter in subtext
			char ch = columns[k].charAt(c);
			sb.append((char)('a' +  ((ch-'a') + 1) % 26));  // Shift it by 1
		    }
		    fastIC.add(sb.toString());
		    fastIC.subtract(columns[k]);
		    columns[k] = sb.toString();              // Replace with the shifted column
		    double ioc = fastIC.getIC();
		    //		    System.out.println("k= " + k + " j= " + j + " ioc= " + ioc + "\t" + 
		    //				       (char)('a' + ((shiftword.charAt(k) -'a' + j) % 26)) + " " + actualKeywd.charAt(k));
		    if (ioc > maxioc) {                      // Remember the good ones
			//			System.out.println("******** ");
			//			char ch1 = actualKeywd.charAt(k);    // Used in development
			//			char ch2 = (char)('a' + ((shiftword.charAt(k) - 'a' + j) % 26));
			//			System.out.println("m=" + m + " k= " + k + "\tj= " + j + "\tch= " + ch2 +"\tsum= " + (((ch1-'a')+(ch2-'a'))%26) +  "\tic= " + ioc + "\timpr= " + (ioc-maxioc));
			maxioc = ioc;
			bestshift = j;
			beststring = columns[k];
			if (lastlap && Math.abs(ioc-prevmaxIC) < BACKUP_EPSILON && ioc != prevmaxIC) {
			    char ch = (char)('a' + ((shiftword.charAt(k) - 'a' + j) % 26));
			    StringBuffer shiftwrd = new StringBuffer(shiftword.toString());
			    shiftwrd.setCharAt(k, (char)('a' + ((shiftwrd.charAt(k) -'a' + j) % 26)));    // Copy and update the keyword
			    AlbertiAnalyzerBackupIC backup = new AlbertiAnalyzerBackupIC(ioc,k,(int)(ch-'a'), reduce(shiftwrd.toString()));
			    icBackupTree.put (backup,backup);
			    //			    System.out.println("BACKUP CANDIDATE " + (++nbackups) + " = "  + ioc + "\tk=" + k + "\tch='" + ch  +"'");
			}
		    }
		    //		    System.out.println("REVISED TEXT= " + mergeColumns(columns).substring(0,40));
		} // j loop for each shift
		fastIC.subtract(columns[k]);
		fastIC.add(beststring);
		columns[k] = beststring;                                                                // Replace column k with best
		shiftword.setCharAt(k, (char)('a' + ((shiftword.charAt(k) -'a' + bestshift) % 26)));    // Update the shift keyword
		//		System.out.println("Shiftword= " + shiftword + " keywd= " + actualKeywd);
		if (maxioc > thismaxIC) 
		    thismaxIC = maxioc;
	    } // k loop for each column
	    //	    System.out.println(">>>>>>>>>>>>>>>>>>>>>>>>>> Keywd= "+ shiftword.toString()
	    //			       + "\tReducedKeywd= " +  reduce(shiftword.toString())
	    //			       + "\tReducedActual= " + reduce(actualKeywd)
	    //			       + "\tnWrong= "  + countWrongs(reduce(shiftword.toString()),reduce(actualKeywd))
	    //			       + "\tText= " + mergeColumns(columns).substring(0,40)
	    //			       );
	    if (thismaxIC > prevmaxIC) 
		prevmaxIC = thismaxIC;
	    else
		lastlap = true;
	    ++m;
	} // m loop
	//	if (m > MAX_TRIES)                            // Development use
	//	    System.out.print(" --MAX TRIES-- ");

	//	printBackupICs(icBackupTree,actualKeywd);
	//	String repairedKeyword = repairShiftword(complementWord(shiftword.toString()), actualKeywd);   // Development
	//        return result.toString();
	//	return complementWord(shiftword.toString());  // Return the encryption keyword
	return reduce(shiftword.toString());  // Return the decryption keyword
    }







    /**
     * Phase 3:  Find a collection of the best shift keywords given the partially correct
     *  permutation key and partially correct shift key.
     * @result return the collection as a space delimited String
     */
    private String findBestShiftKeywords(String permKey, String shiftKey) {
	String text = substitute(cleantext, permKey);
	System.out.println("SUBSTITUTED TEXT=" + text.substring(0,40));
	//	System.out.println("DECRYPT=\n" + shiftText(text, shiftKey));

	StringBuffer shiftword = new StringBuffer(shiftKey.substring(0,3));
	//        StringBuffer  shiftword = new StringBuffer();     // Initialize shiftword to "aaaaa..."
	//	for (int k = 1; k <= shiftKey.length(); k++) 
	for (int k = 3; k < shiftKey.length(); k++) 
	    shiftword.append('a');
	String columns[] = getKColumns(text, shiftLen);
	System.out.println("INITIAL TEXT= " + mergeColumns(columns).substring(0,40));
	FastIC fastIC = new FastIC(text, shiftLen);

	boolean noimprovement = false;  // Variables used to control the search
	boolean lastlap = false;
	double prevmaxIC = 0;
	double thismaxIC = 0;
	int nbackups = 0;
	int m = 1;                      // Number of tries
	nKeys = 0;
	
	System.out.println("KEYWD= " + shiftword.toString());
	
	//	/*****************
	for (int k = 1; k < 3; k++) {
	    fastIC.subtract(columns[k]);
	    columns[k] = shift(columns[k], shiftword.charAt(k)-'a');
	    fastIC.add(columns[k]);
	    System.out.println("NEW TEXT= " + mergeColumns(columns).substring(0,40));
	}
	//	****************/

	icBackupTree = new TreeMap(new AlbertiAnalyzerBackupIC());  // Stores backup shifts by column

	while (!noimprovement && m <= MAX_TRIES) {           // Try this repeatedly for best results
	    if (lastlap) {
		noimprovement = true;
		//		System.out.println("No improvement " + prevmaxIC);
	    }
	    thismaxIC = 0;
	    //	    for (int k = 0; k < shiftKey.length(); k++) {                // For each subtext column
	    for (int k = 3; k < shiftKey.length(); k++) {                // For each subtext column
		double maxioc = 0;                                    // Keep track of best IC
		int bestshift = 0;
		String beststring = null;
		for (int j = 1; j <= 26; j++) {                       // For each possible shift
		    ++nKeys;
		    StringBuffer sb = new StringBuffer();
		    for (int c=0; c < columns[k].length(); c++) {     // For each letter in subtext
			char ch = columns[k].charAt(c);
			sb.append((char)('a' +  ((ch-'a') + 1) % 26));  // Shift it by 1
		    }
		    fastIC.add(sb.toString());
		    fastIC.subtract(columns[k]);
		    columns[k] = sb.toString();              // Replace with the shifted column
		    double ioc = fastIC.getIC();          // Calculate the text's new IC
		    double icFirstK = fastIC.getAvgFirstNIC(k);          // Calculate the text's new IC
		    //		    System.out.println("k= " + k + " j= " + j + " ioc= " + ioc + " icFirstK= " + icFirstK + " " + (char)('a' + j % 26) + " " + actualKeywd.charAt(k));
		    //		    System.out.println("k= " + k + " j= " + j + " ioc= " + ioc + " icFirstK= " + icFirstK + " " + 
		    //				       (char)('a' + ((shiftword.charAt(k) -'a' + j) % 26)) + " " + actualKeywd.charAt(k));
		    if (ioc > maxioc) {                      // Remember the good ones
			//		    System.out.println("******** ");
			//			char ch1 = actualKeywd.charAt(k);    // Used in development
			//			char ch2 = (char)('a' + ((shiftword.charAt(k) - 'a' + j) % 26));
			//			System.out.println("m=" + m + " k= " + k + "\tj= " + j + "\tch= " + ch2 +"\tsum= " + (((ch1-'a')+(ch2-'a'))%26) +  "\tic= " + ioc + "\timpr= " + (ioc-maxioc));
			maxioc = ioc;
			bestshift = j;
			beststring = columns[k];
			if (lastlap && Math.abs(ioc-prevmaxIC) < BACKUP_EPSILON && ioc != prevmaxIC) {
			    char ch = (char)('a' + ((shiftword.charAt(k) - 'a' + j) % 26));
			    StringBuffer shiftwrd = new StringBuffer(shiftword.toString());
			    shiftwrd.setCharAt(k, (char)('a' + ((shiftwrd.charAt(k) -'a' + j) % 26)));    // Copy and update the keyword
			    //			    AlbertiAnalyzerBackupIC backup = new AlbertiAnalyzerBackupIC(ioc,k,(int)(ch-'a'),complementWord(shiftwrd.toString()));
			    AlbertiAnalyzerBackupIC backup = new AlbertiAnalyzerBackupIC(ioc,k,(int)(ch-'a'), reduce(shiftwrd.toString()));
			    icBackupTree.put (backup,backup);
			    //			    System.out.println("BACKUP CANDIDATE " + (++nbackups) + " = "  + ioc + "\tk=" + k + "\tch='" + ch  +"'");
			    //			    System.out.println("BACKUP CANDIDATE " + shiftwrd.toString());
			}
		    }
		    //		    System.out.println("REVISED TEXT= " + mergeColumns(columns).substring(0,40));
		} // j loop for each shift
		fastIC.subtract(columns[k]);
		fastIC.add(beststring);
		columns[k] = beststring;                                                                // Replace column k with best
		shiftword.setCharAt(k, (char)('a' + ((shiftword.charAt(k) -'a' + bestshift) % 26)));    // Update the shift keyword
		//		System.out.println("Shiftword= " + shiftword + " keywd= " + actualKeywd);
		if (maxioc > thismaxIC) 
		    thismaxIC = maxioc;
	    } // k loop for each column
	    System.out.println(">>>>>>>>>>>>>>>>>>>>>>>>>> Keywd= "+ shiftword.toString()
			       + "\tReducedKeywd= " + reduce(shiftword.toString())
			       + "\tReducedActual= " + reduce(actualKeywd)
			       + "\tnWrong= "  + countWrongs(reduce(shiftword.toString()),reduce(actualKeywd))
			       + "\tText= " + mergeColumns(columns).substring(0,40)
);
	    if (thismaxIC > prevmaxIC) 
		prevmaxIC = thismaxIC;
	    else
		lastlap = true;
	    ++m;
	} // m loop
	//	if (m > MAX_TRIES)                            // Development use
	//	    System.out.print(" --MAX TRIES-- ");

	//	printBackupICs(icBackupTree);
	String keywds = reduce(shiftword.toString()) + getBackupKeywords(icBackupTree);
	return keywds;

	//	String repairedKeyword = repairShiftword(complementWord(shiftword.toString()), actualKeywd);   // Development
	//        return shiftword.toString();
	//	return complementWord(shiftword.toString());  // Return the encryption keyword
    }

    /**
     * getBackupKeywords returns of list of the keywords on the backup list.
     */
    private String getBackupKeywords(TreeMap tree) {
	String list = " ";
	Vector v = new Vector(tree.values());
	int k = 0;
	while (k < v.size() && k < MAX_BACKUPS) {
	    AlbertiAnalyzerBackupIC backup = (AlbertiAnalyzerBackupIC)v.get(k);
	    list = list + backup.shiftword + " ";
	    ++k;
	}
	return list;
    }

    /**
     * printBackupICs is used to test that the icBackupTree is working properly. When called
     *  it should print the entries in the tree sorted in increasing order by IC.
     */
    //    private void printBackupICs(TreeMap tree, String keyword) {
    private void printBackupICs(TreeMap tree) {
	Vector v = new Vector(tree.values());
	int k = 0;
	while (k < v.size()) {
	    AlbertiAnalyzerBackupIC backup = (AlbertiAnalyzerBackupIC)v.get(k);
	    System.out.println("backup " + k + ":" +  backup.toString());
	    //	    if (backup.shiftword.equals(keyword))
	    //		System.out.println(" CORRECT");
	    //	    else
	    //		System.out.println();
	    ++k;
	}
    }


    private int countWrongs(String s1, String s2) {
	int sum = 0;
	for (int k = 0; k < s1.length(); k++)
	    if (s1.charAt(k) != s2.charAt(k))
		++sum;
	return sum;
    }

    /**
     * complementWord() computes the 26-ch complement of a word
     */
    private String complementWord(String s) {
	StringBuffer result = new StringBuffer();             
	for (int k = 0; k < s.length(); k++) 
	    result.append((char)('a' + ((26 - s.charAt(k) + 'a') % 26) )); // Complement of s
	return result.toString();
    }

    /**
     * analyzeSimpleSubstitution() creates an NgramAnalyzer and uses it to analyze the text.
     * @param text -- the text to be analyzed.
     * @result -- a report giving the NgramAnalyzer's evaluation value and part of the decryption.
     */
    private String analyzeSimpleSubstitution(String text, String permKey) {
	String result="";
	try {
            Analyzer analyzer;
            analyzer = new NgramAnalyzer();
	    Alphabet alpha = AlphabetFactory.getInstance(AlphabetFactory.ALPH_az);
	    //	    System.out.println("solution= " + solution);
	    ((NgramAnalyzer)analyzer).setup(text+"$$$"+solution, params.book);
	    ((NgramAnalyzer)analyzer).setNgramAnalyzer(4,params.book,alpha,26,NgramAnalyzer.SIMPLESUB,100000);
	    //	    ((NgramAnalyzer)analyzer).setInitialSubstitution(permKey);
	    analyzer.run();
	    result =  analyzer.getReport(); 
	    //	    System.out.println(result);

	} catch (Exception e) {
	    System.out.println(e.toString());
	    e.printStackTrace();
	}
	return result;
    }


    /**
     * analyzeAndRetry() repeatedly calls NgramAnalyzer until the message is
     *   decrypted or it runs out of alternative shifts.  The alternative shifts
     *   are stored in icBackupTree.  Each alternative indicates a column in the
     *   keyword and a alternative shift for that column.  
     * @param columns -- an array storing the cryptogram broken into P columns,
     *   where P is the period.
     * @param shiftKeyword -- a String of letters of length P that holds 
     *  the encryption keyword, the (26-k) converse of which gives the shifts
     *  that produced the current columns
     * @param baseVal -- the initial evaluation returned by NgramAnalyzer.
     *   baseVal = Size/P/RawNgramValue
     *   
     */
    private void analyzeAndRetry(String[] columns, String shiftKeyword, double baseVal, String permKey) {
	System.out.println(mergeColumns(columns).substring(0,30));
	AlbertiAnalyzerBackupIC key =  null, 
	    backup = null;
	StringBuffer newShiftKeyword = new StringBuffer(shiftKeyword);
	String oldShiftKeyword = shiftKeyword;                   // Save the shiftKeyword
	double diff = 0, rawEval = 0, eval= 0, 
	    baseEval = baseVal;

	for (int k = 1; k <= MAX_RETRIES && icBackupTree.size() != 0; k++) {
	    
	    /**** Use the backup (column,shift) to modify the encryption keyword ****/
	    key = (AlbertiAnalyzerBackupIC)icBackupTree.firstKey();
	    backup = (AlbertiAnalyzerBackupIC)icBackupTree.remove(key);
	    shiftKeyword = oldShiftKeyword;       
	    newShiftKeyword = new StringBuffer(shiftKeyword);
	    //	    newShiftKeyword.setCharAt(backup.column, (char)('a' + (26-backup.shift)));  // Modify shift keyword 
	    newShiftKeyword.setCharAt(backup.column, (char)('a' + (backup.shift)));  // Modify shift keyword 
	    char oldCh = shiftKeyword.charAt(backup.column);                            // Calculate the shift
	    char newCh = newShiftKeyword.charAt(backup.column);
	    //	    System.out.println("Col= " + backup.column + " j= " + backup.shift + " old= '" + shiftKeyword.charAt(backup.column) + "' new= '" + newShiftKeyword.charAt(backup.column) + "'");
	    shiftKeyword = newShiftKeyword.toString();                                  
	    System.out.println("Retrying with " + shiftKeyword);

	    /**** Shift columns[column] by shift, giving a new simple substitution cryptogram.  ****/
	    String columnK = new String(columns[backup.column]);     // Preserve the column being shifted
	    int shift = 0;                                           // Calculate the shift
	    //	    if (oldCh > newCh)
	    //		shift = oldCh - newCh;
	    //	    else
	    //		shift = 26 + (oldCh - newCh);

	    if (newCh > oldCh)
		shift = newCh - oldCh;
	    else
		shift = 26 + (newCh - oldCh);
	    //	    System.out.println("Shifting column " + backup.column + " by " + shift);
	    columns[backup.column] = shift(columns[backup.column], shift);              // Rearrange the text
	    System.out.println(mergeColumns(columns).substring(0,30));


	    /***** Call NgramAnalyzer to analyze the text. *****/
	    String analysisReport = analyzeSimpleSubstitution(mergeColumns(columns), permKey);   // Retry the analysis
	    rawEval = Double.parseDouble(analysisReport.substring(analysisReport.indexOf("Value=") + 6, analysisReport.indexOf("DECRYPT")));
	    eval = (cleantext.length()/shiftLen)/rawEval;
	    diff = eval - baseEval;

	    if (eval >= EVAL_EPSILON) {             // Success
		System.out.println(analysisReport + "\t+"   + "\tRatio=" + eval);
		break;
	    } else if (diff < EVAL_DELTA) {        // No improvement
		columns[backup.column] = new String(columnK);  // Restore the column 
		shiftKeyword = oldShiftKeyword;    // Restore the keyword (Development)
		System.out.println(analysisReport + "\t-"  + "\tRatio=" + eval);
	    } else {                              // Improvement
		baseEval = eval;                  // Update the eval
		oldShiftKeyword = shiftKeyword;   // Forget the old shiftword (Development)
		System.out.println(analysisReport + "\t+"  + "\tRatio=" + eval);
	    }
	} // for max tries
    } // analyzeAndRetry()


    /**
     * ************AlbertiAnalyzerBackupIC ***************************
     * Inner class stores a record that can be used to repair the
     *  shift keyword. Gives the column and the shift that would
     *  produce the ic. 
     */
    class AlbertiAnalyzerBackupIC implements Comparator {
	public double ic;       // The Index of Coincidence Value
	public int column;      // The column number in the keyword
	public int shift;       // The shift value, 0..25
	public String shiftword; 

	public AlbertiAnalyzerBackupIC() { }

	public AlbertiAnalyzerBackupIC (double ic, int col, int shift, String s) {
	    this.ic = ic;
	    this.column = col;
	    this.shift = shift;
	    this.shiftword = new String(s);
	}

	/**
	 * Used to sort in increasing order. If two records have the same IC,
	 *  the column is used to break the tie.
	 */
	public int compare (Object o1, Object o2) {
	    if (((AlbertiAnalyzerBackupIC)o1).equals((AlbertiAnalyzerBackupIC)o2))
		return 0;
	    if ( ((AlbertiAnalyzerBackupIC)o1).ic > ((AlbertiAnalyzerBackupIC)o2).ic )
		return -1;
	    if ( ((AlbertiAnalyzerBackupIC)o1).ic == ((AlbertiAnalyzerBackupIC)o2).ic  && 
		 ((AlbertiAnalyzerBackupIC)o1).column < ((AlbertiAnalyzerBackupIC)o2).column )
		return -1;
	    return 1;
	}

	/**
	 * Two records are equal when their ICs and columns are equal.
	 */
	public boolean equals (Object o) {
	    return ((AlbertiAnalyzerBackupIC)o).ic == ic && ((AlbertiAnalyzerBackupIC)o).column == column;
	}

	public String toString() {
	    return "col= " + column + " shift= " + (char)('a' + shift) + " ic= " + ic + " " + shiftword;
	}
    }




}

