/*
 * File: PolyGaAnalyzer.java
 * @author R. Morelli <ralph.morelli@trincoll.edu> 
 * 
 * Description: This class assumes that the text is
 *  encrypted with simple substitution and uses a
 *  frequency-based genetic algorithm to analyze it.
 * 
 *  Its evaluation function uses an NgramArray with frequencies of
 *  N = 2, 3, or 4  (bigrams, trigrams, or  tetragrams) 
 *  to cryptanalyze transposition and substitution ciphers.
 *  Using the sum of the inverses of the frequencies as an evaluation of how
 *  accurately a decrypt matches the usual N-gram frequencies in a
 *  language is an idea described by Alex Griffing the developer of the
 *  "Automatic Cryptogram Solver".  This program is based on R. Walde's
 *  NgramClimber.java.
 *
 *  To compile and run from the TestAnalyzer application:
 *
 *  cd ~crypto/hcryptoj/1.4/applications/testanalyzer
 *  javac -classpath ../../classes -d ../../classes PolyGaAnalyzer.java
 *  java -classpath ../../classes:. TestGaAnalyzer analyzers.PolyGaAnalyzer ga_paramfiles/albertiparam.txt
 *
 */

package analyzers;

import hcrypto.analyzer.*;
import hcrypto.cipher.*;
import hcrypto.engines.*;
import hcrypto.provider.*;

public class PolyGaAnalyzer extends CryptoAnalyzer implements Analyzer 
{
    
    protected GaPopulation population;     

    /**
     * PolyGaAnalyzer() -- Default constructor
     */
    public PolyGaAnalyzer() 
    {
	super();
    }

    /**
     * PolyGaAnalyzer() -- this constructor is given an object containing parameter settings
     * @param params -- an object containing param1=val1 param2=val2 ...
     */
    public PolyGaAnalyzer(GaParameters params) 
    {
	super(params);
    }

    /**
     * setup() creates the population for the GA.
     */
    public void setup(String text) 
    {
	super.setup(text);
	if (params == null) 
	{
	    params = new GaParameters();       // Default GaParameters
	}
        this.text = text.toLowerCase();    // Overrides Analyzer.DECIPHER_LIMIT
        System.out.println("Revised CRYPTOGRAM Length= " + this.text.length());
	//	population = new PolyGaPopulation(this.text, this.solution, params);
    }

    private int findShiftLength(String text) {
        double ioc[] = new double[10];   // Index of Coincidence of shifts of 1..10
        for (int k = 0; k < 10; k++) {
            String subtext = getKthSubtext(text,k+1);
            IndexOfCoincidence IoC = new IndexOfCoincidence(subtext);
            ioc[k] = IoC.getIOC();
	    //            System.out.println("SUBTEXT["+(k+1)+"]\n " + subtext);
            System.out.println("IOC["+(k+1)+"] =" + ioc[k]);  // + "\t" + subtext);
	}
        int best = 1;
        double bestval = ioc[0];
        for (int k = 1; k < 10; k++) 
            if (Math.abs(ioc[k]-0.0656)  <  Math.abs(bestval-0.0656)) {
                best = k+1;
                bestval = ioc[k];
	    }
        return best;
    }

    private String getKthSubtext(String s, int k) {
        if (k <= 0) 
            throw new IllegalArgumentException("Skip size must be positive");
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < s.length(); i+=k)
            sb.append(s.charAt(i));
        return sb.toString();
    }

    private String[] getKColumns(String s, int k) {
        String columns[] = new String[k];
        if (k <= 0) 
            throw new IllegalArgumentException("Skip size must be positive");
	for (int j = 0; j < k; j++) {
            StringBuffer sb = new StringBuffer();
            for (int i = j; i < s.length(); i+=k) 
                sb.append(s.charAt(i));
            columns[j] = sb.toString();
	    //            System.out.println("Column " + j + ":" + columns[j]);
	}
        return columns;
    }

    private String mergeColumns(String columns[]) {
        StringBuffer sb = new StringBuffer();
        for (int k = 0; k < columns[columns.length-1].length(); k++) 
            for (int j = 0; j < columns.length; j++) 
                sb.append(columns[j].charAt(k));
        return sb.toString();
    }

    private String findShiftKeyword(String columns[]) {
        String text;
        StringBuffer  result = new StringBuffer();
        int shift = columns.length;

        String alltext = mergeColumns(columns);              
        System.out.println("MESSAGE=" + alltext.substring(0,27) + "...");
        IndexOfCoincidence IOC = new IndexOfCoincidence(alltext);
        System.out.println("IOC= " + IOC.getIOC());

	//	/***************
	//        result.append("ralph");
	//        System.out.println("Columns[0]= " + columns[0]);

	//       columns[0] = shift(columns[0], 26 - ('r'-'a'));
       //       columns[1] = shift(columns[1], 26 -  ('a'-'a'));
       //       columns[2] = shift(columns[2], 26 - ('l'-'a'));
       //       columns[3] = shift(columns[3], 26 - ('p'-'a'));
       //       columns[4] = shift(columns[4], 26 - ('h'-'a'));

       alltext = mergeColumns(columns);              
       System.out.println("MESSAGE1=" + alltext.substring(0,27) + "...");
       IOC = new IndexOfCoincidence(alltext);
       System.out.println("IOC= " + IOC.getIOC());

       //       columns[0] = shift(columns[0],  ('r'-'a'));
       //       columns[1] = shift(columns[1],  ('a'-'a'));
       //       columns[2] = shift(columns[2], ('l'-'a'));
       //       columns[3] = shift(columns[3], ('p'-'a'));
       //       columns[4] = shift(columns[4], ('h'-'a'));

       //       alltext = mergeColumns(columns);              
       //       System.out.println("MESSAGE=" + alltext.substring(0,27) + "...");
       //       IOC = new IndexOfCoincidence(alltext);
       //       System.out.println("IOC= " + IOC.getIOC());

       //       ********************/
       //        for (int k = 3; k < shift; k++) {
	for (int k = 0; k < shift; k++) {    // For each subtext 
            System.out.println("Subtext length " + k + " " +  columns[k].length());
            double maxioc = 0;
            int maxshift = 0;
            String maxstring = null;
            for (int j = 1; j <= 26; j++) { // Each possible shift
                StringBuffer sb = new StringBuffer();
                for (int c=0; c < columns[k].length(); c++) { // Shift each subtext letter
                    char ch = columns[k].charAt(c);
                    sb.append((char)('a' +  ((ch-'a') + 1) % 26));
		}
                columns[k] = sb.toString();
                text = mergeColumns(columns);              
		//                text = sb.toString();
                IndexOfCoincidence IoC = new IndexOfCoincidence(text);
                double ioc = IoC.getIOC();
		//                System.out.println("k=" + k + " j=" + (char)('a' + j % 26) + " ioc=" + ioc + " str= " +
		//                             columns[k].substring(0,10));
                if (ioc > maxioc) {
                    maxioc = ioc;
                    maxshift = j;
                    maxstring = columns[k];
		    //                    maxstring = text;
		}
	    } // j loop
            columns[k] = maxstring;
            char ch = (char)('z' - maxshift);
	    //            System.out.println("Best shift = " + ch + " The subtext was shifted by " + (char)('a' + (26 - (ch-'a'))));
            System.out.println("Best shift = " + (char)('a' + (maxshift % 26)) + " Subtext was shifted by " 
			       + (char)('z' - (maxshift % 26)));
            result.append(ch);
	} // k loop
        alltext = mergeColumns(columns);              
        System.out.println("MESSAGE=" + alltext.substring(0,27) + "...");
        IOC = new IndexOfCoincidence(alltext);
        System.out.println("IOC= " + IOC.getIOC());
	//        return result.toString();
        return alltext;
    }

    private String shift (String s, int n) {
        StringBuffer sb = new StringBuffer();
        for (int k = 0; k < s.length(); k++) {
            char ch = s.charAt(k);
            sb.append((char)('a' + ((ch - 'a') + n) % 26));
	}
        return sb.toString();
    }    

    /**
     * run() conducts the GA run and reports the results.
     */
    public void run() 
    {
	//	String text = "WKVHDG XWLNA BLDDLIS FI DWA XONN, \n"
	//            + "UJFPA YFXI OIY DWAI UASOI DF UOXN, \n"
	//	    + "UAZOKBA BFVA DGPA, ONN BDOJJG AGAY, \n"
	//		    + "WOY HOLIDAY ZFNFJB FI WLB WLYA.";

        String cleantext = TextUtilities.cleanString(text.toLowerCase());
        cleantext = TextUtilities.removeWhiteSpace(cleantext);
        System.out.println("CLEANTEXT\n" + cleantext.length());

	//       String fakecolumns[] = new String[5];
	//       for (int k = 0; k < 5; k++)
	//           fakecolumns[k] = cleantext;
	//       cleantext = mergeColumns(fakecolumns);
	//       System.out.println("CLEANTEXT\n" + cleantext);
       //       int shiftLen = 5;

        int shiftLen = findShiftLength(cleantext);
        String subtext[] = new String[shiftLen];
        System.out.println("The likely length of the shift keyword is " + shiftLen);
        System.out.println("We will now search for the letters in the shift keyword.");

	String columns[] = new String[shiftLen];
	columns = getKColumns(cleantext, shiftLen);

	//        String keyword = findShiftKeyword(columns);
        String shiftedText = findShiftKeyword(columns);
	try {
	    NgramAnalyzer nga = new NgramAnalyzer();
	    Alphabet alpha = AlphabetFactory.getInstance(AlphabetFactory.ALPH_az);
	    nga.setup(shiftedText, params.book);
	    nga.setNgramAnalyzer(4,params.book,alpha,26,NgramAnalyzer.SIMPLESUB,100000);
	    nga.run();
	    //	    System.out.println("DECRYPTED MESSAGE:\n " + nga.getPlainText());
	    System.out.println("\n" + nga.getReport());
	} catch (Exception e) {
	    System.out.println(e.toString());
	}
    }
}

