/*
 * File: ShiftLengthAnalyzer.java
 * @author R. Morelli <ralph.morelli@trincoll.edu>
 * 
 * Description: This class provides the public findShiftLength() method, which
 * determines the length of the shift keyword for a period polyalphabetic cipher.
 *
 * To compile: javac -classpath ../../classes/:. -d ../../classes/ ShiftLengthAnalyzer.java
 */

package analyzers;
import hcrypto.analyzer.*;

public class ShiftLengthAnalyzer extends CryptoAnalyzer implements Analyzer {

    private final int KEYWD_MAX = 34;
    private final double IC_EPSILON = 0.008;    

    /**
     * ShiftLengthAnalyzer() -- Default constructor
     */
    public ShiftLengthAnalyzer() {
	super();
    }

    /**
     * ShiftLengthAnalyzer() -- this constructor is given an object containing
     *  parameter settings, param1=val1, param2=val2, ...
     * @param params -- an object containing param1=val1 param2=val2 ...
     */
    public ShiftLengthAnalyzer(GaParameters params) {
	super(params);
    }

    /**
     * setup() sets up the text and the parameters.. 
     */
    public void setup(String text) {
	super.setup(text);
	if (params == null) {
	    params = new GaParameters(); // Default GaParameters
	}
    }

    /**
     * run() conducts the cryptanalysis and reports the result
     */
    public void run() {
        String cleantext = TextUtilities.cleanString(text.toLowerCase());
        cleantext = TextUtilities.removeWhiteSpace(cleantext);
	String secretshiftkeyword = this.solution.substring(this.solution.indexOf("#"),this.solution.length()-1);
	String actualKeywd = secretshiftkeyword.substring(secretshiftkeyword.indexOf("#")+1,secretshiftkeyword.indexOf("("));
        int actualKeywdLen = Integer.parseInt(secretshiftkeyword.substring(secretshiftkeyword.indexOf("(")+1,secretshiftkeyword.indexOf(")")));
        int shiftLen = findShiftLength(cleantext);
	java.text.NumberFormat nf = java.text.NumberFormat.getNumberInstance();
        System.out.print("\tP= " + shiftLen + "\tActualShift= " + actualKeywdLen + "\tN/P= " + nf.format(1.0*cleantext.length()/actualKeywdLen));
    }


    /**
     * findShiftLength() determines the length of the Alberti shift key through
     *  an exhaustive search. For each possible length, up to KEYWD_MAX, we break
     *  the text into columns and compute the average IC of the columns.  The
     *  length with the greatest IC is returned.  As an optimization, we break
     *  out of the search when we find an IC with 0.008 or 0.066. This works well
     *  for shifts that are not prime numbers, whose multiples would frequently
     *  show a higher IC than the correct lower value. 
     */
    public int findShiftLength(String text) {
        double ioc[] = new double[KEYWD_MAX];   // Index of Coincidence of shifts of 1..10
	boolean done = false;
	int best = 1;
        double bestval = ioc[0];
        for (int k = 0; k < KEYWD_MAX; k++) {
	    double iclist[] = getICs(getKColumns(text,k+1));
	    ioc[k] = mean(iclist);
            if (Math.abs(ioc[k]-0.066)  <  IC_EPSILON || ioc[k] > IC_EPSILON + 0.066) {
		best = k+1;
		bestval = ioc[k];
		done = true;
		break;
	    }
	}
        for (int k = 1; !done && k < KEYWD_MAX; k++) { 	// If best value not already found, use the highest value
            if (ioc[k] >  bestval) {
                best = k+1;
                bestval = ioc[k];
	    }
	}
        return best;
    }

    /**
     * returns an array of the ICs for each string in the
     *  array, s.
     */
    private double[] getICs(String[] s) {
	double ic[] = new double[s.length];
	for (int k = 0; k < s.length; k++) {
	    IndexOfCoincidence IC = new IndexOfCoincidence(s[k]);
	    ic[k] = IC.getIOC();
	}
	return ic;
    }
    /**
     * returns the mean of an array 
     */
    private double mean(double[] a) {
	double sum = 0;
	for (int k = 0; k < a.length; k++)
	    sum += a[k];
	return sum/a.length;
    }
    /**
     * returns the variance of an array
     */
    private double variance(double[] a, double mean) {
	double v = 0;
	for (int k = 0; k < a.length; k++)
	    v += (a[k]-mean) * (a[k]-mean);
	return v/a.length;
    }

    /**
     * getKColumns() converts the text, s, into an array of k columns. If k
     *  corresponds to the period of the Alberti cipher, then each column
     *  is a simple substitution cipher.
     * @param s -- the ciphertext
     * @param k -- the period
     */
    private String[] getKColumns(String s, int k) {
        String columns[] = new String[k];
        if (k <= 0) 
            throw new IllegalArgumentException("Skip size must be positive");
	for (int j = 0; j < k; j++) {
            StringBuffer sb = new StringBuffer();
            for (int i = j; i < s.length(); i+=k) 
                sb.append(s.charAt(i));
            columns[j] = sb.toString();
	    //            System.out.println("Column " + j + ":" + columns[j]);
	}
        return columns;
    }

    /**
     * mergeColumns() merges the columns into a single string of text.
     *  This method is used to reconstruct the original message
     * @param columns -- an array of the k simple substitution subtexts
     *  of the original message.
     */
    private String mergeColumns(String columns[]) {
	int d = Math.min(columns[0].length(), columns[columns.length-1].length());
        StringBuffer sb = new StringBuffer();
        for (int k = 0; k < d; k++) 
            for (int j = 0; j < columns.length; j++) 
                sb.append(columns[j].charAt(k));
	for (int j = 0; j < columns.length; j++)
	    if (columns[j].length() > d)
		sb.append(columns[j].charAt(d));
        return sb.toString();
    }

    /**
     * substitute() applies the permutation alphabet to a cryptotext
     */
    private String substitute(String s, String alpha) {
	StringBuffer sb = new StringBuffer();
	for (int k = 0; k < s.length(); k++)
	    sb.append(alpha.charAt(s.charAt(k)-'a'));
	return new String(sb.toString());
    }


    /**
     * shiftText() breaks text into keywd.length() columns,
     *  shifts each columns by the keywd[k], and returns the merged
     *  columns.
     */
    private String shiftText(String text, String keywd) {
	int period = keywd.length();
	String columns[] = new String[period];
	columns = getKColumns(text, period);
	for (int k = 0; k < period; k++) 
	    columns[k] = shift(columns[k], keywd.charAt(k)-'a');
	return mergeColumns(columns);
    }
    
    /**
     * shift() shifts each character in the string s by n, assuming a 26
     *  letter alphabet
     * @param s, a text
     * @param n -- the amount to shift each letter
     */
    private String shift (String s, int n) {
        StringBuffer sb = new StringBuffer();
        for (int k = 0; k < s.length(); k++) {
            char ch = s.charAt(k);
            sb.append((char)('a' + ((ch - 'a') + n) % 26));
	}
        return sb.toString();
    }    

    /**
     * reduces a shift keyword such as 'ralph' to one that
     *  begins with an 'a', such as 'ajuyq' -- that is with a 0 shift in column 0
     */
    public static String reduce(String key) {
	StringBuffer sb = new StringBuffer(key);
	int diff = key.charAt(0) - 'a';
	//	System.out.println("Reduce by " + diff);
	for (int k = 0; k < (26 - diff) % 26 ; k++) 
	    for (int j = 0; j < key.length(); j++)
		sb.setCharAt(j, (char)('a' + ((sb.charAt(j) - 'a' + 1) % 26)));
	return sb.toString();
    }
}
