package hcrypto.analyzer; // This class in part of the AlbertiGaAnalyzer analyzer in hcrypto 

import hcrypto.cipher.*;
import hcrypto.engines.*;
import hcrypto.provider.*;

public class AlbertiGaIndividual extends GaIndividual implements Comparable
{
    private final boolean MUTATE_BENEFICIAL = true;  // Mutate only if beneficial
    private final char SPACE = ' ';
    private NgramArray ngramArr;
    private int NN;  // Ngram length = 2, 3, 4, ...
    //    private int index[] = new int[2];          // Stores location of max(worst) and min(best) nGram scores
    //    private double scores[]; // records the scores of individuals
    private int dividePos;
    private AlbertiKey aKey;
    private final char divide = ',';
    private int shiftLen;
    private String text;

    private String shiftKey;
    //    private StringBuffer permKey;

    public AlbertiGaIndividual(String text, String permKey, String shiftKey, GaParameters params, NgramArray nga)  {
	//	super(text, permKey + "," + shiftKey, params);
	super(text, permKey, params);
	this.text = text;
	//	this.permKey = new StringBuffer(permKey);
	this.shiftKey = shiftKey;
	this.shiftLen = shiftKey.length();
	ngramArr = nga;
	NN = ngramArr.getNN();
	//	scores = new double[text.length()];
	mutate_policy = params.mutate_policy;
        calcFitness();
    }

    /**
     * Copy constructor
     */
    public AlbertiGaIndividual (GaIndividual i) 
    {
	super(i);
	ngramArr = ((AlbertiGaIndividual)i).ngramArr;
	NN = ngramArr.getNN();
	shiftKey = ((AlbertiGaIndividual)i).shiftKey;
	//	permKey = ((AlbertiGaIndividual)i).permKey;
	this.text = ((AlbertiGaIndividual)i).text;
	//	for (int k = 0; k < index.length; k++)
	//	    index[k] = ((AlbertiGaIndividual)i).index[k];
	
	//	scores = new double[((AlbertiGaIndividual)i).scores.length];
	//	for (int k = 0; k < scores.length; k++)
	//	    scores[k] = ((AlbertiGaIndividual)i).scores[k];
    }

    public boolean equals(Object i) {
        return key.equals(((NgramGaIndividual)i).key);
    }

    /** 
     *  One-point crossover: Pick a random point in i1 and i2
     *   and swap up to NN+1 characters starting from that point.
     */
    public void cross(GaIndividual i) 
    {
	AlbertiGaIndividual i1 = this;
	AlbertiGaIndividual i2 = (AlbertiGaIndividual)i;

	//	System.out.println("i1 befor: " + i1.getKey());
	//	System.out.println("i2 befor: " + i2.getKey());

	int p = (int)(Math.random() * key.length());
	//	int pstop = p + NN+1;
	int pstop = p + NN+2;
	//	while (p < pstop && p < key.length()) 
	//	System.out.println("Swapping from " + p + " to " + pstop);
	while (p < pstop && p < key.length()) 
	{
	    char ch1 = i1.key.charAt(p);
	    char ch2 = i2.key.charAt(p);
	    i1.swap(ch1, ch2);
	    i2.swap(ch1, ch2);
	    p++;
	}
	//	System.out.println("i1 after: " + i1.getKey());
	//	System.out.println("i2 after: " + i2.getKey());
	
	i1.calcFitness();  
	i2.calcFitness();
    }

    protected void swap (char ch1, char ch2) {
	//	System.out.print("key " + key + " --> ");
        int m = key.toString().indexOf(ch1);
        int n = key.toString().indexOf(ch2);
        key.setCharAt(m, ch2);
        key.setCharAt(n, ch1);        
	//	System.out.println("Swap " + ch1 + " " + ch2 + " " + key);
    }

    /**
     * This version of mutate swaps two random characters
     *  in the key. The mutation is done with frequency equal
     *  to rate. The resulting mutant is discarded if it
     *  does not improve the fitness of the individual.
     * @param rate is the mutation rate
     */
    public int mutate(double rate) {
	int a = 0, b = 0;

        if (Math.random() < rate) {
	    AlbertiGaIndividual tempIndy = new AlbertiGaIndividual(this);
            a = (int)(Math.random() * 26);
            b = (int)(Math.random() * 26);
            char ch = key.charAt(a);
            key.setCharAt(a, key.charAt(b));
            key.setCharAt(b, ch);
            this.calcFitness();
	    if (mutate_policy == GaParameters.ELITIST_MUTATION && this.compareTo(tempIndy) > 0)  {     // We mutate only if better
		this.key = tempIndy.key;             // So, restore the individual's state  if mutation is not an improvement
		this.fitness = tempIndy.fitness;
		this.decrypt = tempIndy.decrypt;
		return 0;
	    } else {
		return 1;
	    }
        } else {
	    return 0;  
	}
    }

    public void calcFitness() {
	try {
	    String substituteText = new String(substitute(this.text,key));         
	    //	    String decrypt = shiftText(substituteText, shiftKey.toString());
	    decrypt = shiftText(substituteText, shiftKey.toString());
	    fitness = ngramArr.recipDistSkip(decrypt);
	    //	    System.out.println("Key = " + key + " shift= " + shiftKey + " score= " + fitness);
	} catch (Exception e) {
	    e.printStackTrace();
	}
    }


    public double calcIC(int nCols) {
	String substituteText = new String(substitute(this.text,key));         
	String decrypt = shiftText(substituteText, shiftKey.toString());
	FastIC fastIC = new FastIC(decrypt, shiftKey.length());
	double sum = 0;
	for (int k = 0; k < nCols; k++)
	    sum += fastIC.getColumnIC(k);
	return sum / nCols;
    }


    public double calcCHI(int nCols) {
	String substituteText = new String(substitute(this.text,key));         
	String decrypt = shiftText(substituteText, shiftKey.toString());
	FastIC fastIC = new FastIC(decrypt, shiftKey.length());
	double sum = 0;
	for (int k = 0; k < nCols; k++)
	    sum += fastIC.getBestChiTestPerEnglish(k);
	return sum / nCols;
    }


    private String substitute(String s, StringBuffer key) {
	//	System.out.println("length = " + s.length());
	StringBuffer sb = new StringBuffer();
	for (int k = 0; k < s.length(); k++) {
	    sb.append(key.charAt(s.charAt(k)-'a'));
	}
	return new String(sb.toString());
    }

    /**
     * shiftText() breaks text into keywd.length() columns,
     *  shifts each columns by the keywd[k], and returns the merged
     *  columns.
     */
    private String shiftText(String text, String keywd) {
	int period = keywd.length();
	String columns[] = new String[period];
	columns = getKColumns(text, period);
	for (int k = 0; k < period; k++) 
	    columns[k] = shift(columns[k], keywd.charAt(k)-'a');
	return mergeColumns(columns);
    }

    /**
     * shift() shifts each character in the string s by n, assuming a 26
     *  letter alphabet
     * @param s, a text
     * @param n -- the amount to shift each letter
     */
    private String shift (String s, int n) {
        StringBuffer sb = new StringBuffer();
        for (int k = 0; k < s.length(); k++) {
            char ch = s.charAt(k);
            sb.append((char)('a' + ((ch - 'a') + n) % 26));
	}
        return sb.toString();
    }    


    /**
     * mergeColumns() merges the columns into a single string of text.
     *  This method is used to reconstruct the original message
     * @param columns -- an array of the k simple substitution subtexts
     *  of the original message.
     */
    private String mergeColumns(String columns[]) {
	int d = Math.min(columns[0].length(), columns[columns.length-1].length());
        StringBuffer sb = new StringBuffer();
        for (int k = 0; k < d; k++) 
            for (int j = 0; j < columns.length; j++) 
                sb.append(columns[j].charAt(k));
	for (int j = 0; j < columns.length; j++)
	    if (columns[j].length() > d)
		sb.append(columns[j].charAt(d));
        return sb.toString();
    }

    /**
     * getKColumns() converts the text, s, into an array of k columns. If k
     *  corresponds to the period of the Alberti cipher, then each column
     *  is a simple substitution cipher.
     * @param s -- the ciphertext
     * @param k -- the period
     */
    private String[] getKColumns(String s, int k) {
        String columns[] = new String[k];
        if (k <= 0) 
            throw new IllegalArgumentException("Skip size must be positive");
	for (int j = 0; j < k; j++) {
            StringBuffer sb = new StringBuffer();
            for (int i = j; i < s.length(); i+=k) 
                sb.append(s.charAt(i));
            columns[j] = sb.toString();
	    //            System.out.println("Column " + j + ":" + columns[j]);
	}
        return columns;
    }

    /**
     *  Overrides the default version
     */
    public int compareTo(Object o) {
        AlbertiGaIndividual indy = (AlbertiGaIndividual)o;
	
	//System.out.println(this.fitness + " vs.  " + indy.fitness);
        
	if (this.fitness < indy.fitness)
            return -1;
        else if (this.fitness > indy.fitness)
            return +1;
        else
            return 0;
    }

    public String getKey() {
	return key + "," + shiftKey;
    }

    public String getShiftKey() {
	return shiftKey;
    }

    public String getPermKey() {
	return key.toString();
    }

}

