/*
 * File: AlbertiGaPopulation.java
 * @author R. Morelli <ralph.morelli@trincoll.edu>
 *
 * Description: This class implements an instance of a GaPopulation
 *  by providing implementations of the init()
 *  and run() methods for a AlbertiGaAnalyzer.
 */

package hcrypto.analyzer;

import java.util.*;
import java.text.*;
import hcrypto.cipher.*;

public class AlbertiGaPopulation extends GaPopulation {
    public static final int N_SHUFFLES = 20;
    private final int MAX_TRIES = 7;        // For findShiftKeyword()

    private int NN;
    private String book;
    private Alphabet alphabet;
    private NgramArray ngramArr;
    private NumberFormat num;
    private int keyCount = 0;
    private int shiftLen = 0;  // shiftLen controls the length of the randomized Keyword2 (AlbertiKey) solutions
    private char chseed[];
    private String correctPermKey = "qreptizgkfsuvwxyabcdhjlmno";
    private GaParameters params;
    private int nWrong = 0;
    private int skipLen = 1;

    private int phase = 0;
    private String correctShiftword;
    private String bestKey = "";

    //    private GaIndividual phase1Best[];
    private int correctPhaseOneIndividual;
    private int correctPhaseTwoIndividual;
    private int correctPhaseThreeIndividual;

    private GaIndividual phase1Winners[];
    private GaIndividual phase2Winners[];
    private GaIndividual phase3Winners[];

    /**
     * AlbertiGaPopulation() constructor
     * @param text -- the text to be cryptanalyzed
     * @param params -- a collection of parameters to control the run
     */

    //    public AlbertiGaPopulation (String text, String solution, GaParameters params, int shiftlen) {
    //    public AlbertiGaPopulation (String text, String solution, GaParameters params, int shiftlen, String shiftwd) {
    public AlbertiGaPopulation (String text, String solution, GaParameters params, int shiftlen, String keyspec) {
	super(text, solution, params);
	//	correctShiftword = shiftwd;
	correctShiftword = keyspec.substring(keyspec.indexOf(',')+1);
        correctPermKey = keyspec.substring(0, keyspec.indexOf(','));
	//	System.out.println("permkey= " + correctPermKey + " shiftkey= " + correctShiftword);
	this.params = params;
	NN = params.NN;
	book = params.book;
	alphabet = params.alphabet;
	num = NumberFormat.getInstance();
	num.setMaximumFractionDigits(2);
	this.shiftLen = shiftlen;
	phase = 1;
	skipLen = this.shiftLen;
	initNgramArray();
    }

    public void init() {}

    /**
     * Default init method.
     */
    private void initNgramArray() {
	try 
	{
	    Alphabet alphabet = AlphabetFactory.getInstance(AlphabetFactory.ALPH_az);
	    //	    ngramArr = new NgramArray(NN, book, alphabet);
	    ngramArr = new NgramArray(NN, book, alphabet, skipLen);
	} 
	catch (Exception e) 
	{
	    e.printStackTrace();
	}
    }

    private void initPopulation(String permKey, String shiftKeys) {
	StringTokenizer st = new StringTokenizer(shiftKeys);
	String shiftWords[] = new String[st.countTokens()];
	int k = 0;
	while (st.hasMoreTokens()) {
	    shiftWords[k++] = st.nextToken();
	}
	int j = 0;
        individual = new AlbertiGaIndividual[size * 2];
        for (k = 0; k < individual.length; k++) {
	    individual[k] = new AlbertiGaIndividual(cleantext, permKey, shiftWords[j], params, ngramArr);
	    j++;
	    //	    System.out.println("Individual shiftKey= " + shiftWords[j]);
	    //	    individual[k] = new AlbertiGaIndividual(cleantext, permKey, "ajuyqrn", params, ngramArr);
	    if (j == shiftWords.length)
		j = 0;
	    ++keyCount;
	}
    }

    private void initPopulation(int pool, int gen) {
        individual = new AlbertiGaIndividual[size * 2];

	String pkey = "";
	String skey = "";

        for (int k = 0; k < individual.length; k++) {
	    pkey = makePermutationKey(pool);
	    if (phase == 1)
		skey = makeShiftKey(shiftLen, pool, k);
	    else
		skey = makeShiftKey(shiftLen, pool, gen);
	    //	    System.out.println(k + " Key= " + pkey + "," + skey);
	    individual[k] = new AlbertiGaIndividual(cleantext, pkey, skey, params, ngramArr);
	    ++keyCount;
	}
    }

    /**
     * This method creates either a randomized encryption key for
     *  Alberti Cipher (phase 1), or uses an initial permutation key from the
     *  parameter file (phase 2). 
     */
    private String makePermutationKey(int pool) {
	if (phase == 1 || phase == 4)
	    return shuffle("abcdefghijklmnopqrstuvwxyz");
	else if (phase == 2)
	    return ((AlbertiGaIndividual)phase1Winners[pool]).getPermKey();
	else 
	    return ((AlbertiGaIndividual)phase2Winners[pool]).getPermKey();
    }
    
    /**
     * shiftKey() creates an the shift key, which remains constant in
     *  phases 1 and 2. 
     * @param n gives the length of the shift key.
     * @param pool gives the pool number
     * @param k gives the individual's number, which determines the last letter
     */
    private String makeShiftKey(int n, int pool, int k) {
	String tempKey;
	if (phase == 1)
	    tempKey = "a" + (char)('a' + pool % 26);
	    //    tempKey = "aj";
	else if (phase == 2)
	    tempKey = ((AlbertiGaIndividual)phase1Winners[pool]).getShiftKey().substring(0,2)
		+ (char)('a' + k % 26); 
	else 
	    tempKey = ((AlbertiGaIndividual)phase2Winners[pool]).getShiftKey().substring(0,3)
		+ (char)('a' + k % 26); 

	n -= tempKey.length();
	for(int i = 0; i < n; i++) {
	    tempKey += (char)('a'+(int)(Math.random()*26));		
	}
	return tempKey;
    }

    private int countWrongs(String s1, String s2) {
	if (s1.length() != s2.length())
	    return Math.max(s1.length(), s2.length());
	int sum = 0;
	for (int k = 0; k < s1.length(); k++)
	    if (s1.charAt(k) != s2.charAt(k))
		++sum;
	return sum;
    }

    // PHASE 1: 2-letter shifts **********************
    private void runPhaseOne() {  
	phase1Winners = new GaIndividual[26];
	for (int k = 0; k < 26; k++) {  // For each of 26 pools
	    initPopulation(k, 0);
	    runPool();
	    phase1Winners[k] = individual[0];
	}
	//	System.out.println("Phase 1 winners");
	java.util.Arrays.sort(phase1Winners);    
	for (int k = 0; k < 26; k++) {
	    String key = phase1Winners[k].toString();
	    String shiftwd = key.substring(key.indexOf(',')+1);
	    if (shiftwd.substring(0,2).equals(correctShiftword.substring(0,2))) {
		String permkey = key.substring(0, key.indexOf(','));
		int nWrong = countWrongs(correctPermKey, permkey);
		System.out.println("PHASE_1: index= " + k + "\tNN= " + params.NN + "\tnWrongPermKey= " + nWrong 
				   + "\tkey= " + key + "\t" + num.format(phase1Winners[k].getFitness())
				   + "\tkeyCount= " + keyCount);
		correctPhaseOneIndividual = k;
	    }
	}
    }

    // PHASE 2: 3-letter shifts **********************
    private void runPhaseTwo() {
	phase = 2;
	params.NN = NN = 3;
	//	size = 104;
	initNgramArray();
	//	/*****************  Production Version
	phase2Winners = new GaIndividual[26];
	for (int j = 0; j < 26; j++) { // For 'a' to 'z'
	    initPopulation(correctPhaseOneIndividual,j); // TESTING: Just use correct shiftword prefix
	    nWrong = 0;
	    runPool();
	    phase2Winners[j] = individual[0];
	}
	java.util.Arrays.sort(phase2Winners);    
	for (int i = 0; i  < 26; i++) {
	    String key = phase2Winners[i].toString();
	    //	    System.out.println(key);
	    String shiftwd = key.substring(key.indexOf(',')+1);
	    if (shiftwd.substring(0,3).equals(correctShiftword.substring(0,3))) {
		correctPhaseTwoIndividual = i;
		bestKey = phase2Winners[correctPhaseTwoIndividual].toString();
		String permkey = key.substring(0, key.indexOf(','));
		int nWrong = countWrongs(correctPermKey, permkey);
		System.out.println("PHASE_2: index= " + i + "\tNN= " + params.NN + "\tnWrongPermKey= " + nWrong 
				   + "\tkey= " + key + "\t" + num.format(phase2Winners[i].getFitness())
				   + "\tkeyCount= " + keyCount);
	    }
	}
    }

    private void runPhaseThree() {
	phase = 3;
	params.NN = NN = 4;
	initNgramArray();
	phase3Winners = new GaIndividual[26];
	for (int j = 0; j < 26; j++) { // For 'a' to 'z'
	    initPopulation(correctPhaseTwoIndividual,j); // TESTING: Just use correct shiftword prefix
	    nWrong = 0;
	    runPool();
	    phase3Winners[j] = individual[0];
	}
	java.util.Arrays.sort(phase3Winners);    
	for (int i = 0; i  < 26; i++) {
	    String key = phase3Winners[i].toString();
	    //	    System.out.println(key);
	    String shiftwd = key.substring(key.indexOf(',')+1);
	    if (shiftwd.substring(0,4).equals(correctShiftword.substring(0,4))) {
		correctPhaseThreeIndividual = i;
		bestKey = phase3Winners[correctPhaseThreeIndividual].toString();
		String permkey = key.substring(0, key.indexOf(','));
		int nWrong = countWrongs(correctPermKey, permkey);
		System.out.println("PHASE_3: index= " + i + "\tNN= " + params.NN + "\tnWrongPermKey= " + nWrong 
				   + "\tkey= " + key + "\t" + num.format(phase3Winners[i].getFitness())
				   + "\tkeyCount= " + keyCount);
	    }
	}
    }

    public void runPhaseFour(String shiftKeys) {
	phase = 4;
	size = 416;
	params.NN = NN = 4;
	skipLen = 1;
	initNgramArray();
	initPopulation(getBestPermutationKey(), shiftKeys);
	//	initPopulation(makePermutationKey(0), shiftKeys);
	runPool();
	System.out.println("PHASE_4: nIters= " + iterations 
			   + "\tNN= " + params.NN
			   + "\tnWrongPermKey= " + countWrongs(correctPermKey, ((AlbertiGaIndividual)individual[0]).getPermKey())
			   + "\tnWrongShifts= " + countWrongs(correctShiftword, ((AlbertiGaIndividual)individual[0]).getShiftKey())
			   + "\tkey= " + ((AlbertiGaIndividual)individual[0]).getKey()
			   + " " + num.format(bestScore) 
			   + " Decrypt= " +  individual[0].getDecrypt().substring(0,30)
			   + "\tkeyCount= " + keyCount
			   );

    }

    /**
     * This method implements the abstract run() method inherited from
     *  GaPopulation.
     */
    public void run() {
	printStartMessage();
	runPhaseOne();
	runPhaseTwo();
	runPhaseThree();
    }

    public String getBestPermutationKey() {
	//	String bestkey= "ywzuptmnhcsiexdjlvrbaokqgf,achhnkvrfowmnsiiawez"; // 3000_1  10 wrong
	//	String bestkey= "ywzputmnhcsiexdjlvrbaokqgf,achhnkvrfowmnsiiawez"; // 3000_1  8 wrong
	//	String bestkey= "wyzputmnhcsiexdjlvrbaokqgf,achhnkvrfowmnsiiawez"; // 3000_1  6 wrong  ==> Converges
	//	String bestkey= "wyzputmdhcsnexrjlvibaokqgf,achzffdtheivwzyjnuzn";  // hard20shift3000_1.txt  3 wrong
	//	String bestkey= "wyzputmdhcsnaxrjlvebiokqgf,achlejkkkxxanssajuzk";  // 3000_1  0 wrong perm,   ==> Diverges
	//	String bestkey= "vexwlcjsmndbuyqrpfogzkihat,alsmsgguslngzevqlswz"; // hard20shift3000_0.txt 
	//	String bestkey = "sgdlwqtxizrmjaneucfopvbkhy,airmdwxebpldroivprvg"; // hard20shift3000_4.txt
	//	String bestkey = "lmjucpgzvsakeohwdtnbyrixqf,afjvyydaeekykoekvjpk";  // hard20shift3000_2.txt   8 wrong
	//	String bestkey = "lmjucpgzvsakeohwdtnbyrfqxi,afjvyydaeekykoekvjpk";  // hard20shift3000_2.txt   4 wrong
	//	String bestkey = "mljucpgzvsakeohwdtnbyrfqxi,afjvyydaeekykoekvjpk";  // hard20shift3000_2.txt   2 wrong
	//	String bestkey = "mljucpgzvsakeohtdwnbyrfqxi,afjvyydaeekykoekvjpk";  // hard20shift3000_2.txt           0 wrong

	//	String bestkey = "qrehsizgkfmlvwxjaptucdybno,ajuimac";  // 12 wrong
	//	String bestkey = "qrehtizgkfmuvwxyabcdsjlpno,ajuyyjo"; // 4 wrong
	//	String bestkey = "qrehtizgkfsuvmxjabcwydlpno,ajusqsf";    // 7 wrong  ==> shiftKey 7 wrong
	//	String bestkey = "qreptizgkfsuvwxyabcdhjlmno,ajulhva";    // 0 wrong  ==> shiftkey 0 wrong
	//	String bestkey = "qrehtizgkfsuvwxyabcdmjlpno,ajuneyo";    // 3 wrong  ==> shiftKey 0 wrong
	//	String bestkey = phase2Winners[0].toString();

	//	String bestkey = phase2Winners[correctPhaseTwoIndividual].toString();
	String bestkey = bestKey;
	return bestkey.substring(0,bestkey.indexOf(","));
    }

    public String getBestShiftKey() {
	//	String bestkey= "ywzuptmnhcsiexdjlvrbaokqgf,achhnkvrfowmnsiiawez"; // 3000_1  10 wrong
	//	String bestkey= "ywzputmnhcsiexdjlvrbaokqgf,achhnkvrfowmnsiiawez"; // 3000_1  8 wrong
	//	String bestkey= "wyzputmnhcsiexdjlvrbaokqgf,achhnkvrfowmnsiiawez"; // 3000_1  6 wrong
	//	String bestkey= "wyzputmdhcsnexrjlvibaokqgf,achzffdtheivwzyjnuzn";  // hard20shift3000_1.txt  3 wrong
	//	String bestkey= "wyzputmdhcsnaxrjlvebiokqgf,achlejkkkxxanssajuzk";  // 3000_1  0 wrong
	//	String bestkey= "vexwlcjsmndbuyqrpfogzkihat,alsmsgguslngzevqlswz"; // hard20shift3000_0.txt
	//	String bestkey = "sgdlwqtxizrmjaneucfopvbkhy,airmdwxebpldroivprvg"; // hard20shift3000_4.txt
	//	String bestkey = "lmjucpgzvsakeohwdtnbyrixqf,afjvyydaeekykoekvjpk";  // hard20shift3000_2.txt   8 wrong
	//	String bestkey = "lmjucpgzvsakeohwdtnbyrfqxi,afjvyydaeekykoekvjpk";  // hard20shift3000_2.txt   4 wrong
	//	String bestkey = "mljucpgzvsakeohwdtnbyrfqxi,afjvyydaeekykoekvjpk";  // hard20shift3000_2.txt   2 wrong
	//      String bestkey = "mljucpgzvsakeohtdwnbyrfqxi,afjvyydaeekykoekvjpk";  // hard20shift3000_2.txt           0 wrong

	//	String bestkey = "qrehsizgkfmlvwxjaptucdybno,ajuimac";  // 12 wrong
	//	String bestkey = "qrehtizgkfmuvwxyabcdsjlpno,ajuyyjo"; // 4 wrong
	//	String bestkey = "qrehtizgkfsuvmxjabcwydlpno,ajusqsf";    // 7 wrong  ==> shiftKey 7 wrong
	//	String bestkey = "qreptizgkfsuvwxyabcdhjlmno,ajulhva";    // 0 wrong  ==> shiftkey 0 wrong
	//	String bestkey = "qrehtizgkfsuvwxyabcdmjlpno,ajuneyo";    // 3 wrong  ==> shiftKey 0 wrong
	//	String bestkey = phase2Winners[0].toString();

	//String bestkey = phase2Winners[correctPhaseTwoIndividual].toString();
	String bestkey = bestKey;
	return bestkey.substring(bestkey.indexOf(",")+1);
    }

    private void runPool() {
	int ptr = 0;
	iterations = 0;
        previousBestScore = Double.MAX_VALUE;
	java.util.Arrays.sort(individual);    // Sort the population
	//	displayAll();
        do 
	{
	    ptr = size;
	    for (int k = 0; k < size/2; k++) {      // Perform crosses on k andj 		                                   
		int j = size/2 +(int)(Math.random()* size/2); 		                 
		//		int j = (int)(Math.random()* size); 		                 
		double rand = Math.random();                 
		if (rand <= cross_rate) {
		    GaIndividual i1 = new AlbertiGaIndividual(individual[k]);
		    GaIndividual i2 = new AlbertiGaIndividual(individual[j]);
		    i1.cross(i2);
		    individual[ptr++] = i1;   // CHILDREN AND PARENTS COMPETE
		    individual[ptr++] = i2;
		}
	    } 
	    mutateAll();                          // Mutate all	    
            ++iterations;
	    java.util.Arrays.sort(individual);    // Sort the population
	    nWrong = countWrongs(correctPermKey, ((AlbertiGaIndividual)individual[0]).getPermKey());
	    //	    displayAll();
	    updateScore();
	    if (verbose) 
	       displayBest();
	    keyCount += individual.length;
        } 
	//	while(iterations < maxtrials && !success(individual[0]) && nWrong != 0);
	while(iterations < maxtrials && !success(individual[0]));
	//	displayAll();
	if (verbose)
	    displaySummary();
    }

    private boolean success(GaIndividual indy) {
	if (bestScore == individual[individual.length/2].getFitness()) 
	{
	    return true;
	} 
	else
	    return false;
    }

    public GaIndividual getFittest(int n) 
    {
        return individual[n];
    }

    /**
     * shuffle() randomly shuffles the characters in a StringBuffer.
     */
    private void shuffle(StringBuffer sb) 
    {
	sb = new StringBuffer(shuffle(sb.toString()));
    }

    /**
     * shuffle() randomly shuffles the characters in a String.
     */
    private String shuffle(String s) 
    {
	StringBuffer sb = new StringBuffer(s);
	for (int k = 0; k < N_SHUFFLES; k++) 
	{
            int a = (int)(Math.random() * sb.length());
            int b = (int)(Math.random() * sb.length());
            char ch = sb.charAt(a);
            sb.setCharAt(a, sb.charAt(b));
            sb.setCharAt(b, ch);
	}
	return sb.toString();
    }

    private void shuffle(int arr[]) 
    {
	for (int k = 0; k < arr.length - 1 ; k++) 
	{
	    int m = (int)(Math.random() * (arr.length -1));
	    int temp = arr[k];
	    arr[k] = arr[m];
	    arr[m] = temp;
	}
    }

    private void displayCrossData(GaIndividual parent, GaIndividual child) 
    {
	System.out.println("    " + "abcdefghijklmnopqrstuvwxyz");
	System.out.println("P : " + parent.displayCrossData());
	System.out.println("C : " + child.displayCrossData());
    }

    private void printStartMessage() 
    {
        System.out.print("GA_ANALYSIS:" 
			   + "\t nChars= " + cleantext.length()
                           + "\t nIndivs= " + size  
			   //                         +  "\nTEXT: " + cleantext + "\n"
			   );
	if (selection_policy == GaParameters.ELITIST_SELECTION) 
	    System.out.print("\tselection= ELITIST");
	else 
	    System.out.print("\tselection= PROPORTIONAL");
	if (mutate_policy == GaParameters.RANDOM_MUTATION) 
	    System.out.println("\tmutation= RANDOM");
	else 
	    System.out.println("\tmutation= ELITIST");
        
    }

    private boolean isBetter(GaIndividual child, GaIndividual parent) 
    {
	//	double  better = child.getFitness() - parent.getFitness();
	double  better = parent.getFitness() - child.getFitness();
	if (better > 0) 
	{
	    //	    System.out.println("Parent= " + parent.getFitness() + " Child= "  + child.getFitness());
	    improved++;
	}
	else if (better < 0)
	    worsened++;
	else
	    nochange++;
	return better > 0;
    }

    private void displayAll() 
    {
	for (int k = 0; k < size; k++)
	    System.out.println("---------------- " + k + " " + individual[k].displayCrossData() 
			       + " nWrong= " + countWrongs(correctPermKey, ((AlbertiGaIndividual)individual[k]).getPermKey())
			       + " CHI= " + ((AlbertiGaIndividual)individual[k]).calcCHI(NN)
			       );
    }

    private void mutateAll() 
    {
	mutated=0;
	for (int k = 0; k < individual.length; k++) 
	{ // Perform Mutations
	    mutated += individual[k].mutate(mutate_rate);
	}
    }

    public void displayBest() 
    {
	System.out.println(iterations + " " +  num.format(bestScore) + " "
			   + individual[0].toString()
			   + " (" + improved + "," + worsened + "," + nochange + ")"
			   + " mutated=" + mutated
			   + " median = " + num.format(individual[individual.length/4].getFitness())
			   + " worst = " + num.format(individual[individual.length/2].getFitness())
			   + " nWrong= " + countWrongs(correctPermKey, ((AlbertiGaIndividual)individual[0]).getPermKey())
			   //			   + " IC= " + ((AlbertiGaIndividual)individual[0]).calcIC(NN)
			   //			   + " CHI= " + ((AlbertiGaIndividual)individual[0]).calcCHI(NN)
			   );
    }

    public void displaySummary() 
    {
        System.out.println("Finished: Iterations= " + iterations + " Score= " + num.format(bestScore) + " KeyCount= " + keyCount
			   + " key= " + ((AlbertiGaIndividual)individual[0]).getKey()
			   //			   + " CHI= " + ((AlbertiGaIndividual)individual[0]).calcCHI(NN)
			   + " nWrong= " + countWrongs(correctPermKey, ((AlbertiGaIndividual)individual[0]).getPermKey())
			   + " Decrypt= " +  individual[0].getDecrypt().substring(0,30)
			   );
	//        System.out.println("Got all " + nWords + " words at iteration number = " + gotall);
	//        System.out.println("Crosses improved= " + improved + " worsened= " + worsened + " nochange= " + nochange);
	//        System.out.println(decrypt(true));
    }

    private void updateScore() 
    {
	bestScore = individual[0].getFitness();
	//	bestScore = individual[individual.length-1].getFitness();
	//	if (bestScore > previousBestScore) {
	if (bestScore < previousBestScore) 
	{
	    previousBestScore = bestScore;
	    lastScoreChange = iterations;
	}
    }
}
