 /*
 *  File: AnnealerAnalyzer.java
 *  Author: R. Morelli <ralph.morelli@trincoll.edu>
 * 
 *  Description: This class uses a simulated annealing algorithm
 *    to analyze traditional substitution ciphers. The algorithm
 *    is based on the algorithm described in A.J. Clark, "Optimisation
 *    Heuristics for Cryptology," Ph.D. Thesis, Information Security
 *    Research Group, Queensland University of Technology, 1998.  It
 *    also borrows certain ideas from J.W. Stumpel. See his website at
 *    http://www.jw-stumpel.nl/playfair.html
 *
 *  Description: This class uses an NgramArray with frequencies of N =
 *  2, 3, or 4 (bigrams, trigrams, or tetragrams) of a langauge to
 *  perform cryptanalysis. Using the sum of the inverses of the
 *  frequencies as an evaluation of how accurately a decrypt matches
 *  the usual N-gram frequencies in a language is an idea described by
 *  Alex Griffing the developer of the "Automatic Cryptogram Solver".
 *  This program is based on R. Walde's NgramClimber.java.
 *
 * <P>Copyright: This program is in the public domain. You can modify it as you
 *  see fit as long as you properly acknowledge its original author.
 *  It would also be nice if you forwarded your changes to
 *  <A HREF= "mailto:ralph.morelli@trincoll.edu">ralph.morelli@trincoll.edu</A> so
 *  they can possibly be added to the "official" version.
 *
 *  To compile: javac -classpath hcryptoclasses -d hcryptoclasses AnnealerAnalyzer.java
 *  To run main(): java -classpath hcryptoclasses AnnealerAnalyzer
 *
 *  To compile and run from the TestAnalyzer application:
 *
 *  cd ~crypto/hcryptoj/1.4/applications/testanalyzer
 *  javac -classpath ../../classes -d ../../classes ../../source/hcrypto/analyzer/NgramAnalyzer.java
 *  java -classpath ../../classes:. TestAnalyzer analyzers.NgramAnalyzer ga_paramfiles ngram.cgrams.txt
 *
 */

package analyzers;

import hcrypto.analyzer.*;
import hcrypto.cipher.*;
import hcrypto.provider.*;
import java.io.*;
import java.text.NumberFormat;
import java.lang.Math;

public class AnnealerAnalyzer extends CryptoAnalyzer implements Analyzer {

    private final int MAX_BLOCKSIZE = 8;
    private final int MAX_KEYS = 100000;
    public static final int SIMPLESUB = 0;
    public static final int PERMUTATION = 1;
    public static final int RAILFENCE = 2;
    public static final int PLAYFAIR = 3;
    public static final int MAX_ITERATIONS = 100;
    public static final int MAX_SUCCESS = 120;
    public static final int MAX_TRIES = 1000;
    public static final double TEMP_FACTOR = 0.98;

    private Alphabet alphabet; // Defines which chars occur in N-grams.
    private int randSize;      // The portion of the subst alphabet to permute
    private int alphSize;      // The size of the alphabet.
    private int cipherType;    // The type of cipher
    private int NN;            // The N of the N-gram = sequence of N chars.
    private NgramArray ngramArr; //inverse of frequency of each N-gram

    private String plainText = "";  // The current best plaintext found.
    private String book = null;       // Book used for NgramArray

    private int charErrs = 0;
    private double percentTokens = 0;
    private int iterations = 0;
    private int maxKeys = MAX_KEYS;
    private double temperature = 0;
    private double bestValue;            
    private double testValue;            
    private NgramDecrypter decrypter;  // Performs cipher-specific decryption
    private int hillCount = 0;
    private double bestGlobalValue = Double.MAX_VALUE;

    /**
     * AnnealerAnalyzer() -- Default constructor
     */
    public AnnealerAnalyzer() {
	super();
    } 

    /**
     * AnnealerAnalyzer() -- this constructor is given an object containing parameter settings
     * @param params -- an object containing param1=val1 param2=val2 ...
     */
    public AnnealerAnalyzer(GaParameters params) {
	super(params);
    }

    /**
     * setup(cText) - from the Analyzer interface.
     * Justs assigns the text to an instance variable.
     * @param text, a String storing the cryptogram
     */
     public void setup(String cText){
	 super.setup(cText); 
	 init();
	 num.setMaximumFractionDigits(2);
	 try {
	     setAnnealerAnalyzer(NN, book, alphabet, randSize, cipherType, maxKeys);
	 } catch (Exception e) {
	     e.printStackTrace();
	 }
     } // setup()

    /**
     * setup(cText) - This version is passed both the cryptogram and a book
     *  to be used in scoring.
     * @param text, a String storing the cryptogram
     * @param book, a String storing a book used for statistics
     */
     public void setup(String cText, String book){
	 this.book = book;
	 super.setup(cText);
	 init();
     } // setup()


    /**
     * init() - Sets various variables
     * NN = N is the size of the N-gram N=2, 3, or 4
     * fileName is a file containing a large text typical of the language
     *   of the cryptotext
     */
    public void init() {
	if (params == null) {
	    params = new GaParameters();
	}
	NN = params.NN;
	maxKeys = params.nkeys;
	randSize = params.randSize;
	cipherType = params.cipherType;
	if (book == null)
	    book = params.book;
	alphabet = params.alphabet;
    }

    /** Used to interface with main() 
     */
    public void setAnnealerAnalyzer(int N, String book, Alphabet alpha, int rSize, int cType, int mkeys) throws Exception { 
	this.NN = N;
	this.maxKeys = mkeys;
	this.randSize = rSize;
	this.cipherType = cType;
	this.book = book;
	this.alphabet = alpha;
	alphSize = alphabet.getSize();
	ngramArr = new NgramArray(NN, book, alphabet);

	if (cipherType == SIMPLESUB) {
	    decrypter = NgramDecrypterFactory.getInstance("Substitution", text, alphabet, ngramArr);
	    ((SubstitutionNgramDecrypter)decrypter).setRandSize(randSize);
	}
	else if (cipherType == PERMUTATION)
	    decrypter = NgramDecrypterFactory.getInstance("Permutation", text, alphabet, ngramArr);
	else if (cipherType == RAILFENCE)
	    decrypter = NgramDecrypterFactory.getInstance("Railfence", text, alphabet, ngramArr);
	else if (cipherType == PLAYFAIR)
	    decrypter = NgramDecrypterFactory.getInstance("Playfair", text, alphabet, ngramArr);
	else
	    decrypter = NgramDecrypterFactory.getInstance("Substitution", text, alphabet, ngramArr);
	resultSB.append("NGramAnalyzer: N= " + NN + " alphsize= " + alphSize + " cipherType= " + getCipherType() + " maxKeys= " + maxKeys + "\n");
    }  


    /**
     * From the Analyzer interface.
     */
    public String getReport() {
        return toString();
    }

    /**
     * From the CryptoAnalyzer interface.
     */
    public void run() {
	stopThread = false;           // stopThread declared in superclass
	runAnnealer(text, maxKeys);
        System.out.println("Finished: Iterations = " + iterations + " Best score is " + num.format(bestValue) 
			   + " KeyCount = " + keyCount + " HillCount = " + hillCount);
	resultSB.append("-------------\nDECRYPTED MESSAGE :\n" + plainText);
	resultSB.append("SOLUTION          : " + solution + "\n");
	resultSB.append("PERCENT WORDS: " + num.format(TextUtilities.percentWords(solution,plainText)) + "\n");
	resultSB.append("WRONG CHARS: " + TextUtilities.countInCorrectChars(solution,plainText) + "\n-------------\n");

	if (display != null) {
	    display.append("\nFinished");
	}
    }

    /**
     * Puts characters with frequencies into a string.
     */
    public String toString(){
	return resultSB.toString();
    } // toString()

    public void print(){
        System.out.println(toString());
    } // print()

   public int getAlphSize() {
        return alphSize;
   } //getAlphSize()

   public String getCryptoText(){
       if (text != null) return text;
       else return "";
   } // getCryptoText()

   public String getPlainText(){
       if (plainText != null) return plainText;
       else return "";
   } // getPlainText()

    public String getCipherType() {
	switch (cipherType) {
	case SIMPLESUB: return "SimpleSubstitution"; 
	case PERMUTATION: return "Permutation";
	case RAILFENCE: return "Railfence";
	case PLAYFAIR: return "Playfair";
	default: return "UNKNOWN CIPHER TYPE";
	}
    }

    /**
     * runAnnealer(cText, mkeys) attempts to cryptanalyze the
     * cryptoText cText by finding the best N-gram fit assuming a transposition cipher. 
     * mLoops is used as the max number of attempts at finding a better solution
     * without success before stopping.
     */

    public void runAnnealer(String cText, int mkeys) {
	boolean verbose = params.verbose;
	verbose = true;
	String plain = null;
	double deltakey = 0;
	hillCount = 0;
	keyCount = 0;
	iterations = 0;
        double tempFactor = params.tFactor; // TEMP_FACTOR;
	int succLimit = MAX_SUCCESS;
	int triesLimit = MAX_TRIES;
	int nTries = 0;
	System.out.println("Running Annealer: tempFactor= " + tempFactor + " MaxSUCCESS= " + succLimit + " MaxTRIES= " + triesLimit + " MaxKEYS= " + mkeys + "\n"); 
	double testValue;
	try {
	    decrypter.randomizeClimb();
	    bestValue = decrypter.currentEval();
	    temperature = bestValue/100;
	    //   //	    temperature = params.temperature;
	    ++keyCount;
	    int loopCount = 0;
	    int nSucc = 0;
	    do {
		nSucc = 0;
		nTries = 0;
		++hillCount;
		do { 
		    ++nTries;
		    int n1 = (int)(Math.random() * randSize);
		    int n2 = (int)(Math.random() * randSize);
		    decrypter.swapInKey(n1, n2);
		    testValue = decrypter.currentEval();
		    //		    System.out.println("Value = " + testValue + " n1=" + n1 + " n2=" + n2);
		    ++keyCount;
		    deltakey = testValue - bestValue;
		    double val1 = Math.random();
		    double val2 = Math.exp(-deltakey/temperature);
		    if (deltakey < 0)                         // Key improved 
			++nSucc;
		    if (deltakey < 0 || val1 < val2) {        // Key improved OR we keep a weaker key
			decrypter.saveState();
			bestValue = testValue;
			plainText = decrypter.getPlainText();
		    }
		    else {
			decrypter.swapInKey(n1, n2);    // Undo swap
		    }
		} while (nSucc < succLimit && nTries < triesLimit);
		// NOTE: WHEN THIS LOOP EXITS THERE'S NO GUARANTEE THAT bestValue WAS THE BEST VALUE
		//   ACHIEVED B/C WE OFTEN KEEP AN INFERIOR VALUE.

		if (verbose && bestValue < bestGlobalValue) {
		    bestGlobalValue = bestValue;
		    plainText = decrypter.getPlainText();
		    //		    System.out.println(iterations + "," + nTries + ',' + nSucc + " Value = " + bestValue + " KeyCount = " + keyCount + " T= " + temperature);
		    System.out.println("Value = " + bestValue + " KeyCount = " + keyCount + " T= " + temperature);
		    System.out.println("\t" + plainText);
		    if (display != null)
			display.setText(plainText);
		}
		temperature = temperature * tempFactor;
		++iterations;
		//	    } while (keyCount < maxKeys && nSucc != 0 && !threadIsStopped());
		// } while (keyCount < maxKeys && !threadIsStopped());  // IJAIT Version
	    } while (!threadIsStopped());  // CryptoToolJ Version
	} catch (Exception e) {
	    System.out.println(e.toString());
	    e.printStackTrace();
	}
    }

   public static void main(String[] args){
      try{
	  if (args.length != 4) {
	      System.out.println("Usage: java AnnealerAnalyzer TestNum N book mKeys");
	      return;
	  }
	  int testNum = Integer.parseInt(args[0]);
	  int NN = Integer.parseInt(args[1]);
	  String file = args[2];
	  int mKeys = Integer.parseInt(args[3]);

	  int randSize=26;
	  int cipherNum = 0;
	  String test;
	  char[] arr;
	  
	  switch (testNum) {
//        Test 0 - cryptotext 27 chars (a-z + space) with space preserved 
	  case 0:  // TEST 0: Substitution cipher gadsby, no letter 'e'
	      cipherNum = 0;
	      test = "PKJI OEXN AGNXN X GH TJXIT OJ NEJR \n"
	            + "VJP EJR G APIDE JY AMXTEO VJPIT YJFCN SXS YXIS \n" 
		  + "G DEGHKXJI; G HGI RXOE AJVN GIS TXMFN JY EXN JRI. \n";
	      arr = new char[4]; arr[0]='a'; arr[1]='z';arr[2]=arr[3]=' ';
	      randSize = 26;
	      break;
//        Test 1 - cryptotext 27 chars (a-z + space) with space preserved 
	  case 1:  // TEST 1: Substitution cipher with word breaks
	      cipherNum = 0;
	      test = "WKVHDG XWLNA BLDDLIS FI DWA XONN, \n"
	            + "UJFPA YFXI OIY DWAI UASOI DF UOXN, \n"
		    + "UAZOKBA BFVA DGPA, ONN BDOJJG AGAY, \n"
		    + "WOY HOLIDAY ZFNFJB FI WLB WLYA.";
	      arr = new char[4]; arr[0]='a'; arr[1]='z';arr[2]=arr[3]=' ';
	      randSize = 26;
	      break;
	//	Test 2 - cryptotext 27 chars (a-z + space) with space NOT preserved
	  case 2: // TEST 2: Substitution with SPACE not preserved
	      cipherNum = 0;
	      test = "WKV DGHXWLNAHBLDDLISHFIHDWAHXONNH\n"
             + "UJFPAHYFXIHOIYHDWAIHUASOIHDFHUOXNH\n"
             + "UAZOKBAHBFVAHDGPAHONNHBDOJJGHAGAYH\n"
             + "WOYH OLIDAYHZFNFJBHFIHWLBHWLYA";
	      arr = new char[4]; arr[0]='a'; arr[1]='z';arr[2]=arr[3]=' ';
	      randSize = 27;
	      break;
	//        Test 3 - cryptotext 26 chars (a-z) with word boundaries removed.
	  case 3: // TEST 3: Substitution without word breaks
	      cipherNum = 0;
	      test = "WKVHDGXWLNABLDDLISFIDWAXONN\n"
	    + "UJFPAYFXIOIYDWAIUASOIDFUOXN\n"
            + "UAZOKBABFVADGPAONNBDOJJGAGAY\n"
	    + "WOYHOLIDAYZFNFJBFIWLBWLYA";
	      arr = new char[2]; arr[0]='a'; arr[1]='z';
	      randSize = 26;
	      break;
//        Test 4 - permutation cipher -- gadsby 40312
	  case 4: // TEST 4: Permutation
	      cipherNum = 1;
	      test = "pntouibashiiassginomtshogwouyooabwhnhocu\n"
		  + "ribfhyotgnfogukdislfndidcamhainaopawinmhoybt\n"
		  + "ndasilsrgfishowccno";
	      arr = new char[2]; arr[0]='a'; arr[1]='z';
	      randSize = 26;
	      break;
//        Test 5 - permutation cipher -- tobe 40312
	  case 5: // TEST 5: Permutation 
	      cipherNum = 1;
	      test = "oeobtnttorbtheotstiaeueqhtonis";
	      arr = new char[2]; arr[0]='a'; arr[1]='z';
	      randSize = 26;
	      break;
//        Test 6 -- railfence cipher tobe 40312
	  case 6: // TEST 6: Railfence
	      cipherNum = 2;
	      test = "othtentroahsettsuoboeiqionbtet";
	      arr = new char[2]; arr[0]='a'; arr[1]='z';
	      randSize = 26;
	      break;
//        Test 7 -- playfair cipher thisisatestofplayfairwithjoytotheworld
//	      test =               "skbpbpcqrtqepvodvkbaoxcpfbewqeskoyredndd";
	  case 7: // TEST 7: Playfair with 6-letter keyword
	      cipherNum = 3;
	      test = "skbpbpcqrtqekpkrumcwgibmdasfledgcufbprrtqbkobpelodpcymezmrfhdd";
	      //              test = test + "skbpbpcqrtqekpkrumcwgibmdasfledgcufbprrtqbkobpelodpcymezmrfh";
	      //              test = test + "skbpbpcqrtqekpkrumcwgibmdasfledgcufbprrtqbkobpelodpcymezmrfhdd";
	      arr = new char[2]; arr[0]='a'; arr[1]='z'; 
	      randSize = 26;
	      break;
	  case 8: // TEST 8: Playfair with 12-letter keyword
	      cipherNum = 3;
	      test ="vuwdwdixscvmcqrdzcswebsgcoabqavuwdolddinmsdcswemzerknu";
	      arr = new char[2]; arr[0]='a'; arr[1]='z'; 
	      randSize = 26;
	      break;
	  case 9: // TEST 9: Playfair with 9-letter keyword
	      cipherNum = 3;
	      //	      test ="hkpypygzryklcyagfqipodpsqmsrahtihsqrmlhhleqosxrefc";
	      /*********************
	      test = "hkpy is clshg kl be a xoes lrkg olyygte";
	      test = test + "hkgz uyrs aoiagysm cpsqmr ushk a kyfvrsn";
	      test = test + "of hsqr llqqlsd we sle krw qlanshg unlqqmr";
	      test = test + "hke mltdhk of hke olyygte tis staqubtd to";
	      test = test + "cr wahh hke gygk ttit hke tdlsm stgaaxle";
	      test = test + "drls krt kpkwlefl b";
	      ****************/

	      test = "ocoyfolbvnpiasakopvygeskovmufguwmlnooedrncforsocvmtuuty" +
		  "erpfolbvnpiasakopvivkyeocnkoccaricvvltsocoytrfdvcvooueg" +
		  "kpvooyvkthzscvmbtwtrhpnklrcuegmslnvlzscansckopormzckizu" +
		  "slccvfdlvorthzscleguxmifolbimvivkiuayvuufvwvccbovovpfrh" +
		  "cacsfgeolckmocgeumohuebrlxrhemhpbmpltvoedrncforsgisthog" +
		  "ilcvaioamvzirrlniiwusgewsrhcaugimforskvzmgclbcgdrnkcvcp" +
		  "yuxlokfyfolbvcckdokuuhavococlciusycrgufhbevkroicsvpftuq" +
		  "umkigpecemgcgpggmoqusyefvgfhralauqolevkroeokmuqirxccbcv" +
		  "maodclanoynkbmvsmvcnvroedrncgeskysysluuxnkgegmzgrsonlcv" +
		  "agebglbimordprockinankvcnfolbceumnkptvktcgefhokpdulxsue" +
		  "opclanoynkvkbuoyodorsnxlckmglvcvgrmnopoyofocvkocvkvwofc" +
		  "lanyefvuavnrpncwmipordgloshimocnmlccvgrmnopoyhxaifoouep" +
		  "gchk";

	      /***********************
	      test = "hkpypyclshtzmclyxoesmrtdolyygte" +
		  "hkgzwiryaoiagysmbpikleushkptyfvrsn" +
		  "odqshlmlqqlsdxrslrqrvqlanshdzqmhkle" +
		  "tqmmltdhkechkloryyifltiyiqshkshtzo" +
		  "crushkhkyqpghhtihhqmtdlslitseaxlr" +
		  "crrykrkgrkxoldedd";
	      ****************/
	      arr = new char[2]; arr[0]='a'; arr[1]='z'; 
	      randSize = 26;
	      break;

//        Test 10 - cryptotext 27 chars (a-z + space) very short
	  case 10:  // TEST 10: to be or not to be...
	      cipherNum = 0;
	      test = "sgzs ptdrshnm vgdsgdq hs hr adssdq sn ad nq mns sn ad ";
	      arr = new char[4]; arr[0]='a'; arr[1]='z';arr[2]=arr[3]=' ';
	      randSize = 26;
	      break;

	  default: // TEST 1: Substitution cipher with word breaks
	      cipherNum = 0;
	      test = "WKVHDG XWLNA BLDDLIS FI DWA XONN, \n"
	            + "UJFPA YFXI OIY DWAI UASOI DF UOXN, \n"
		    + "UAZOKBA BFVA DGPA, ONN BDOJJG AGAY, \n"
		    + "WOY HOLIDAY ZFNFJB FI WLB WLYA.";
	      arr = new char[4]; arr[0]='a'; arr[1]='z';arr[2]=arr[3]=' ';
	      randSize = 26;
	      break;
	  }

        test = test.toLowerCase();
        System.out.println("Testing = " + testNum + "\n" + test + "\n");
        Alphabet alph = new Alphabet(arr);
        AnnealerAnalyzer annealer = new AnnealerAnalyzer();
	annealer.setup(test,file);
	annealer.setAnnealerAnalyzer(NN,file,alph,randSize,cipherNum,mKeys); 
        annealer.run();  
        System.out.println("\n" + annealer.getReport());
      }  //try
      catch(Exception exc){
        System.out.println("In AnnealerAnalyzer main() - " + exc.toString());
	exc.printStackTrace();
      } //catch
    }//main()

} // AnnealerAnalyzer class


