/*
 * File: CaesarAnalyzer.java
 * @author R. Morelli <ralph.morelli@trincoll.edu>
 * 
 * Description: This class assumes that the text it is passed
 *  is encrypted with a Caesar shift. It performs a brute-force
 *  Chi-square analysis to determine the likely shift.

 * <P>Copyright: This program is in the public domain. You can modify it as you 
 *  see fit as long as you properly acknowledge its original author.
 *  It would also be nice if you forwarded your changes to 
 *  <A HREF= "mailto:ralph.morelli@trincoll.edu">ralph.morelli@trincoll.edu</A> so 
 *  they can possibly be added to the "official" version.
 */

package hcrypto.analyzer;

import hcrypto.cipher.*;
import hcrypto.engines.*;
import hcrypto.provider.*;

//public class CaesarAnalyzer extends AnalyzerFrame implements Analyzer {
public class CaesarAnalyzer extends AnalyzerFrame implements ExpertAnalyzer {

    private String text;
    private StringBuffer resultSB;
    private FrequencyTable ft;
    
    private int shift;
    private TextStatistics stats = null;
    
    public CaesarAnalyzer() {} // Required for CryptoTool

    public CaesarAnalyzer(TextStatistics ts) throws NullPointerException {
        stats = ts;
        if (ts == null)
            throw new NullPointerException("CaesarAnalyzer: TextStatistics object is not instantiated");
    }

    public void setup(String text) { 
        stats = new TextStatistics(text, true);
        this.text = text.toLowerCase();
        resultSB = new StringBuffer();
        ft = stats.getFrequencyTable();
	display.setText("Caesar Analyzer: Begin Analysis\n");
	run();
	//        ft.setAlphabetRange(AlphabetFactory.ALPH_az);
    }

    public void run() {
        doAnalysis();
	display.append(toString());
    }
    
    public String getReport() {
        return toString();
    }

    public String getKeywordString() {
	return shift + "";
    }

    public String toString() {
        return resultSB.toString();    
    }

    public void doAnalysis(String text) {       // For ExpertAnalyzer interface
        stats = new TextStatistics(text, true);
        this.text = text.toLowerCase();
        resultSB = new StringBuffer();
        ft = stats.getFrequencyTable();
	display.setText("Caesar Analyzer: Begin Analysis\n");
	doAnalysis();
    }

   /**
     *  This method performs an analysis of the text assuming a Caesar
     *  (shift) cipher was used. The goal is to find the shift (the Caesar key).
     *  It performs a Chi-square analysis on every possible shift to find the
     *  most likely shift.
     */
    public void doAnalysis() { 
        shift = getOptimalShift();
        resultSB.append("\nIf this is Caesar cipher, a possible shift is " + shift + "\n");
        try {
           resultSB.append("Here's a decryption based on a shift of " + shift + "\n\n");
           Provider.addProvider(new DefaultProvider("Default"));
           Cipher c = Cipher.getInstance("Caesar");
           CaesarKey key = (CaesarKey) HistoricalKey.getInstance("Caesar", c.getProvider());
           key.init(shift + "/az");
           c.init(key);
           if (text.length() <= DECIPHER_LIMIT)
               resultSB.append(c.decrypt(text) + "\n");
           else
               resultSB.append(c.decrypt(text.substring(0,DECIPHER_LIMIT)) + "\n");
        } catch (Exception e) {
	    e.printStackTrace();
        }
    }
    
    
    /** 
     *  This method performs a Chi-Square test to find the optimal shift.
     *  This monoalphabetic version calls the more general polyalphabetic
     *  version, which computes a chi-square test on a text beginning with
     *  the first (0th) character and proceeding through each adjacent 
     *  character.
     */
    public int getOptimalShift() {
        return getOptimalShift(0, 1);
    }

    /** 
     * This method performs a Chi-Square test to find the optimal Caesar shift
     *  on a polyalphabetic text. It assumes that every _displ_ character
     *  starting at _firstChar_ belongs to the same alphabet. It is used primarily
     *  for analyzing Vigenere-like and Caesar-shift cryptograms.
     * @param firstChar - the location of the first character in the cryptotext
     * @param displ - the displacement (keyword length) or cycle length
     */
    public int getOptimalShift (int firstChar, int displ) {
        String text = this.text;
        double tChiSqrs[] = new double[26];
        int tFreqs[] = new int[26];
        for (int k = 0; k < 26; k++) 
            tFreqs[k] = 0;              // initialize frequency array
        for (int k = 0; k < 26; k++) 
            tChiSqrs[k] = 0.0;		// initialize Chi Squares
	
	// Compute the frequencies for the characters in the text
        // For every displ char starting at firstChar ...

        for (int k = firstChar; k < text.length(); k+= displ ) {
            char ch = text.charAt(k);
            if (Character.isLetter(ch)) {
                ch = Character.toLowerCase(ch);
                ++tFreqs[ch - 'a'];
            }
        }
//        System.out.println("Finished counting");

	// Compute the Chi-square values for each possible shift

        for (int j = 0; j < 26; j++)       // for each possible shift j = 0..25
            for (char k = 'a'; k <= 'z'; k++) {  // for each character
                int index = ((k - 'a') + j) % 26;
                double freq = (TextStatistics.getEnglishFrequency(k) - tFreqs[index]);
                freq = freq * freq;
                freq = freq / TextStatistics.getEnglishFrequency(k);
                tChiSqrs[j] = tChiSqrs[j] +  freq;
            }
		
	// Return the minimum Chi-square value
	
        return TextStatistics.getIndexOfMinimum(tChiSqrs);
    }




}
