/*
 * File: CryptoAnalyzer.java
 * @author R. Morelli <ralph.morelli@trincoll.edu> 
 * @author R. Walde <rwalde@nyc.rr.com> 
 * 
 * Description: This class serves as the foundation for
 *  all cryptologic analyzers, including GA analyzers.  It should be 
 *  subclassed with a class containing extensions of its run() and setup()
 *  methods.
 */

package analyzers;

import java.text.*;
import hcrypto.analyzer.*;
import hcrypto.cipher.*;
import hcrypto.engines.*;
import hcrypto.provider.*;

public class CryptoAnalyzer implements Analyzer, Runnable {
    protected GaParameters params;       // Stores the parameters

    protected String text;               // Text to be cryptanalyzed
    protected String solution = "";      // Solution, if given
    protected StringBuffer resultSB;     // Stores results
    protected NumberFormat num;
    protected int keyCount = 0;
    protected boolean stopThread = false;  // Stops the analyzer's thread
    protected AnalyzerFrame display;

    /**
     * CryptoAnalyzer() -- Default constructor
     */
    public CryptoAnalyzer() { 
    }

    /**
     * CryptoAnalyzer() -- this constructor is given an object containing parameter settings
     * @param params -- an object containing param1=val1 param2=val2 ...
     */
    public CryptoAnalyzer(GaParameters params) {
	this.params = params;
    }

    /**
     * setup() -- this version of setup was added to make the
     *  analyzer compatible with CryptoToolJ. It assigns the
     *  analyzer a window to display its results.
     */
    public void setup(String text, AnalyzerFrame f) {
	display = f;
	setup(text);
    }

    /**
     * setup() initializes the text and the result string. Note that for files containing
     *  test cryptograms, the solution is sometimes also provided following the $$$ marker.
     */ 
    public void setup(String text) { 
	num = NumberFormat.getInstance();
	num.setMaximumFractionDigits(2);
	if (text.indexOf("$$$") != -1) {
	    this.solution = text.substring(text.indexOf("$$$") + 3);
	    this.text = text.substring(0, text.indexOf("$$$"));
	}
	else {
	    this.text = text;
	    this.solution = new String();
	}

	//	solution = TextUtilities.markWords(solution);
	//	System.out.println("CryptogramAnalyzer Text = " + this.text + "\nSolution = " + this.solution);

        if (this.text.length() > params.text_limit) {
            this.text = this.text.substring(0, params.text_limit) + " ";
            this.text = this.text.substring(0, this.text.lastIndexOf(' ')).toLowerCase();
	}
        else
            this.text = this.text.toLowerCase();
	//        System.out.println("CRYPTOGRAM Length= " + this.text.length());
	//        reportMissingLetters(this.text);
        resultSB = new StringBuffer();
    }

    protected void reportMissingLetters(String s) {
        System.out.print("The following letters are missing from the text: " );
        for (int k = 0; k < 26; k++) {
            char ch = (char)('a' + k);
	    if (s.indexOf(ch) == -1)
		System.out.print(ch + " ");
	}
	System.out.println();
    }


    /**
     * Part of Analyzer and Runnable interface. Override this method in the subclass.
     */
    public void run() { }
    
    /**
     * Stops the analyzer. Part of Analyzer interface. Override this method in the subclass.
     */
    public void stopThread() { 
	stopThread = true;
    } 

    public boolean threadIsStopped() {
	return stopThread == true;
    }


    /**
     * Part of Analyzer interface.
     */
    public String getReport() {
        return toString();
    }

    /**
     * Every class should have a toString() method.
     */
    public String toString() {
        return resultSB.toString();    
    }
}

