/*
 * File: FreqGaPopulation.java
 * @author R. Morelli <ralph.morelli@trincoll.edu>
 *
 * Description: This class implements an instance of a GaPopulation
 *  by providing implementations of the init()
 *  and run() methods for a FreqGaAnalyzer.
 */

package hcrypto.analyzer;

import java.util.*;
import hcrypto.cipher.*;

public class FreqGaPopulation extends GaPopulation {
    public static final int NO_CHANGE_LIMIT = 30;
    public static final int N_SHUFFLES = 10;

    private String text;               // Raw cipher text
    private String preparedText;       // Text with ' ' replaced by '{'

    //    private TextStatistics stats = null; // Stats obj for this text
    private FrequencyTable ft;

    private FrequencyRecord eng_freqs[];    // Character frequencies
    private FrequencyRecord crypto_freqs[];  

    private double english_digrams[][];
    private double crypto_digrams[][];
    private int map_crypt[];              // Index into digram table
    private int map_engl[];               // English freqs in descending order
    private int d_key[];                  // The d_key maps a-->e(a), b-->e(b), etc.


    /**
     * FreqGaPopulation() constructor
     * @param text -- the text to be cryptanalyzed
     * @param params -- a collection of parameters to control the run
     */
    public FreqGaPopulation (String text, String solution, GaParameters params) {
	super(text, solution, params);
    }

    /**
     * This method initializes the analyzer. It sorts the tokens in the
     *  cryptogram to give preference to those with the smallest (non empty)
     *  pattern sets. It then creates the population, giving each individual
     *  an encryption key based on a pairing of a token from the cryptogram
     *  and a word from the pattern word dictionary.
     */
    public void init() {
	//        nTokens = TextUtilities.countTokens(cleantext);
	//        nWords = TextUtilities.countWords(cleantext);

	text = ciphertext;
	preparedText = text.toLowerCase();

	ft = new FrequencyTable(preparedText, AlphabetFactory.ALPH_cryptogram);
	preparedText = cleanText(preparedText);

	crypto_freqs = ft.getSortedCryptogramFrequencies(); 
	crypto_freqs[26].ch = '{';
	//	freqPrint("CRYPTO: ", crypto_freqs);

	// The map_crypt contains the crypto chars in order of decreasing frequency
	map_crypt = makeMap(crypto_freqs);
	//	alphaPrint("MAP_CRYPT: ", map_crypt);

	eng_freqs = makeEnglishDigramTable(ft);
	//	freqPrint("ENGLISH: ", eng_freqs);
	// The map_crypt contains the crypto chars in order of decreasing frequency
	map_engl = makeMap(eng_freqs);
	//	alphaPrint("MAP_ENGL: ", map_engl);

	d_key = initKey(map_crypt, map_engl);
	initDigramTables(eng_freqs, crypto_freqs);

        individual = new FreqGaIndividual[size];
	String key = "";
        for (int k = 0; k < individual.length; k++) {
	    d_key = initKey(map_crypt, map_engl);
	    shuffle(d_key);
         individual[k] = new
         FreqGaIndividual(d_key,english_digrams,crypto_digrams,crypto_freqs);
         /* System.out.println("individual[" + k + "]: "
+individual[k].getKey()                          + " ("
+individual[k].getFitness() +")");   */       	                    
         }        
       }

    /**
     * This method creates an encryption key that is then used to
     *  seed the individual members of the population. The word pairs
     *  are created during initialization and stored in pairs[]. 
     *  Pairs of crypto,plain are selected at random.
     */
    private String makeKey() {
          return shuffle("abcdefghijklmnopqrstuvwxyz");
    }

    /**
     * This method implements the abstract run() method inherited from
     *  GaPopulation.
     */
    public void run() {
	printStartMessage();
        
        do {
	    if (tweakingOn && iterations - lastScoreChange > NO_CHANGE_LIMIT) {
		tweak();
	    }
      
	  for (int k = 0; k < size/2; k++) {      // Perform crosses on k andj 		                                   
           int j = size/2 +(int)(Math.random()* size/2); 		                 
           double rand =Math.random();                 
           if (rand <= cross_rate) {
		    GaIndividual i1 = new FreqGaIndividual(individual[k]);
		    GaIndividual i2 = new FreqGaIndividual(individual[j]);
		    i1.cross(i2);

		    //		    displayCrossData(individual[k], i1);
		    //		    displayCrossData(individual[j], i2);
		    if (isBetter(i1, individual[k]))       // Select fitter
			individual[k] = i1;
		    if (isBetter(i2, individual[j]))
			individual[j] = i2;
		}
	    } 
	    mutateAll();                          // Mutate all
            ++iterations;
            java.util.Arrays.sort(individual);    // Sort the population
	    //	    displayAll();
	    updateScore();
	    if (verbose) 
	       displayBest();
	    //	    if (nWords > 0 && individual[0].getFitness() >= nWords && gotall==0)
	    //		gotall = iterations;
		
        } while (iterations < maxtrials);
	//        } while (iterations < maxtrials && individual[2].getFitness() < nTokens);
	//        } while (iterations < maxtrials && individual[2].getFitness() < nWords);
	displaySummary();
    }

    public GaIndividual getFittest(int n) {
	//        return individual[n];
        return individual[individual.length-1];
    }

    /**
     * shuffle() randomly shuffles the characters in a StringBuffer.
     */
    private void shuffle(StringBuffer sb) {
	sb = new StringBuffer(shuffle(sb.toString()));
    }

    /**
     * shuffle() randomly shuffles the characters in a String.
     */
    private String shuffle(String s) {
	StringBuffer sb = new StringBuffer(s);
	for (int k = 0; k < N_SHUFFLES; k++) {
            int a = (int)(Math.random() * sb.length());
            int b = (int)(Math.random() * sb.length());
            char ch = sb.charAt(a);
            sb.setCharAt(a, sb.charAt(b));
            sb.setCharAt(b, ch);
	}
	return sb.toString();
    }

    private void shuffle(int arr[]) {
	for (int k = 0; k < arr.length - 1 ; k++) {
	    int m = (int)(Math.random() * (arr.length -1));
	    int temp = arr[k];
	    arr[k] = arr[m];
	    arr[m] = temp;
	}
    }


     private void tweak() {
	 //	 if (verbose)
	    System.out.println("Giving it some new blood!!!!!!!!!!!!!!!!");
	for (int k = 0; k < individual.length; k += 2)         // Tweak every other one
	    if (Math.random() > 0.25) {
		d_key = initKey(map_crypt,map_engl);
		shuffle(d_key);
		individual[k] = new FreqGaIndividual(d_key,english_digrams, crypto_digrams,crypto_freqs); 	 
             //	 individual[k] = new FreqGaIndividual(cleantext,makeKey()); 	 
            individual[k].calcFitness(); 		
       //System.out.println(" NEW BLOOD " + k + " " +individual[k].displayCrossData()); 	    
	    } 	
        lastScoreChange = iterations; 	
        previousHighScore = 0;
    }
    
    private void displayCrossData(GaIndividual parent, GaIndividual child) {
	System.out.println("    " + "abcdefghijklmnopqrstuvwxyz");
	System.out.println("P : " + parent.displayCrossData());
	System.out.println("C : " + child.displayCrossData());
    }

    private void printStartMessage() {
        System.out.println("\nSTARTING ANALYSIS" +
			   //                           "\t nTokens " + nTokens + 
			   //                           "\t nWords " + nWords +
                           "\t nChars " + cleantext.length() + 
                           "\t nIndivs " + size + 
                           "\nTEXT: " + cleantext + "\n");
    }

    private boolean isBetter(GaIndividual child, GaIndividual parent) {
	double  better = child.getFitness() - parent.getFitness();
	if (better > 0)
	    improved++;
	else if (better < 0)
	    worsened++;
	else
	    nochange++;
	return better > 0;
    }

    private void displayAll() {
	for (int k = 0; k < individual.length; k++)
	    System.out.println("---------------- " + k + " " + individual[k].displayCrossData());
    }

    private void mutateAll() {
	mutated=0;
	for (int k = 0; k < individual.length; k++) { // Perform Mutations
	    mutated += individual[k].mutate(mutate_rate);
	}
    }

    public void displayBest() {
	System.out.println(iterations + " " +  highScore + " "
			   //			   + individual[0].toString()
			   + individual[individual.length-1].toString()
			   + " (" + improved + "," + worsened + "," + nochange + ")"
			   + " mutated=" + mutated);
    }

    public void displaySummary() {
	//        System.out.println("Finished: Iterations = " + iterations + " Max score is " + nWords);
	//        System.out.println("Got all " + nWords + " words at iteration number = " + gotall);
        System.out.println("Crosses improved= " + improved + " worsened= " + worsened + " nochange= " + nochange);
        System.out.println(decrypt(true));
    }

    private void updateScore() {
	//	highScore = individual[0].getFitness();
	highScore = individual[individual.length-1].getFitness();
	if (highScore > previousHighScore) {
	    previousHighScore = highScore;
	    lastScoreChange = iterations;
	}
    }

    /* *************** Digram Specific UTILITY FUNCTIONS ****************************/

    /**
     * This method replaces  BLANK with '{' which is the character 
     * following 'z' in ASCII. This allows arithmetic to work easier.
     */
    private String cleanText(String text) {
	StringBuffer sb = new StringBuffer();
	for (int k = 0; k < text.length(); k++) {  // HACK: Replace blank with {
	    char ch = text.charAt(k);
	    if (ch == ' ')
		sb.append('{');
	    else if (ch >= 'a' && ch <= 'z')
		sb.append(ch);
	    else 
		sb.append('{');
	}
	return sb.toString();
    }

    /**
     * This method initializes key in a trivial way.
     * @param ft the FrequencyTable for this message
     */
    private int[] initKey(int map_c[], int map_e[]) {
	d_key = new int[27];
	for (int k = 0; k < 26; k++) {
	    d_key[k] = k;
//	    	    d_key[k] = map_c[k];
	}
	d_key[26] = '{' - 'a';
	//	alphaPrint("KEY: ", d_key);
	return d_key;
    }            

    /**
     * This method initializes both the English- and Crypto- digram tables.
     * Data for the english table are taken from TextStatistics. 
     */
    private void initDigramTables(FrequencyRecord eng_freqs[], FrequencyRecord crypto_freqs[]) {
	int total_chars = TextStatistics.digram_chars;
	int digram_data[][] = TextStatistics.digram_data;

	crypto_digrams = new double[27][27];
	english_digrams = new double[27][27];

	for (int j = 0; j < 27; j++)
	    for (int k = 0; k < 27; k++) {
		english_digrams[j][k] = 1.0 * digram_data[j][k]/total_chars;
		crypto_digrams[j][k] = 0;
	    }

	// Initialize crypto digrams
	String decrypt = decrypt(false);
	double incr = 1.0/(preparedText.length()-1); 

	for (int k = 1; k < decrypt.length(); k++) {
	    char ch1 = decrypt.charAt(k-1);
	    char ch2 = decrypt.charAt(k);
	    crypto_digrams[ch1 - 'a'][ch2 - 'a'] += incr;
	}
    }

    /**
     * This methods creates an array of chars (represented as 0..26) 
     *  arranged in order of decreasing frequency from an array 
     *  of frequencies in ascending order.
     */
    private int[] makeMap(FrequencyRecord freqs[]) {
	int map[] = new int[27];
	int j = 0;
	for(int k = freqs.length-1; k >= 0; k--)
	    map[j++] = freqs[k].ch - 'a';
	//	map[0] = '{' - 'a'; // Space
	map[26] = '{' - 'a'; // Space
	return map;
    }

    private FrequencyRecord[] makeEnglishDigramTable(FrequencyTable ft) {
	FrequencyRecord freqs[] = new FrequencyRecord[27];

	int count = ft.getCharCount();
	freqs[0] = new FrequencyRecord('{', (int)Math.round(0.25 * count));
	for (char ch = 'a'; ch <= 'z'; ch++) {
	    freqs[ch - 'a' + 1] = 
		new FrequencyRecord(ch, (int)Math.round(TextStatistics.englishFrequency[ch] * count));
	}
	java.util.Arrays.sort(freqs);
	return freqs;
    }

    public String decrypt(boolean replace_brace) {
	StringBuffer sb = new StringBuffer();
	for (int k = 0; k < preparedText.length(); k++) {
	    char ch = preparedText.charAt(k);
	    if (ch == '{')
		if (replace_brace)
		    sb.append(' ');
		else
		    sb.append(ch);
	    else {
		sb.append((char)('a' + d_key[ch - 'a']));
	    }
	}
	return sb.toString();
    }

}
