package hcrypto.analyzer;

import hcrypto.cipher.*;

public class FrequencyTable implements Analyzer {

    public static final int MAX_CHARS = 128;          // Possibly all ASCII characters
    private int alphabetRange = AlphabetFactory.ALPH_ascii; // Default: ASCII
    private String text;
    Alphabet alphabet = null;
    private int charCount = 0;                        // Various counters
    private int alphabeticsCount = 0;   
    private int highestFreq;                          // Highest f for bar chart

    private FrequencyRecord frequencies[] = new FrequencyRecord[MAX_CHARS];    
    private FrequencyRecord sortedFrequencies[] = new FrequencyRecord[frequencies.length];    
    
    private int histogram[] = new int[MAX_CHARS];

    public FrequencyTable(String text) {
       setup(text);
    }

    public FrequencyTable(String text, int alpha){
        alphabetRange = alpha;
        setup(text);
    }

    /**
     * From the Analyzer interface.
     */
    public void setup(String text) {
        this.text = text;
        count();
        sort();
    }
    /**
     * Everything is done in setup.
     */
    public void run() {
    }
    public String getReport() {
        return toString();
    }

    /**
     * Sets alphabetRange to an ASCII subset.
     */
    public void setAlphabetRange(int range) {
        if (alphabetRange == range) 
            return;
        alphabetRange = range;
        setup(text);
    }
    /**
     * Arranges the frequencies as a 2-D table.
     */
    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append("\n\n");
        for (int k = 0; k < frequencies.length; k++) {
            sb.append(charToString(frequencies[k].ch) + ":" + frequencies[k].count + " ");
            if ((k + 1) % 10 == 0)
                sb.append("\n");
        }
        sb.append("\n");
        return sb.toString();
    }

    public void print() {
        System.out.println(toString());
    }
    
    /**
     * Counts the characters in the text.
     */
    public void count() {
        alphabet = null;
        int size = MAX_CHARS;
        frequencies = new FrequencyRecord[size];    
        for (int k = 0; k < frequencies.length; k++)
            frequencies[k] = new FrequencyRecord((char)k, 0);
        
        try {  
            alphabet = AlphabetFactory.getInstance(alphabetRange);
        } catch (Exception e) {
            e.printStackTrace();
        }
        
        charCount = 0;
        alphabeticsCount = 0;
        highestFreq = 0;
        
        for (int k = 0; k < text.length(); k++) {
            char ch = text.charAt(k);
            if (Character.isLetter(ch))
                 ++alphabeticsCount;
            if (alphabet.isInAlphabet(ch)) { // < MAX_CHARS) {       // Ignore chars > 127
                charCount++;
                frequencies[ch].count++;
                if (frequencies[ch].count > highestFreq) 
                    highestFreq = frequencies[ch].count; // Keep track of highest f
            }
        }  
    }
    
    /**
     * returns the highest frequency character
     */
    public int getHighestFreq() {
        return highestFreq;
    }
    
    /**
     * returns the the kth highest frequency character
     */
    public String getKHighestFreqChar (int k) {
        return charToString(sortedFrequencies[k-1].ch);
    }

    public FrequencyRecord[] getCryptogramFrequencies() {
	FrequencyRecord freqs[] = new FrequencyRecord[27];  // a-z + SPACE
	freqs[0] = frequencies[32];
	for (int k = 1; k <= 26; k++)
	    freqs[k] = frequencies['a' + k - 1];
	return freqs;
    }

    public FrequencyRecord[] getSortedCryptogramFrequencies() {
	FrequencyRecord freqs[] = getCryptogramFrequencies();
	java.util.Arrays.sort(freqs);
	return freqs;
    }


    /**
     * returns the frequencies as an array
     */
    public int[] getFrequencies() {
        return getHistogramData();
    }

    /**
     * Constructs an array of int out of the frequency tables
     */
    public int[] getHistogramData() {
        for (int k = 0; k < histogram.length; k++) 
            histogram[k] = frequencies[k].count;
        return histogram;    
    }
    
    /**
     * Converts a nonprintable character to a string
     */
    protected String charToString(char ch) {
        if (Character.isLetterOrDigit(ch))
            return ch + "";
        else
            return "\\" + (int)ch;

    }

    /**
     * Returns a count of all characters.
     */
    public int getCharCount() {
        return charCount;
    }
    
    /**
     * Returns a count of just alphabetic characters
     */
    public int getAlphabeticsCount() {
        return alphabeticsCount;
    }

    /**
     * Returns the count  for a specific character
     */
    public int getCount(int ch) {
        return frequencies[ch].count;
    }

    /**
     * Uses the java.util.Arrays sort method to sort the 
     *  frquencies and stores the results in sortedFrequencies.
     */
    protected void sort() {
        sortedFrequencies = new FrequencyRecord[frequencies.length];  
        for (int k = 0; k < frequencies.length; k++)
            sortedFrequencies[k] = new FrequencyRecord(frequencies[k]);
        java.util.Arrays.sort(sortedFrequencies);
    }


}
