/*
 * File: GaPopulation.java
 * @author R. Morelli <ralph.morelli@trincoll.edu>
 *
 * Description: This abstract class should be subclassed
 *  with a class containing implementations of the init()
 *  and run() methods for a particular type of GaPopulation.
 */

package hcrypto.analyzer;

import java.util.*;

public abstract class GaPopulation {
    protected int size = GaParameters.POPULATION_SIZE;            // Should be a multiple power of 2
    protected double mutate_rate = GaParameters.MUTATE_RATE;
    protected double cross_rate = GaParameters.CROSS_RATE;
    protected int maxtrials = GaParameters.MAX_GENERATIONS;
    protected boolean verbose = GaParameters.IS_VERBOSE;
    protected int seeding = GaParameters.RANDOM_SEEDING;
    protected boolean tweakingOn = GaParameters.TWEAK_ON;      
    protected int tweak_at = GaParameters.TWEAK_AT;         // Tweak threshhold; OFF if <= 0
    protected int mutate_policy = GaParameters.RANDOM_MUTATION;
    protected int selection_policy = GaParameters.ELITIST_SELECTION;

    protected String ciphertext;       // The secret message
    protected String cleantext;        // Text with punctuation removed
    protected String solution = "";    // IN case a solution is provided
    protected AnalyzerFrame display = null; // For use with CryptoToolJ
 
    protected GaParameters params;
    protected GaIndividual individual[]; // The individuals that make up the population

    /**
     * There variables are used to monitor the GA.
     */
    protected int nochange=0, improved=0, worsened=0, gotall=0, mutated=0;
    protected int lastScoreChange = 0;
    protected double bestScore = 0.0, previousBestScore = 0.0;
    protected int iterations = 0;

    /**
     * GaPopulation() constructor
     * @param text -- the text to be cryptanalyzed
     * @param params -- a GaParameters object containing all needed parameters
     */
    public GaPopulation (String text, String solution, GaParameters params) {
	this.solution = solution;
	if (params != null) {
	    this.params = params;
	    this.size = params.size;
	    this.mutate_rate = params.mutate_rate;
	    this.mutate_policy = params.mutate_policy;
	    this.selection_policy = params.selection_policy;
	    this.cross_rate = params.cross_rate;
	    this.verbose = params.verbose;
	    this.tweakingOn = params.tweaking;
	    this.maxtrials = params.generations;
	    this.seeding = params.seeding;
	    this.tweak_at = params.tweak_at;
	} 
	tweakingOn = tweak_at > 0;          // tweak_at overrides tweakingOn

        ciphertext = text;
	//        System.out.println("TEXT=" + ciphertext);
        cleantext = TextUtilities.cleanString(ciphertext);   
	//        cleantext = TextUtilities.removeDuplicates(cleantext);
	init();
    }

    public GaIndividual getFittest(int n) {
        return individual[n];
    }

    /**
     * This method initializes the analyzer for the particular
     *  type of population. It should be implemented in the subclass.
     */
    public abstract void init(); 


    /**
     * The generic GA run method. 
     */
    public abstract void run();
    
    public void setDisplay(AnalyzerFrame d) {
	display=d;
    }

    /**
     * Individuals are kept proportionally to their fitness score.
     * The top individual is kept with 95% probability. For lesser
     * individuals their chance of being selected is less.
     */
    protected void selectProportional() {
	int ptr = 0, k = 0;
        double max = individual[0].getFitness();
	max *= 1.05;
	while (ptr < size && size - ptr < 2*size - k) {
	    double val = individual[k].getFitness();
	    if (Math.random() <= val/max)
		individual[ptr++] = individual[k++];
	    else
		k++;
	}
	//	if (size-ptr >= 2*size-k) 
	for ( ; ptr < size; ptr++) {
	    individual[ptr] = individual[k];
	    ++k;
	}
    }


}
