/*
 * File: Histogram.java
 * @author R. Morelli <ralph.morelli@trincoll.edu>
 * 
 * Description: An interactive canvas which displays a histogram of frequency data. 
 *  The histogram is added to a StatsFrame, which also incorporates numeric data
 *  about the text being analyzed.
 *
 *  <P>Copyright: This program is in the public domain. You can modify it as you 
 *  see fit as long as you properly acknowledge its original author.
 *  It would also be nice if you forwarded your changes to 
 *  <A HREF= "mailto:ralph.morelli@trincoll.edu">ralph.morelli@trincoll.edu</A> so 
 *  they can possibly be added to the "official" version.
 */
 
package hcrypto.analyzer;
 
import java.awt.*;
import java.awt.event.*;


public class Histogram extends Canvas implements //Analyzer
                                      MouseMotionListener {

    public static final int WIDTH = 350;
    public static final int HEIGHT = 200;
    public static final int XMARGIN = 10;
    public static final int YMARGIN = 12;//= 20;
    public static final int FIRST_CHAR = 2;
	
    private int histogram[];
    private FrequencyTable freqTable;

    public Histogram(FrequencyTable ft) {
//	System.out.println("Histogram Tool");
        freqTable = ft;
        histogram = ft.getHistogramData();
        setSize(WIDTH, HEIGHT);
        addMouseMotionListener(this);
    }

    public void paint(Graphics g) {
        draw(g);
    }
    
    public void redraw() {
        repaint();
    }

    private void draw(Graphics g) {
    	int x, y;
    	double maxFreq = freqTable.getHighestFreq();
        histogram = freqTable.getHistogramData();
    	int maxbar = HEIGHT - 2 * YMARGIN -2;
    	g.setColor(Color.white);
    	g.fillRect(0, 0, WIDTH, HEIGHT);
		    	
	g.setColor(Color.black);
			        
        g.drawRect(XMARGIN, YMARGIN, WIDTH-2*XMARGIN, HEIGHT-2*XMARGIN); // Bounding rectangle
        x = XMARGIN + FIRST_CHAR;
        y = HEIGHT-YMARGIN;
        int count;
        for (int i = 0; i < histogram.length; i++) {
	    count = histogram[i];
//			System.out.println(count);
	    g.drawLine(x + 2 * i, y, x + 2 * i, y - (int)(maxbar * count / maxFreq));
        }
    }

    public void mouseDragged(MouseEvent e){ }
    public void mouseMoved(MouseEvent e) { mouseMoved(e.getX(), e.getY()); }
    public void mouseMoved(int x, int y) {
        java.text.NumberFormat nf = java.text.NumberFormat.getNumberInstance();
        nf.setMaximumFractionDigits(3); 
        
        Graphics g = getGraphics();
        g.setColor(Color.white);
        g.fillRect(0,0,WIDTH,YMARGIN-1);
        g.setColor(Color.red);
        char ch = (char)((x - XMARGIN - FIRST_CHAR)/2);
        if (ch >= 0 && ch < 128) {
            setCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
            g.drawString("Char: " + charToString(ch) + " Count: " + freqTable.getCount(ch) +
			 " Freq: " +  
			 nf.format(1.0 * freqTable.getCount(ch)/freqTable.getCharCount()), XMARGIN, YMARGIN-3);
        }
        else
            g.drawString("Char: " + " " + " Count: " + " " + " Freq: " + " ", XMARGIN, YMARGIN-3);
	}

    protected String charToString(char ch) {
        if (Character.isLetterOrDigit(ch))
            return ch + "";
        else
            return "\\" + (int)ch;
    }


}
