/*
 * File: NgramGaAnalyzer.java
 * @author R. Morelli <ralph.morelli@trincoll.edu> 
 * 
 * Description: This class assumes that the text is
 *  encrypted with simple substitution and uses a
 *  frequency-based genetic algorithm to analyze it.
 * 
 *  Its evaluation function uses an NgramArray with frequencies of
 *  N = 2, 3, or 4  (bigrams, trigrams, or  tetragrams) 
 *  to cryptanalyze transposition and substitution ciphers.
 *  Using the sum of the inverses of the frequencies as an evaluation of how
 *  accurately a decrypt matches the usual N-gram frequencies in a
 *  language is an idea described by Alex Griffing the developer of the
 *  "Automatic Cryptogram Solver".  This program is based on R. Walde's
 *  NgramClimber.java.
 *
 *  To compile and run from the TestAnalyzer application:
 *
 *  cd ~crypto/hcryptoj/1.4/applications/testanalyzer
 *  javac -classpath ../../classes -d ../../classes ../../source/hcrypto/analyzer/NgramGaAnalyzer.java
 *  java -classpath ../../classes:. TestAnalyzer analyzers.NgramAnalyzer ga_paramfiles ngram.ga.cgrams.txt
 *
 */

package analyzers;

import hcrypto.analyzer.*;
import hcrypto.cipher.*;
import hcrypto.engines.*;
import hcrypto.provider.*;

public class NgramGaAnalyzer extends CryptoAnalyzer {
    
    protected GaPopulation population;     

    /**
     * NgramGaAnalyzer() -- Default constructor
     */
    public NgramGaAnalyzer() {
	super();
    }

    /**
     * NgramGaAnalyzer() -- this constructor is given an object containing parameter settings
     * @param params -- an object containing param1=val1 param2=val2 ...
     */
    public NgramGaAnalyzer(GaParameters params) {
	super(params);
    }

    /**
     * setup() creates the population for the GA.
     */
    public void setup(String text) {
	super.setup(text);
	if (params == null) {
	    params = new GaParameters();       // Default GaParameters
	}
	population = new NgramGaPopulation(this.text, this.solution, params);
    }


    /**
     * run() conducts the GA run and reports the results.
     */
    public void run() {
        String decrypt;
	if (display != null) {                          // For use with CryptoToolJ
	    population.setDisplay(display);
	}
	population.run();
	int n = 1;
	resultSB.append("HERE'S THE TOP " + n + "  RESULTS:\n");
	for (int k = 0; k < n; k++) {

	    String key = population.getFittest(k).getKey();
	    resultSB.append("\nKey (" + k + ") = " + population.getFittest(k).displayCrossData() + "\n");

	    try {
		Provider.addProvider(new DefaultProvider("Default"));
		Cipher cipher = Cipher.getInstance("Substitution");
		SubstitutionKey sKey = (SubstitutionKey) HistoricalKey.getInstance("Substitution", cipher.getProvider());
		sKey.init(key + "/az");
		cipher.init(sKey);
		if (text.length() <= Analyzer.DECIPHER_LIMIT)
		    decrypt = cipher.decrypt(text);
		else
		    decrypt = cipher.decrypt(text.substring(0, Analyzer.DECIPHER_LIMIT));
                reportMissingLetters(decrypt);
		resultSB.append("-------------\nDECRYPTED MESSAGE(" + k + "):\n" + decrypt);
		resultSB.append("\nSOLUTION(" + k + "):\n" + solution + "\n");
		//		resultSB.append("PERCENT WORDS: " + num.format(TextUtilities.percentWords(solution,decrypt)) + "\n");
		resultSB.append("WRONG CHARS: " + TextUtilities.countInCorrectChars(solution,decrypt) + "\n-------------\n");
	    } catch (Exception e) {
		e.printStackTrace();
	    }
	    if (display != null) {                          // For use with CryptoToolJ
		display.append(resultSB.toString());
	    }
	}
    }

}

