package hcrypto.analyzer;

import hcrypto.cipher.*;
import hcrypto.engines.*;
import hcrypto.provider.*;

public class NgramGaIndividual extends GaIndividual implements Comparable {
    private final char SPACE = ' ';
    private NgramArray ngramArr;
    private int NN;  // Ngram length = 2, 3, 4, ...
    private int index[] = new int[2];          // Stores location of max(worst) and min(best) nGram scores
    private double scores[];

    public NgramGaIndividual(String text, String key, GaParameters params, NgramArray nga) {
	super(text, key, params);
	ngramArr = nga;
	NN = ngramArr.getNN();
	scores = new double[text.length()];
	mutate_policy = params.mutate_policy;
        calcFitness();
    }

    /**
     * Copy constructor
     */
    public NgramGaIndividual (GaIndividual i) {
	super(i);
	ngramArr = ((NgramGaIndividual)i).ngramArr;
	NN = ngramArr.getNN();
	for (int k = 0; k < index.length; k++)
	    index[k] = ((NgramGaIndividual)i).index[k];
	scores = new double[((NgramGaIndividual)i).scores.length];
	for (int k = 0; k < scores.length; k++)
	    scores[k] = ((NgramGaIndividual)i).scores[k];
    }

    public boolean equals(Object i) {
        return key.equals(((NgramGaIndividual)i).key);
    }

    /** 
     *  One-point crossover: Pick a random point in i1 and i2
     *   and swap up to NN characters starting from that point.
     */
    public void cross(GaIndividual i) {
	NgramGaIndividual i1 = this;
	NgramGaIndividual i2 = (NgramGaIndividual)i;

	int p = (int)(Math.random() * key.length());
	int pstop = p + NN;
	while (p < pstop && p < key.length()) {
	    char ch1 = i1.key.charAt(p);
	    char ch2 = i2.key.charAt(p);
	    i1.swap(ch1, ch2);
	    i2.swap(ch1, ch2);
	    p++;
	}
        i1.calcFitness();
        i2.calcFitness();
    }


    /**
     * This version of mutate swaps two random characters
     *  in the key. The mutation is done with frequency equal
     *  to rate. The resulting mutant is discarded if it
     *  does not improve the fitness of the individual.
     * @param rate is the mutation rate
     */
    public int mutate(double rate) {
	int a = 0, b = 0;

        if (Math.random() < rate) {
	    NgramGaIndividual tempIndy = new NgramGaIndividual(this);
            a = (int)(Math.random() * 26);
            b = (int)(Math.random() * 26);
            char ch = key.charAt(a);
            key.setCharAt(a, key.charAt(b));
            key.setCharAt(b, ch);
            this.calcFitness();
	    if (mutate_policy == GaParameters.ELITIST_MUTATION && this.compareTo(tempIndy) > 0)  {     // We mutate only if better
		this.key = tempIndy.key;             // So, restore the individual's state  if mutation is not an improvement
		this.fitness = tempIndy.fitness;
		this.decrypt = tempIndy.decrypt;
		return 0;
	    } else {
		return 1;
	    }
        } else {
	    return 0;  
	}
    }

    /**
     *  calcFitness creates a cipher object and decrypts the message with
     *   the current key.
     */
    public void calcFitness() {
	try {
	    sKey.init(key.toString() + "/az");
	    cipher.init(sKey);
	    decrypt = cipher.decrypt(ciphertext);
	    //	    fitness = ngramArr.recipDist(decrypt, index);
	    fitness = ngramArr.recipDist(decrypt, scores);
	} catch (Exception e) {
	    e.printStackTrace();
	}
	
    }

    /**
     *  Overrides the default version
     */
    public int compareTo(Object o) {
        NgramGaIndividual indy = (NgramGaIndividual)o;
	//	System.out.println(this.fitness + " vs.  " + indy.fitness);
        if (this.fitness < indy.fitness)
            return -1;
        else if (this.fitness > indy.fitness)
            return +1;
        else
            return 0;
    }

    public String displayCrossData() {
	return getKey() + " (" + getFitness() + ") ";
    }

    public String toString() {
	return getKey();
    }

    public String getKey() {
	return key.toString();
    }
}

