/*
 * File: NgramGaPopulation.java
 * @author R. Morelli <ralph.morelli@trincoll.edu>
 *
 * Description: This class implements an instance of a GaPopulation
 *  by providing implementations of the init()
 *  and run() methods for a NgramGaAnalyzer.
 */

package hcrypto.analyzer;

import java.util.*;
import java.text.*;
import hcrypto.cipher.*;

public class NgramGaPopulation extends GaPopulation {
    public static final int N_SHUFFLES = 20;
    public static final int KUCERA10 = 5;    // Very small seeding dictionary

    protected PatternDictionary seed_dict;   // The seeding dictionary
    private Token tokens[];           // An array of the tokens in the message
    private WordPairs pairs[];        // Pairs of crypto/plain from dictionary
    private int nPairs;           
    private int nTokens;
    private String text;               // Raw cipher text
    private int NN;
    private String book;
    private Alphabet alphabet;
    private NgramArray ngramArr;
    private int nTweaks = 0;
    private NumberFormat num;
    private int keyCount = 0;
    private int charErrs = 0;
    private double percentTokens = 0;

    /**
     * NgramGaPopulation() constructor
     * @param text -- the text to be cryptanalyzed
     * @param params -- a collection of parameters to control the run
     */
    public NgramGaPopulation (String text, String solution, GaParameters params) {
	super(text, solution, params);
	NN = params.NN;
	book = params.book;
	alphabet = params.alphabet;
	num = NumberFormat.getInstance();
	num.setMaximumFractionDigits(2);
	initNgramArray();
    }

    public void init() {}

    /**
     * Default init method.
     */
    private void initNgramArray() {
	try {
	    char arr[] = {'a','z',' ',' '};
	    alphabet = new Alphabet(arr);
	    ngramArr = new NgramArray(NN, book, alphabet);
	} catch (Exception e) {
	    e.printStackTrace();
	}
	//        individual = new NgramGaIndividual[size];
        individual = new NgramGaIndividual[size * 2];

        for (int k = 0; k < individual.length; k++) {
	    individual[k] = new NgramGaIndividual(cleantext, makeKey(), params, ngramArr);
	    ++keyCount;
	}
    }

    /**
     * This method creates an encryption key that is then used to
     *  seed the individual members of the population. The word pairs
     *  are created during initialization and stored in pairs[]. 
     *  Pairs of crypto,plain are selected at random.
     */
    private String makeKey() {
	return shuffle("abcdefghijklmnopqrstuvwxyz");
    }



    /**
     * This method implements the abstract run() method inherited from
     *  GaPopulation.
     */
    public void run() {
	int ptr = 0;
	printStartMessage();
	if (selection_policy == GaParameters.ELITIST_SELECTION) 
	    System.out.print("ELITIST SELECTION ");
	else 
	    System.out.print("PROPORTIONAL SELECTION ");
	if (mutate_policy == GaParameters.RANDOM_MUTATION) 
	    System.out.println("RANDOM MUTATION");
	else 
	    System.out.println("ELITIST MUTATION ");
        
        previousBestScore = Double.MAX_VALUE;
	//	reportDuplicates();
	
        do {
	    //	    if (tweakingOn && iterations - lastScoreChange > tweak_at) {
	    //		tweak();
	    //		++nTweaks;
	    //	    }
      
	    ptr = size;
	    for (int k = 0; k < size/2; k++) {      // Perform crosses on k andj 		                                   
		//		int j = size/2 +(int)(Math.random()* size/2); 		                 
		int j = (int)(Math.random()* size); 		                 
		double rand =Math.random();                 
		if (rand <= cross_rate) {
		    GaIndividual i1 = new NgramGaIndividual(individual[k]);
		    GaIndividual i2 = new NgramGaIndividual(individual[j]);
		    i1.cross(i2);
		    //		    individual[k] = i1;       // CHILDREN REPLACE PARENTS
		    //		    individual[j] = i2;
                    // NEW SOLUTION MERGE STRATEGY
                    if (newSolution(i1, ptr))
			individual[ptr++] = i1;   // CHILDREN AND PARENTS COMPETE
		    if (newSolution(i2, ptr))
			individual[ptr++] = i2;

		    //		    displayCrossData(individual[k], i1);
		    //		    displayCrossData(individual[j], i2);
		    //		    if (isBetter(i1, individual[k]))       // Select fitter
		    //			individual[k] = i1;
		    //		    if (isBetter(i2, individual[j]))
		    //			individual[j] = i2;
		}
	    } 
	    mutateAll();                          // Mutate all
            ++iterations;
	    //            reportDuplicates();
            java.util.Arrays.sort(individual);    // Sort the population
	    if (selection_policy == GaParameters.PROPORTIONAL_SELECTION) 
		selectProportional();             // PROPORTIONAL SELECTION
	    updateScore();
	    
	    keyCount += individual.length;
	    // COUNT ERRORS
	    //	    percentTokens = TextUtilities.percentWords(solution,individual[0].getDecrypt());
	    charErrs = TextUtilities.countInCorrectChars(solution,individual[0].getDecrypt());
    //	    displayAll();
	    if (verbose) 
	       displayBest();
	    if (display != null)
		display.setText(individual[0].getDecrypt());
	    //        } while (iterations < maxtrials && !success(individual[0]));
        } while (iterations < maxtrials && charErrs > 0);      
	    //        } while (iterations < maxtrials && (iterations - lastScoreChange) < 10);
	//        } while (iterations < maxtrials);
	    //        } while (iterations < maxtrials && !success(individual[0]) && (display == null || !display.threadIsStopped()));
	//        reportDuplicates();
	//	displayAll();
	displaySummary();
    }

    /** newSolution compares the individual, i, with all other individuals to
     *   determine whether it is unique.
     */
    private boolean newSolution(GaIndividual i, int ptr) {
        int k = 0;
	//        while (k < ptr) {
        while (k < individual.length) {
            if (i.getKey().equals(individual[k].getKey())) {
		//		System.out.println(k + ": " + i.toString() + " =  " + individual[k].toString());
		return false;
	    }
	    k++;
	}
	//	System.out.println("New individual : " + i.toString());
        return true;
    }

    private void reportDuplicates()  {
	int dupes = 0;
        int dupescores = 0;
	for (int k = 0; k < individual.length; k++) {
            for (int j = 0; j < individual.length; j++) {
                if (individual[j].getKey().equals(individual[k].getKey()) && j != k) {
                    System.out.println(k + " = " + j);
                    ++dupes;
		}
		
		if (individual[j].getFitness() == individual[k].getFitness() && j != k)
		    ++ dupescores;
	    }
	}
        System.out.println("In the population of size " + individual.length + " there were " + dupes 
		   + " duplicate individuals and " + dupescores + " identical scores.");
    }
    

    private boolean success(GaIndividual indy) {
	if (bestScore == individual[individual.length/2].getFitness()) {
	    return true;
	    //	} else if (bestScore == individual[individual.length/4].getFitness()) {
	    //	    tweak();
	    //	    return false;
	} else
	    return false;
	    //	    return ((NgramGaIndividual)indy).getFitness() < 8.
    }

    public GaIndividual getFittest(int n) {
        return individual[n];
    }

    /**
     * shuffle() randomly shuffles the characters in a StringBuffer.
     */
    private void shuffle(StringBuffer sb) {
	sb = new StringBuffer(shuffle(sb.toString()));
    }

    /**
     * shuffle() randomly shuffles the characters in a String.
     */
    private String shuffle(String s) {
	StringBuffer sb = new StringBuffer(s);
	for (int k = 0; k < N_SHUFFLES; k++) {
            int a = (int)(Math.random() * sb.length());
            int b = (int)(Math.random() * sb.length());
            char ch = sb.charAt(a);
            sb.setCharAt(a, sb.charAt(b));
            sb.setCharAt(b, ch);
	}
	return sb.toString();
    }

    private void shuffle(int arr[]) {
	for (int k = 0; k < arr.length - 1 ; k++) {
	    int m = (int)(Math.random() * (arr.length -1));
	    int temp = arr[k];
	    arr[k] = arr[m];
	    arr[m] = temp;
	}
    }

    private void tweak() {
	 if (verbose)
	     System.out.println("************************ Tweaking ***************************");
	 for (int k = 0; k < individual.length; k++)         // REPLACE THE ENTIRE POPULATION
	     individual[k] = new NgramGaIndividual(cleantext, makeKey(), params, ngramArr); 	 

	 java.util.Arrays.sort(individual);    // Sort the population
	 displayAll();
	 lastScoreChange = iterations; 	
	 previousBestScore = Double.MAX_VALUE;
	 bestScore = individual[0].getFitness();
    }
    
    private void displayCrossData(GaIndividual parent, GaIndividual child) {
	System.out.println("    " + "abcdefghijklmnopqrstuvwxyz");
	System.out.println("P : " + parent.displayCrossData());
	System.out.println("C : " + child.displayCrossData());
    }

    private void printStartMessage() {
        System.out.println("\nSTARTING ANALYSIS" +
                           "\t nChars " + cleantext.length() + 
                           "\t nIndivs " + size + 
                           "\nTEXT: " + cleantext + "\n");
    }

    private boolean isBetter(GaIndividual child, GaIndividual parent) {
	//	double  better = child.getFitness() - parent.getFitness();
	double  better = parent.getFitness() - child.getFitness();
	if (better > 0) {
	    //	    System.out.println("Parent= " + parent.getFitness() + " Child= "  + child.getFitness());
	    improved++;
	}
	else if (better < 0)
	    worsened++;
	else
	    nochange++;
	return better > 0;
    }

    private void displayAll() {
	for (int k = 0; k < size; k++)
	    System.out.println("---------------- " + k + " " + individual[k].displayCrossData());
    }

    private void mutateAll() {
	mutated=0;
	for (int k = 0; k < individual.length; k++) { // Perform Mutations
	    mutated += individual[k].mutate(mutate_rate);
	}
    }

    public void displayBest() {
	System.out.println(iterations + " " +  num.format(bestScore) + " " //********   + num.format(percentTokens) + "% " 
			   + charErrs + " ERRS " 
			   + keyCount + " KEYS "
			   + individual[0].toString()
			   //			   + " (" + improved + "," + worsened + "," + nochange + ")"
			   + " mutated=" + mutated
			   + " median = " + num.format(individual[individual.length/4].getFitness())
			   + " worst = " + num.format(individual[individual.length/2].getFitness())
			   );
    }

    public void displaySummary() {
        System.out.println("Finished: Iterations = " + iterations + " Best score is " + num.format(bestScore) + " KeyCount = " + keyCount);
	//        System.out.println("Got all " + nWords + " words at iteration number = " + gotall);
	//        System.out.println("Crosses improved= " + improved + " worsened= " + worsened + " nochange= " + nochange);
	//        System.out.println(decrypt(true));
    }

    private void updateScore() {
	bestScore = individual[0].getFitness();
	if (bestScore < previousBestScore) {
	    previousBestScore = bestScore;
	    lastScoreChange = iterations;
	}
    }
}
