package hcrypto.analyzer; // This class in part of the PolyGaAnalyzer analyzer in hcrypto 

import hcrypto.cipher.*;
import hcrypto.engines.*;
import hcrypto.provider.*;

public class PolyGaIndividual extends GaIndividual implements Comparable
{
    private final boolean MUTATE_BENEFICIAL = true;  // Mutate only if beneficial
    private final char SPACE = ' ';
    private NgramArray ngramArr;
    private int NN;  // Ngram length = 2, 3, 4, ...
    private int index[] = new int[2];          // Stores location of max(worst) and min(best) nGram scores
    private double scores[]; // records the scores of individuals
    private int dividePos;
    private AlbertiKey aKey;
    private final char divide = ',';
    private int shiftLen = 5;

    public PolyGaIndividual(String text, String key, GaParameters params, NgramArray nga) 
    {
	super(text, key, params);
	try 
        {
	    Provider.addProvider(new RamProvider("Ram"));
	    cipher = Cipher.getInstance("Alberti");
	    aKey = (AlbertiKey) HistoricalKey.getInstance("Alberti", cipher.getProvider());
	} 
	catch (Exception e) 
	{
	    e.printStackTrace();
	}
	ngramArr = nga;
	NN = ngramArr.getNN();
	scores = new double[text.length()];
	dividePos = key.length() - shiftLen - 1;
        calcFitness();
    }

    /**
     * Copy constructor
     */
    public PolyGaIndividual (GaIndividual i) 
    {
	super(i);
	try 
	{
		Provider.addProvider(new RamProvider("Ram"));
		cipher = Cipher.getInstance("Alberti");
		aKey = (AlbertiKey) HistoricalKey.getInstance("Alberti", cipher.getProvider());
	} 
	catch (Exception e) 
	{
	    e.printStackTrace();
	}

	ngramArr = ((PolyGaIndividual)i).ngramArr;
	NN = ngramArr.getNN();
	
	for (int k = 0; k < index.length; k++)
	    index[k] = ((PolyGaIndividual)i).index[k];
	
	scores = new double[((PolyGaIndividual)i).scores.length];
	
	for (int k = 0; k < scores.length; k++)
	    scores[k] = ((PolyGaIndividual)i).scores[k];
      	
	dividePos = key.length() - shiftLen - 1;
    }

    /** 
     *  One-point crossover: Pick a random point in i1 and i2
     *   and swap up to NN characters starting from that point.
     */
    public void cross(GaIndividual i) 
    {
	PolyGaIndividual i1 = this;
	PolyGaIndividual i2 = (PolyGaIndividual)i;

	int p = (int)(Math.random() * dividePos);
	int q = (int)((Math.random() * shiftLen) + dividePos);
	int pstop = p + NN;
	int qstop = q + 1;		//Altered so that fewer characters are swapped in the second keyword.
	
	//System.out.println(""+i1.key);  //Print out each key as it is crossed

	while (p < pstop && p < dividePos) 
	{
	    char ch1 = i1.key.charAt(p);
	    char ch2 = i2.key.charAt(p);
	    if(ch1 != divide && ch2 != divide)
	    {
		i1.swap(ch1, ch2);
		i2.swap(ch1, ch2);
	    }
	    p++;
	}
	while (q < qstop && q < key.length() && q > dividePos)
	{
	    int b = q;
	    char ch1 = i1.key.charAt(q);
	    char ch2 = i1.key.charAt((int)(Math.random() * dividePos));
	    if(ch1 != divide && ch2 != divide)
	    {
		i1.key.setCharAt(q, ch2);
	    }
	    q++;
	    
	    ch1 = i2.key.charAt(b);
	    ch2 = i2.key.charAt((int)(Math.random() * dividePos));
	    if(ch1 != divide && ch2 != divide)
	    {
		i2.key.setCharAt(b, ch2);
	    }	    
	}
	i1.calcFitness();  
	i2.calcFitness();
    }


    /**
     * This version of mutate swaps two random characters
     *  in the key. The mutation is done with frequency equal
     *  to rate. The resulting mutant is discarded if it
     *  does not improve the fitness of the individual.
     * @param rate is the mutation rate
     */
    public int mutate(double rate) 
    {

	int a = 0, b = 0;
	int c1 = 0, c2 = 0;
	char ch;
	
        if (Math.random() < rate) 
	{
	    PolyGaIndividual tempIndy = new PolyGaIndividual(this);
	    a = (int)(Math.random() * dividePos);
            b = (int)(Math.random() * dividePos);
	    if(a < dividePos && b < dividePos)
	    {
		ch = key.charAt(a);
		key.setCharAt(a, key.charAt(b));
		key.setCharAt(b, ch);
	    }

            a = (int)((Math.random() * shiftLen) + dividePos);
            b = (int)((Math.random() * shiftLen) + dividePos);
	    c1 = (int)(Math.random() * dividePos);
	    c2 = (int)(Math.random() * dividePos);	
	    if(a > dividePos && b > dividePos)
	    {
		key.setCharAt(a, key.charAt(c1));
		key.setCharAt(b, key.charAt(c2));
	    }

            this.calcFitness();

	    if (MUTATE_BENEFICIAL && this.compareTo(tempIndy) > 0)  
	    {     // We mutate only if better
	       	this.key = tempIndy.key;
		this.fitness = tempIndy.fitness;
	       	return 0;
	    } 
            else
	    {
		return 1;
	    } 
        }
	else
	{
	    return 0;  
	}
    }

    public void calcFitness() 
    {
	try 
	{
	    aKey.init(key.toString() + "/az"); // NEEDS TO TAKE IN KEYWORD,KEYWORD ALPHABET ALPHABET AS PARAMETERS OR JUST ONE STRING
	    cipher.init(aKey);
	    decrypt = cipher.decrypt(ciphertext);
	    fitness = ngramArr.recipDist(decrypt, scores);
	} 
	catch (Exception e) 
	{
	    e.printStackTrace();
	}
	
    }

    /**
     *  Overrides the default version
     */
    public int compareTo(Object o) 
    {
        PolyGaIndividual indy = (PolyGaIndividual)o;
	
	//System.out.println(this.fitness + " vs.  " + indy.fitness);
        
	if (this.fitness < indy.fitness)
            return -1;
        else if (this.fitness > indy.fitness)
            return +1;
        else
            return 0;
    }

    /**
     * displayCrossData() returns a String containing information about the individuals
     * @return String
     */
    public String displayCrossData() 
    {
	return getKey() + " (" + getFitness() + ") ";
    }

    /**
     * toString() activates getKey() and returns the key as a String
     * @return String
     */
    public String toString() 
    {
	return getKey();
    }
	
    /**
     * setKey() sets the key to AlphKey and ShiftKey seperated by a ','
     *
    private void setKey() 
    {
	String tempKey = (AlphKey.toString()+","+ShiftKey.toString());
	key = new String(tempKey);
    }*/

    /**
     * getKey() returns the key as a String
     * @return String
     */
    public String getKey() 
    {
	return key.toString();
    }
}

