/*
 * File: PolyGaPopulation.java
 * @author Will Servos <william.servos@trincoll.edu>
 *
 * Description: This class implements an instance of a GaPopulation
 *  by providing implementations of the init()
 *  and run() methods for a PolyGaAnalyzer.
 */

package hcrypto.analyzer;

import java.util.*;
import java.text.*;
import hcrypto.cipher.*;

public class PolyGaPopulation extends GaPopulation {
    public static final int N_SHUFFLES = 20;

    private int nPairs;           
    private int nTokens;
    private String text;               // Raw cipher text
    private int NN;
    private String book;
    private Alphabet alphabet;
    private NgramArray ngramArr;
    private int nTweaks = 0;
    private NumberFormat num;
    private int keyCount = 0;
    private int shiftLen = 5;  // shiftLen controls the length of the randomized Keyword2 (AlbertiKey) solutions

    /**
     * PolyGaPopulation() constructor
     * @param text -- the text to be cryptanalyzed
     * @param params -- a collection of parameters to control the run
     */
    public PolyGaPopulation (String text, String solution, GaParameters params) 
    {
	super(text, solution, params);
	NN = params.NN;
	book = params.book;
	alphabet = params.alphabet;
	num = NumberFormat.getInstance();
	num.setMaximumFractionDigits(2);
	initNgramArray();
    }

    public void init() {}

    /**
     * Default init method.
     */
    private void initNgramArray() 
    {
	try 
	{
	    char arr[] = {'a','z',' ',' '};
	    alphabet = new Alphabet(arr);
	    ngramArr = new NgramArray(NN, book, alphabet);
	} 
	catch (Exception e) 
	{
	    e.printStackTrace();
	}
	
	//individual = new PolyGaIndividual[size];
        individual = new PolyGaIndividual[size * 2];

        for (int k = 0; k < individual.length; k++) 
	{
	    String pkey = makePermutationKey();
	    String skey = makeShiftKey(shiftLen);
	    //System.out.println("Key= " + pkey + "," + skey);
	    individual[k] = new PolyGaIndividual(cleantext, pkey + "," + skey, params, ngramArr);
	    ++keyCount;
	}
    }

    /**
     * This method creates a randomized encryption key for
     * Alberti Cipher, including 
     */
    private String makePermutationKey() 
    {
        return shuffle("abcdefghijklmnopqrstuvwxyz");
	//return "activedbfghjklmnopqrsuwxyz";
	//	return "abcdefghijklmnopqrstuvwxyz";
    }
    
    private String makeShiftKey(int n) 
    {
	String tempKey = "";
	for(int i = 0; i < n; i++)
	{
		tempKey += (char)('a'+(int)(Math.random()*26));		
	}
	//	return tempKey;
	return "ralph";
    }

    /**
     * This method implements the abstract run() method inherited from
     *  GaPopulation.
     */
    public void run() 
    {
	int ptr = 0;
	printStartMessage();
        
        previousBestScore = Double.MAX_VALUE;
        do 
	{
	    if(tweakingOn && iterations - lastScoreChange > tweak_at) 
	    {
		tweak();
		++nTweaks;
	    }
      
	    ptr = size;
	    
	    for (int k = 0; k < size/2; k++) 
	    {      // Perform crosses on k andj 		                                   
		//int j = size/2 +(int)(Math.random()* size/2); 		                 
		
		int j = (int)(Math.random()* size); 		                 
		double rand = Math.random();                 
		if (rand <= cross_rate) 
		{
		    GaIndividual i1 = new PolyGaIndividual(individual[k]);
		    GaIndividual i2 = new PolyGaIndividual(individual[j]);
		    i1.cross(i2);
		    //individual[k] = i1;       // CHILDREN REPLACE PARENTS
		    //individual[j] = i2;
		    individual[ptr++] = i1;   // CHILDREN AND PARENTS COMPETE
		    individual[ptr++] = i2;
		    //displayCrossData(individual[k], i1);
		    //displayCrossData(individual[j], i2);
	
		    //if(isBetter(i1, individual[k]))       // Select fitter
		    //    individual[k] = i1;
		    //if (isBetter(i2, individual[j]))
		    //	  individual[j] = i2;
		}
	    } 
	    
	    mutateAll();                          // Mutate all	    
            ++iterations;
            
	    java.util.Arrays.sort(individual);    // Sort the population
	    
	    //displayAll();
	    
	    updateScore();
	    
	    if (verbose) 
	       displayBest();
	    
	    keyCount += individual.length;
        } 
	while(iterations < maxtrials && !success(individual[0]));
	//	displayAll();
	    displaySummary();
    }

    private boolean success(GaIndividual indy) 
    {
	if (bestScore == individual[individual.length/2].getFitness()) 
	{
	    return true;
	    //	} else if (bestScore == individual[individual.length/4].getFitness()) {
	    //	    tweak();
	    //	    return false;
	} 
	else
	    return false;
            //	    return ((PolyGaIndividual)indy).getFitness() < 8.
    }

    public GaIndividual getFittest(int n) 
    {
        return individual[n];
    }

    /**
     * shuffle() randomly shuffles the characters in a StringBuffer.
     */
    private void shuffle(StringBuffer sb) 
    {
	sb = new StringBuffer(shuffle(sb.toString()));
    }

    /**
     * shuffle() randomly shuffles the characters in a String.
     */
    private String shuffle(String s) 
    {
	StringBuffer sb = new StringBuffer(s);
	for (int k = 0; k < N_SHUFFLES; k++) 
	{
            int a = (int)(Math.random() * sb.length());
            int b = (int)(Math.random() * sb.length());
            char ch = sb.charAt(a);
            sb.setCharAt(a, sb.charAt(b));
            sb.setCharAt(b, ch);
	}
	return sb.toString();
    }

    private void shuffle(int arr[]) 
    {
	for (int k = 0; k < arr.length - 1 ; k++) 
	{
	    int m = (int)(Math.random() * (arr.length -1));
	    int temp = arr[k];
	    arr[k] = arr[m];
	    arr[m] = temp;
	}
    }

    private void tweak() 
    {
	 if (verbose)
	     System.out.println("************************ Tweaking ***************************");
	 for (int k = 0; k < individual.length; k++)         // REPLACE THE ENTIRE POPULATION
	     individual[k] = new PolyGaIndividual(cleantext, makePermutationKey()+","+makeShiftKey(shiftLen), params, ngramArr); 	 

	 java.util.Arrays.sort(individual);    // Sort the population
	 displayAll();
	 lastScoreChange = iterations; 	
	 previousBestScore = Double.MAX_VALUE;
	 bestScore = individual[0].getFitness();
    }
    
    private void displayCrossData(GaIndividual parent, GaIndividual child) 
    {
	System.out.println("    " + "abcdefghijklmnopqrstuvwxyz");
	System.out.println("P : " + parent.displayCrossData());
	System.out.println("C : " + child.displayCrossData());
    }

    private void printStartMessage() 
    {
        System.out.println("\nSTARTING ANALYSIS" +
                           "\t nChars " + cleantext.length() + 
                           "\t nIndivs " + size + 
                           "\nTEXT: " + cleantext + "\n");
    }

    private boolean isBetter(GaIndividual child, GaIndividual parent) 
    {
	//	double  better = child.getFitness() - parent.getFitness();
	double  better = parent.getFitness() - child.getFitness();
	if (better > 0) 
	{
	    //	    System.out.println("Parent= " + parent.getFitness() + " Child= "  + child.getFitness());
	    improved++;
	}
	else if (better < 0)
	    worsened++;
	else
	    nochange++;
	return better > 0;
    }

    private void displayAll() 
    {
	for (int k = 0; k < size; k++)
	    System.out.println("---------------- " + k + " " + individual[k].displayCrossData());
    }

    private void mutateAll() 
    {
	mutated=0;
	for (int k = 0; k < individual.length; k++) 
	{ // Perform Mutations
	    mutated += individual[k].mutate(mutate_rate);
	}
    }

    public void displayBest() 
    {
	System.out.println(iterations + " " +  num.format(bestScore) + " "
			   + individual[0].toString()
			   + " (" + improved + "," + worsened + "," + nochange + ")"
			   + " mutated=" + mutated
			   + " median = " + num.format(individual[individual.length/4].getFitness())
			   + " worst = " + num.format(individual[individual.length/2].getFitness())
			   );
    }

    public void displaySummary() 
    {
        System.out.println("Finished: Iterations = " + iterations + " Best score is " + num.format(bestScore) + " KeyCount = " + keyCount);
	//        System.out.println("Got all " + nWords + " words at iteration number = " + gotall);
        System.out.println("Crosses improved= " + improved + " worsened= " + worsened + " nochange= " + nochange);
	//        System.out.println(decrypt(true));
    }

    private void updateScore() 
    {
	bestScore = individual[0].getFitness();
	//	bestScore = individual[individual.length-1].getFitness();
	//	if (bestScore > previousBestScore) {
	if (bestScore < previousBestScore) 
	{
	    previousBestScore = bestScore;
	    lastScoreChange = iterations;
	}
    }
}
