/*
 * File: TextAnalyzer.java
 * @author R. Morelli <ralph.morelli@trincoll.edu>
 * 
 * Description: This creates a frame in which frequency data are displayed. One
 *  panel of the window contains a histogram. Another contains numerical data. 
 *  Instances of this class are created in CipherFrame.showStatistics().
 *
 *  <P>Copyright: This program is in the public domain. You can modify it as you 
 *  see fit as long as you properly acknowledge its original author.
 *  It would also be nice if you forwarded your changes to 
 *  <A HREF= "mailto:ralph.morelli@trincoll.edu">ralph.morelli@trincoll.edu</A> so 
 *  they can possibly be added to the "official" version.

 */
 
package hcrypto.analyzer;

import java.awt.*;
import java.awt.event.*;
import java.awt.datatransfer.*;

import hcrypto.cipher.*;
import hcrypto.analyzer.*;

public class TextAnalyzer extends AnalyzerFrame implements Analyzer, 
               ItemListener 
{
    static final int HIST_WIDTH = Histogram.WIDTH;
    static final int HIST_HEIGHT = Histogram.HEIGHT;
    static final int WIN_WIDTH = HIST_WIDTH + 50;
    static final int WIN_HEIGHT = HIST_HEIGHT + 200;
    static final int BAR_HEIGHT = 12;

    private Histogram histogram;
    private FrequencyTable freqTable;
    private TextStatistics stats;
    private int alphabetRange;    
    private String text;

    private Panel histoPanel;
    private Checkbox azButton, AZButton, azAZButton, azAZ09Button, printableButton, asciiButton;
    private CheckboxGroup alphabetGroup;
    private Button list, save, copy;
    private Label count, alphaCount, coIndex;
    private Label chValue, chCount;
    private java.text.NumberFormat nf;
    public static int nBins = 256;
    
    //    public TextAnalyzer(TextStatistics stats) {
    public TextAnalyzer() {
	super();
        setTitle("Text Analyzer");
        
        setSize(WIN_WIDTH,WIN_HEIGHT);
        setLocation(400,100);
        setVisible(true);
    }
    
    /**
     * setup() is part of Analyzer interface. It passes in the pointers
     *  to the text. It fills in the statistics that it gets from the text.
     */     
    public void setup(String text) { 
        this.text = text;
        stats = new TextStatistics(text,true);
        freqTable = stats.getFrequencyTable();
        nf = java.text.NumberFormat.getNumberInstance();
        nf.setMaximumFractionDigits(3); 
        histogram = new Histogram(freqTable);
        histoPanel.add(histogram);

	// Fill in the statistics
	count.setText("  Chars: " + stats.charCount);
	alphaCount.setText("  Alphabetics " + stats.alphabeticsCount);
	coIndex.setText("  Coincidence Index " + nf.format(stats.coincidenceIndex));
    }

    /**
     * buildGUI() overrides the default method inherited from AnalyzerFrame.
     *  It is called in the super constructor.
     */
    protected void buildGUI() {
        setLayout(new BorderLayout());
        histoPanel = new Panel();
        add(histoPanel,"Center");
        
        Panel controlPanel = new Panel();
        Panel buttonsPanel = new Panel();
        buttonsPanel.setLayout(new GridLayout(6,1));
        addButtons(buttonsPanel);

        Panel statsPanel = new Panel();
        statsPanel.setLayout(new GridLayout(5, 1));
        addStats(statsPanel);
        
	controlPanel.add(buttonsPanel);
	controlPanel.add(statsPanel);
	add(controlPanel, "South");
	pack();	
        this.addWindowListener(new WindowAdapter() {
		public void windowClosing(WindowEvent evt) {
		    setVisible(false);
		} // windowClosing()
		//		public void windowActivated(WindowEvent e) {
		//		    setMenuBar(Menus.getMenuBar());    
		//		}	
        });    //addWindowListener()    	
    }
    
    private void addStats(Panel statsPanel) {
	count = new Label("                     ");
	count.setFont(new Font("Monospaced", Font.PLAIN, 12));
	
	statsPanel.add(count);
	alphaCount = new Label("                     ");
	alphaCount.setFont(new Font("Monospaced", Font.PLAIN, 12));

	statsPanel.add(alphaCount);
	coIndex = new Label("                     ");
	coIndex.setFont(new Font("Monospaced", Font.PLAIN, 12));
	
	statsPanel.add(coIndex);		
	chValue  = new Label("                     ");
	chValue.setFont(new Font("Monospaced", Font.PLAIN, 12));
	statsPanel.add(chValue);
	chCount = new Label("                     ");
	chCount.setFont(new Font("Monospaced", Font.PLAIN, 12));
	statsPanel.add(chCount);		
    }
    
    private void addButtons(Panel buttonsPanel) {
	alphabetGroup = new CheckboxGroup();   
        azButton = new Checkbox("a-z");
        azButton.setCheckboxGroup(alphabetGroup);
        azButton.addItemListener(this);
        buttonsPanel.add(azButton);
        AZButton = new Checkbox("A-Z");
        AZButton.setCheckboxGroup(alphabetGroup);
        AZButton.addItemListener(this);
        buttonsPanel.add(AZButton);
        azAZButton = new Checkbox("a-z & A-Z"); 
        azAZButton.setCheckboxGroup(alphabetGroup);
        azAZButton.addItemListener(this);
        buttonsPanel.add(azAZButton);
        azAZ09Button = new Checkbox("a-z & A-Z & 0-9");
        azAZ09Button.setCheckboxGroup(alphabetGroup);
        azAZ09Button.addItemListener(this);
        buttonsPanel.add(azAZ09Button);
        printableButton = new Checkbox("printable characters");
        printableButton.setCheckboxGroup(alphabetGroup);
        printableButton.addItemListener(this);
        buttonsPanel.add(printableButton);
        asciiButton = new Checkbox("All ASCII");
        asciiButton.setCheckboxGroup(alphabetGroup);
        asciiButton.setState(true);        // Default state
        asciiButton.addItemListener(this);        
        buttonsPanel.add(asciiButton);
    }
    
    public void itemStateChanged(ItemEvent evt) {
        if (azButton.getState())
           alphabetRange = AlphabetFactory.ALPH_az;
        else if(AZButton.getState())
           alphabetRange = AlphabetFactory.ALPH_AZ;
        else if(azAZButton.getState())
           alphabetRange = AlphabetFactory.ALPH_azAZ;
        else if(azAZ09Button.getState())
           alphabetRange = AlphabetFactory.ALPH_azAZ09;
        else if(printableButton.getState())
           alphabetRange = AlphabetFactory.ALPH_printable;
        else if(asciiButton.getState())
           alphabetRange = AlphabetFactory.ALPH_ascii;
        
        freqTable.setAlphabetRange(alphabetRange);
        stats.calcStatistics();
        histogram.redraw();
        
	count.setText("  Chars: " + stats.charCount);
	alphaCount.setText("  Alphabetics: " + stats.alphabeticsCount);
	coIndex.setText("  Coincidence Index: " + nf.format(stats.coincidenceIndex));        
    } // actionPerformed()
        
    /**
     * run() is part of Analyzer interface. It performs the analysis.
     */     
    public void run() {
    }

    /**
     * getReport() is part of Analyzer interface. It performs the analysis.
     *  In this case the analyzer's report is empty. 
     */     
    public String getReport() {  
	return "";
    }
    
    
    /**
     * toString() is part of Analyzer interface. It can return anything.
     */     
    public String toString() {
        return "TestAnalyzer";
    }
    
    
}
                                       
                                       
