/*
 * File: TypeIVGaPopulation.java
 * @author R. Morelli <ralph.morelli@trincoll.edu>
 *
 * Description: This class implements an instance of a GaPopulation
 *  by providing implementations of the init()
 *  and run() methods for a TypeIVGaAnalyzer.
 */

package hcrypto.analyzer;

import java.util.*;
import java.text.*;
import hcrypto.cipher.*;

public class TypeIVGaPopulation extends GaPopulation {
    private static final boolean NO_DUPLICATES = true;
    private int NN;
    private int nCrosses=0;
    private int nImproved = 0;
    private String book;
    private Alphabet alphabet;
    private NgramArray ngramArr;
    private NumberFormat num;
    private int keyCount = 0;
    private int shiftLen = 0;  // shiftLen controls the length of the randomized Keyword2 (AlbertiKey) solutions
    private String correctPermKey;
    private GaParameters params;
    private int nWrong = 0;

    private String correctShiftword;
    private String bestKey = "";

    private String cycles[];
    //    private String freqmap = "eqaxskittlohxnqjzg"; // EQ, TL, OH, AX, SK, IT, XN, QJ, ZG
    private String freqmap = "egajsaistkoexuqwzd";  // EG, TK, AJ, IS, SA, OE, XU, QW, ZD

    /**
     * TypeIVGaPopulation() constructor
     * @param text -- the text to be cryptanalyzed
     * @param params -- a collection of parameters to control the run
     */

    public TypeIVGaPopulation (String text, String solution, GaParameters params, int shiftlen, String keyspec) {
	super(text, solution, params);
	correctShiftword = keyspec.substring(keyspec.lastIndexOf(',')+1);
        correctPermKey = keyspec.substring(0, keyspec.indexOf(','));
	//	correctPermKey = "eqjucohaxnfrskydpitlzgwvbm";
	correctPermKey = "egprhqwisajvtkxublyncmzdfo";
	correctShiftword = "gcqsv";
	System.out.println("Hack: permkey= " + correctPermKey + " shiftkey= " + correctShiftword);
	this.params = params;
	NN = params.NN;
	book = params.book;
	alphabet = params.alphabet;
	num = NumberFormat.getInstance();
	num.setMaximumFractionDigits(3);
	this.shiftLen = shiftlen;
	initNgramArray();
    }

    public void init() {}

    /**
     * Default init method.
     */
    private void initNgramArray() {
	try 
	{
	    Alphabet alphabet = AlphabetFactory.getInstance(AlphabetFactory.ALPH_az);
	    ngramArr = new NgramArray(NN, book, alphabet);
	    //	    ngramArr = new NgramArray(2, book, alphabet,5);
	    System.out.println("NgramArray initialized for step size = " + ngramArr.getStepSize() + " and NN= " + ngramArr.getNN());
	} 
	catch (Exception e) 
	{
	    e.printStackTrace();
	}
    }

    /** initPopulation() creates random individuals for the population.
     * @param seedPairs -- a string of letter pairs representing likely
     *  mappings of the form E-->G, 'E' encrypts to 'G'. These are derived
     *  from frequency analysis.
     * @param shiftKey -- a string giving the shiftkey.
     */
    private void initPopulation(String seedPairs, String shiftKey) {
        individual = new TypeIVGaIndividual[size * 2];
	int a = 8;
        for (int k = 0; k < individual.length; k++) {
	    String pKey = "";
	    LetterKeyMap onelettermap[] = new LetterKeyMap[1];
	    onelettermap[0] = new LetterKeyMap(shiftKey.charAt(1), a);
	    //	    onelettermap[0] = new LetterKeyMap(shiftKey.charAt(1), 20);
	    LetterKeyMap keymap[] = makeIndividualKeyMap(shiftKey, a);
	    //	    ++a;
	    //	    if (a > 24)
	    //		a = 2;
	    do {
		pKey = seedKey(seedPairs);
		pKey = saltKey(pKey, seedPairs, onelettermap);
	    } while (!TypeIVGaIndividual.isLegal(new StringBuffer(pKey), seedPairs, onelettermap));
	    //	    pKey = "egprhqwisajvtkxublyncmzdfo";
	    //	    String pKey = "eqjucohaxnfrskydpitlzgwvbm";
	    individual[k] = new TypeIVGaIndividual(cleantext, pKey, shiftKey, keymap, params, ngramArr);
	    //	    System.out.print(k + ": " + pKey + " ");
	    //	    System.out.println(((TypeIVGaIndividual)individual[k]).isLegal(seedPairs));
	    ++keyCount;
	}
    }

    private String saltKey(String pkey, String seedpairs, LetterKeyMap map[]) {
	StringBuffer pkeybuffer = new StringBuffer(pkey);
	int indx = pkey.indexOf(map[0].ch);
	char ch1 = pkeybuffer.charAt(map[0].location);  // Swap ch into loc
	pkeybuffer.setCharAt(indx, map[0].ch);
	pkeybuffer.setCharAt(map[0].location, ch1);
	if (TypeIVGaIndividual.isLegal(pkeybuffer, seedpairs, map))
	    return pkeybuffer.toString();
	else
	    return pkey;    // No change if the resulting change is illegal
    }

    /**
     * makeIndividualKeyMap() generates a mapping of the frequent letters for each column
     *
     */
    private LetterKeyMap[] makeIndividualKeyMap(String shiftKey, int a) {
	//	System.out.println("a= " + a + " b= " + b);
	//	LetterKeyMap map[] = new LetterKeyMap[shiftKey.length()];
	LetterKeyMap map[] = new LetterKeyMap[2];
	map[0] = new LetterKeyMap(shiftKey.charAt(0), 1); // g
	map[1] = new LetterKeyMap(shiftKey.charAt(1), a); // c
	//	map[1] = new LetterKeyMap(shiftKey.charAt(3), a); // s
	//	map[2] = new LetterKeyMap(shiftKey.charAt(2), b); // q
	//	map[3] = new LetterKeyMap(shiftKey.charAt(3), 8); // s
	//	map[4] = new LetterKeyMap(shiftKey.charAt(4), 11); // v
	return map;
    }

    /**
     * seedKey() this version of seed key places the letter 'e' for each
     *  column in the correct location in the key.
     *  In this case 'e' goes to 0, 'g' to 1, 'q' to 5, 's' to 8, 'v' to 11, 'c' to 20
     * @param keymap -- an array of LetterKeyMap, holding the locations of the most frequent letters.
     */
    private String seedKey(LetterKeyMap keymap[]) {
	String alpha = TextUtilities.shuffle("abdfhijklmnoprtuwxyzcvq");   // Randomize alphabet
	LetterKeyMap mapcopy[] = new LetterKeyMap[keymap.length];
	for (int k = 0; k < keymap.length; k++)
	    mapcopy[k] = new LetterKeyMap(keymap[k]);
	java.util.Arrays.sort(mapcopy);    // Sort the mappings so locations are small to large
	int mapPtr = 0;
	int alphaPtr = 0;
	StringBuffer pkey = new StringBuffer("e");
	for (int k = 1; k < 26; k++) {
	    if (k == mapcopy[mapPtr].location) {
		pkey.append(mapcopy[mapPtr].ch);
		mapPtr = (mapPtr + 1) % mapcopy.length;
	    }
	    else
		pkey.append(alpha.charAt(alphaPtr++));
	}
	return pkey.toString();
    }

    /**
     * seedKey() randomly inserts letter pairs into the permutation
     *  key, such that the resulting key is a permutation of a-z.
     */
    private String seedKey(String seedPairs) {
	String alpha = TextUtilities.shuffle("abcdefghijklmnopqrstuvwxyz");   // Randomize alphabet
	StringBuffer pkey = new StringBuffer(seedPairs.substring(0,2)); // Insert first pair e-->?
	//	for (char ch = 'a'; ch <= 'z'; ch++) 
	for (int k = 0; k < alpha.length(); k++)                // Insert asterisks and non-pair letters
	    if (seedPairs.indexOf(alpha.charAt(k)) == -1)       //  but don't break pairs
		pkey.append("*"+ alpha.charAt(k));
	//		pkey.append("*");
	pkey.append("*");

	int p = 2;   // Key starts with 'e' so the first two letters are not touched
	while (p < seedPairs.length()) {
	    String pair = seedPairs.substring(p, p+2);  // Get next seed pair
	    //	    System.out.println("pair= " + pair);
	    p += 2;
	    if (pkey.indexOf(pair) == -1) {   // If the pair isn't already in pkey
		int indx1 = pkey.indexOf(""+pair.charAt(0));
		int indx2 = pkey.indexOf(""+pair.charAt(1));
		if (indx1 != -1) {                          // First letter is in pkey
		    pkey.insert(indx1+1, pair.charAt(1));
		} else if (indx2 != -1) {                   // Second letter is in pkey
		    if (indx2 != 0)
			pkey.insert(indx2, pair.charAt(0));
		    else
			pkey.append(pair.charAt(0)+"*");   // Otherwise at end
		} else {                        // Otherwise randomly insert the pair
		    int indx = 2 + (int)(Math.random() * (pkey.length()-4));
		    int insertPt = pkey.indexOf("*",indx);  // Find an asterisk
		    if (insertPt != -1)
			pkey.insert(insertPt, "*"+pair);
		    else
			pkey.append(pair+"*");
		}
	    }
	}
	return TextUtilities.remove(pkey.toString(),'*');
    }

    private int countWrongs(String s1, String s2) {
	if (s1.length() != s2.length())
	    return Math.max(s1.length(), s2.length());
	int sum = 0;
	for (int k = 0; k < s1.length(); k++)
	    if (s1.charAt(k) != s2.charAt(k))
		++sum;
	return sum;
    }

    /**
     * This method implements the abstract run() method inherited from
     *  GaPopulation.
     */
    public void run() {
	printStartMessage();
	//  Key for odd shift (=1) in column 1:  egprhqwisajvtkxublyncmzdfo
	initPopulation("egajsaistkoexuqwzd", "gcqsv"); // EG,ISA,TK,O in correct locations
        //  Key for odd shift (=1) in 3rd column:  eqjucohaxnfrskydpitlzgwvbm
	//	initPopulation("eqaxskittlohxnqjzg", shiftKeys); // EG,ISA,TK,O in correct locations
	runPool();
	System.out.println("PHASE_4: nIters= " + iterations 
			   + "\tNN= " + params.NN + "\tsize= " + params.size + "\tmrate= " + params.mutate_rate
			   + "\tnWrongPermKey= " + countWrongs(correctPermKey, ((TypeIVGaIndividual)individual[0]).getPermKey())
			   + "\tnWrongShifts= " + countWrongs(correctShiftword, ((TypeIVGaIndividual)individual[0]).getShiftKey())
			   + "\tkey= " + ((TypeIVGaIndividual)individual[0]).getKey()
			   + " " + num.format(bestScore) 
			   + " Decrypt= " +  individual[0].getDecrypt().substring(0,30)
			   + "\tkeyCount= " + keyCount
			   );

    }

    /**
     * runPool() runs the GA on the population of individuals. The stopping
     *  criterion depends on how the success() method is defined.
     */
    private void runPool() {
	int ptr = 0;
	iterations = 0;
	nCrosses = 0;
        previousBestScore = Double.MAX_VALUE;
	individual = selectPopulation(individual, size, NO_DUPLICATES);
	//	java.util.Arrays.sort(individual);    // Sort the population
	displayAll();
	int cutoff = size/10;
        do 
	{
	    //	    ++cutoff;
	    //	    if (cutoff > size)
	    //		cutoff = size/10;;
	    ptr = size;
	    GaIndividual i1 = null, i2 = null;
	    int k = 0;
	    for (k = 0; k < size/2; k++) {      // Perform crosses on k andj 		                                   
		int j = 0;
		//		int i = (int)(Math.random() * cutoff);          // First mate from top strata
		int i = (int)(Math.random() * size);          // First mate from top strata
		//		do {
		j = (int)(Math.random() * size); 	// Second mate taken from entire population
		    //		    System.out.print("!");
		    //		} while (individual[i].equals(individual[j]));
		double rand = Math.random();                 
		//		System.out.print(k+","+j+" ");
		//		System.out.println(rand + " " + cross_rate);
		if (rand <= cross_rate) {
		    i1 = new TypeIVGaIndividual(individual[i]);
		    i2 = new TypeIVGaIndividual(individual[j]);
		    boolean isLegal = true;
		    boolean areEqual = false;
		    GaIndividual i1_copy = null;
		    GaIndividual i2_copy = null; 
		    do {
			i1_copy = new TypeIVGaIndividual(i1);
			i2_copy = new TypeIVGaIndividual(i2);
			i1_copy.cross(i2_copy);
			isLegal = ((TypeIVGaIndividual)i1_copy).isLegal(freqmap) &&  ((TypeIVGaIndividual)i2_copy).isLegal(freqmap);
			//			areEqual = i1.equals(i1_copy) || i2.equals(i2_copy) || i1_copy.equals(i2_copy);
			//			System.out.print(",");
		    } while (!isLegal) ;
		    //		    } while (!isLegal || areEqual) ;
		    i1 = i1_copy;
		    i2 = i2_copy;
		    ++nCrosses;
		    individual[ptr++] = i1;   // CHILDREN AND PARENTS COMPETE
		    isBetter(i1,individual[i]);
		    individual[ptr++] = i2;
		    isBetter(i2,individual[j]);
		}
	    }
	    //	    System.out.println();
	    mutateAll();                          // Mutate all	    
            ++iterations;
	    individual = selectPopulation(individual, size, NO_DUPLICATES);

	    //	    java.util.Arrays.sort(individual);    // Sort the population
	    nWrong = countWrongs(correctPermKey, ((TypeIVGaIndividual)individual[0]).getPermKey());
	    if (iterations % 10 == 0)
		displayAll();
	    updateScore();
	    if (verbose) 
	       displayBest();
	    keyCount += individual.length;
        } 
	//	while(iterations < maxtrials && !success(individual[0]) && nWrong != 0);
	while(iterations < maxtrials && !success(individual[0]));
	//	displayAll();
	//	if (verbose)
	    displaySummary();
    }

    private GaIndividual[] selectPopulation(GaIndividual oldPopulation[], int size, boolean noDuplicates) {
	java.util.Arrays.sort(oldPopulation);    // Sort the population
	GaIndividual newPopulation[] = new GaIndividual[oldPopulation.length];
	if (noDuplicates) {
	    int j = 0;
	    for (int k = 0; k < size; k++) {
		newPopulation[k] = oldPopulation[j];
		while (oldPopulation[j].equals(newPopulation[k])) 
		    j++;
	    }
	    // Fill up the rest of the population
	    int k = size;
	    for ( ; j < oldPopulation.length && k < newPopulation.length; k++) {
		newPopulation[k] = oldPopulation[j];
		j++;
	    }
	    while (k < newPopulation.length) {
		newPopulation[k] = oldPopulation[oldPopulation.length-1];
		k++;
	    }
		
	    return newPopulation;
	}
	else return oldPopulation;
    }

    private boolean success(GaIndividual indy) {
	if (bestScore == individual[individual.length/2].getFitness()) 	{
	    return true;
	} 
	else if (bestScore < 1.11)
	    return true;
	else
	    return false;
    }

    public GaIndividual getFittest(int n) 
    {
        return individual[n];
    }

    private void displayCrossData(GaIndividual parent, GaIndividual child) 
    {
	System.out.println("    " + "abcdefghijklmnopqrstuvwxyz");
	System.out.println("P : " + parent.displayCrossData());
	System.out.println("C : " + child.displayCrossData());
    }

    private void printStartMessage() 
    {
        System.out.print("GA_ANALYSIS:" 
			   + "\t nChars= " + cleantext.length()
                           + "\t nIndivs= " + size  
			   //                         +  "\nTEXT: " + cleantext + "\n"
			   );
	if (selection_policy == GaParameters.ELITIST_SELECTION) 
	    System.out.print("\tselection= ELITIST");
	else 
	    System.out.print("\tselection= PROPORTIONAL");
	if (mutate_policy == GaParameters.RANDOM_MUTATION) 
	    System.out.println("\tmutation= RANDOM");
	else 
	    System.out.println("\tmutation= ELITIST");
        
    }

    private boolean isBetter(GaIndividual child, GaIndividual parent) 
    {
	//	double  better = child.getFitness() - parent.getFitness();
	double  better = parent.getFitness() - child.getFitness();
	if (better > 0) 
	{
	    //	    System.out.println("Parent= " + parent.getFitness() + " Child= "  + child.getFitness());
	    improved++;
	}
	else if (better < 0)
	    worsened++;
	else
	    nochange++;
	return better > 0;
    }

    private void displayAll()  {
	int count[] = new int[26];
	double score[] = new double[26];
	for (int k = 0; k < size; k++) {
	    int indx = ((TypeIVGaIndividual)individual[k]).getPermKey().indexOf("isaj");
	    if (indx != -1) {
		++count[indx];
		score[indx] += ((TypeIVGaIndividual)individual[k]).getFitness();
	    }
	    
	    //	    System.out.println("---------------- " + k + " " + individual[k].displayCrossData() 
	    //			       + " nWrong= " + countWrongs(correctPermKey, ((TypeIVGaIndividual)individual[k]).getPermKey())
	    //			       + "\t" + ((TypeIVGaIndividual)individual[k]).getPermKey().indexOf("isaj")
	    //			       );
	}
	for (int k = 0; k < count.length; k++)
	    if (count[k] != 0)
		System.out.print(k + ":" + count[k] + "," + (int)(score[k]/count[k]) + " ");
	System.out.println();
    }

    private void mutateAll() 
    {
	mutated=0;
	for (int k = 0; k < individual.length; k++) 
	{ // Perform Mutations
	    mutated += individual[k].mutate(mutate_rate);
	}
    }

    public void displayBest() 
    {
	nImproved += improved;
	System.out.println(iterations + " " +  num.format(bestScore) + " "
			   + individual[0].toString()
			   + " (" + nCrosses + "," + improved + "," + worsened + "," + nochange + ")"
			   + " mutated=" + mutated
			   + " median = " + num.format(individual[individual.length/4].getFitness())
			   + " worst = " + num.format(individual[individual.length/2].getFitness())
			   + " decrypt= " + individual[0].getDecrypt().substring(0,30)
			   + " nImproved= " + nImproved
			   //			   + " nWrong= " + countWrongs(correctPermKey, ((TypeIVGaIndividual)individual[0]).getPermKey())
			   //			   + " IC= " + ((TypeIVGaIndividual)individual[0]).calcIC(NN)
			   //			   + " CHI= " + ((TypeIVGaIndividual)individual[0]).calcCHI(NN)
			   );
	nCrosses = improved = worsened = nochange = 0;
    }

    public void displaySummary() 
    {
	FastIC ic = new FastIC(individual[0].getDecrypt(), correctShiftword.length());
        System.out.println("Finished: Iterations= " + iterations + " Score= " + num.format(bestScore) + " KeyCount= " + keyCount
			   + " key= " + ((TypeIVGaIndividual)individual[0]).getKey()
			   //			   + " CHI= " + ((TypeIVGaIndividual)individual[0]).calcCHI(NN)
			   + " nWrong= " + countWrongs(correctPermKey, ((TypeIVGaIndividual)individual[0]).getPermKey())
			   + " Decrypt= " +  individual[0].getDecrypt().substring(0,30)
			   + " nWrong= " + countWrongs(individual[0].getDecrypt().substring(0,30), solution.substring(0,30))
			   //			   + " ic= " + num.format(ic.getIC()) + " ic_2= " + num.format(ic.getAvgFirstNIC(2))
			   );
	//        System.out.println("Got all " + nWords + " words at iteration number = " + gotall);
	//        System.out.println("Crosses improved= " + improved + " worsened= " + worsened + " nochange= " + nochange);
	//        System.out.println(decrypt(true));
    }

    private void updateScore() 
    {
	bestScore = individual[0].getFitness();
	//	bestScore = individual[individual.length-1].getFitness();
	//	if (bestScore > previousBestScore) {
	if (bestScore < previousBestScore) 
	{
	    previousBestScore = bestScore;
	    lastScoreChange = iterations;
	}
    }


}
