package hcrypto.cipher;

/** 
 * This class defines properties of the alphabet used in a cipher key.
 * Each <A href="HistoricalKey.html">HistoricalKey</A> has two associated
 * alphabets, a <i>plaintext alphabet</i> and a <i>ciphertext alphabet</i>. 
 * In the <b>default</b> case, both alphabets are identical. An encryption
 * engine <i>encodes</i> a character by mapping it from the plaintext alphabet 
 * to a corresponding character in the ciphertext alphabet.  It <i>decodes</i>
 * a character by mapping it fromt the ciphertext alphabet to a corresponding
 * character in the plaintext alphabet.

 * <P>An alphabet is simply the set of characters for which encryption
 * and decryption are defined. Characters in the plaintext that are not members
 * of the plaintext alphabet are simply left unencrypted. All of the ciphers assume
 * that the encrypt and decrypt steps provide a reciprocal mapping from
 * plaintext alphabet to ciphertext alphabet and vice versa.
 *
 *<P>An Alphabet is defined completely by its <TT>rangeId</TT> which
 * selects a subset of the Unicode character codes. Currently the Alphabet
 * class defines 6 different subsets. The subsets needn't be contiguous.
 * The subsets are selected in the constructor by passing a valid <I>rangeId</I> or
 * <I>range descriptor</I>. The following table shows the valid options:
 <TABLE>
  <TBODY>
  <TR>
    <TH>Range Id</TH>
    <TH>Range Descriptor</TH>
    <TH>ASCII Characters</TH></TR> 
  <TR>
    <TD>AlphabetFactory.ALPH_az</TD>
    <TD>az</TD>
    <TD>'a'..'z'</TD></TR> 
  <TR>
    <TD>AlphabetFactory.ALPH_AZ</TD>
    <TD>AZ</TD>
    <TD>'A'..'Z'</TD></TR> 
  <TR>
    <TD>AlphabetFactory.ALPH_azAZ</TD>
    <TD>azAZ</TD>
    <TD>'a'..'z''A'..'Z'</TD></TR>
  <TR>
    <TD>AlphabetFactory.ALPH_azAZ09</TD>
    <TD>azAZ09</TD>
    <TD>'a'..'z''A'..'Z''0'..'9'</TD></TR> 
  <TR>
    <TD>AlphabetFactory.ALPH_printable</TD>
    <TD>printable</TD>
    <TD>ASCII 32 .. ASCII 126</TD></TR>
  <TR>
    <TD>AlphabetFactory.ALPH_ascii</TD>
    <TD>ascii</TD>
    <TD>ASCII 0 .. ASCII 127</TD></TR> 
  </TBODY>
</TABLE>

*
* <P>For example, the following code segment would create a 
*  <A HREF="../engines/CaesarKey.html">CaesarKey</A> that would
* be defined for all printable ASCII characters.
* <PRE>
*         CaesarKey key = HistoricalKey.getInstance("Caesar");
          key.init("55/printable");
* </PRE>
*
* <P>In this case, because only one alphabet is described (printable), the 
* <tt>init()</tt> method will construct an alphabet consisting of all printable ASCII
* characters that will serve as both the plaintext alphabet and the ciphertext
* alphabet.  The string "55" in this case gives the Caesar keyword
* i.e., the shift used by the cipher engine.

* <P>As another example, the following code segment would create a 
*  <A href="../engines/PolySubstitutionKey.html">PolySubstitutionKey</A> that would
*  for a <A href="../engines/PolySubstitutionEngine.html">PolySubstitutionEngine</A> 
*  that maps lowercase characters to uppercase characters.
* <PRE>
*         PolySubstitutionKey key = HistoricalKey.getInstance("PolySubstitution");
          key.init("javahastwoas/az/AZ");
* </PRE>
*

* <P>In this case, the string "javahastwoas" will be used as the PolySubstitution keyword
* and the <tt>init()</tt> method will construct a plaintext alphabet consisting of
* the lowercase characters a..z and a ciphertext alphabet consisting of the uppercase
* characters A..Z. 

*<P>It is the responsibility of the cipher algorithm to enforce the constraints 
* described by the alphabet. This is usually done by ignoring invalid characters. 
* For example, here is the definition of the <TT>engineEncode()</TT> method from 
* <A href="../engines/CaesarEngine.html">CaesarEngine</A>: 
 <PRE>
    public String engineEncode(String s ) throws Exception {
        if (blocksize != 1)
            throw new Exception("Invalid blocksize for Caesar cipher " + blocksize);
        char ch = s.charAt(0);
        if (alphabet.isInAlphabet(ch)) {
           return "" + encodeShift(ch, shift);
        }
        else
           return s;
    }

 </PRE>

* <P>The Alphabet class also defines methods for adding a padding string to a short
* string in order to bring it up to block size before encrypting and a method for
* removing padding from the last block of a decrypted message.  A form of the modern
* <B>PKCS#5</B> padding scheme is used to choose the padding. Our scheme uses the
* ranking of a character in the alphabet to represent the number of characters added
* where the first letter of the alphabet represents 0.  For example, if 3 characters
* are needed for padding where the alphabet used is a..z then the string "ddd" is
* used. If a decrypted message has last block "endddd" the last 3 characters are
* deleted and the message ends with the letters "end" because d represents 3.
* This scheme requires the last block of the decrypted message to have some padding
* so if no padding is needed, an entire block of padding is added anyway.
* The <A href="BlockCipher.html">*BlockCipher</A> class contains the methods 
* to add and remove padding during encryption.
*
* <P>See also:
* <BLOCKQUOTE>
* <BR><A  href="BlockCipher.html">BlockCipher</A>
* <BR><A href="HistoricalKey.html">HistoricalKey</A>
* <BR><A href="../engines/CaesarEngine.html">CaesarEngine</A>
* <BR><A  href="../engines/CaesarKey.html">CaesarKey</A>
* <BR><A href="../engines/PolySubstitutionEngine.html">PolySubstitutionEngine</A>
* <BR><A  href="../engines/PolySubstitutionKey.html">PolySubstitutionKey</A>
* </BLOCKQUOTE>
*
* Last modified by REW on 01/20/2003 to make constructors public and to
* add an equals() method to override the equals() from the Object class.
*/
public class Alphabet {

/**
* Alphabets are represented internally as possibly disjoint subranges
* of the Unicode character set. The subranges are specified by the
* first and last characters in the range. For example:
*   ['A','Z','a','z'] specifies the upper and lowercase letters
*   ['\u3040', '\u309F', '\u30A0','\u30FF'] specifes the Hiragana and
*      Katakana character sets.
*/

private char[] ranges;

/**
* A unique identifier that represents the characters that
* make up the character set for a particular key.
*/

protected int rangeId;

/**
* The number of characters in the character set for this Alphabet.
*/

private int size;

/**
* The default constructor sets the range to 'a' to 'z'.
*/
public Alphabet() {
  this.ranges = new char[2];
  ranges[0] = 97;
  ranges[1] = 122;
  this.rangeId = AlphabetFactory.ALPH_az;
  size = 26;
} //Alphabet()  default constructor

/**
* Creates an alphabet using an array of char values.
* @param ranges an array of char values of even length thought of as
* a series of pairs of a beginning char value and an ending char value
* of a sequence of char values in the alphabet. For example, 
*   ['A','Z','a','z', '0','9'] would specifie the alphanumeric ascii 
* characters.
*/

public Alphabet(char[] ranges) throws Exception{
  this.ranges = ranges;
  size = 0;
  if (ranges.length % 2 == 1)
     throw new Exception("Invalid array of chars in an Alphabet constructor");
  for (int k = 0; k < ranges.length; k += 2){
      if ( ranges[k] <= ranges[k+1])
        size += (int)(ranges[k+1] - ranges[k] + 1);
      else
        throw new Exception("Invalid array of chars in an Alphabet constructor");
  } //for
}  //Alphabet(someRanges) constructor

/**
* Creates an alphabet using given an even-length String of character ranges.
* @param rangeStr a String of even length thought of as a series
* of pairs of a beginning char value and an ending char value
* of a sequence of char values in the alphabet. For example "AZaz09" would
* specify the alphanumeric ascii characters and "\u3040\u309F" would specify
* the Hiragana character set.
*/

protected Alphabet(String rangeStr) throws Exception{
  char[] theRanges  = new char[rangeStr.length()];
  for (int k = 0; k < theRanges.length; k++){
      theRanges[k] = rangeStr.charAt(k);
  } //for
  ranges = theRanges;
  if (ranges.length % 2 == 1)
     throw new Exception("Invalid array of chars in an Alphabet constructor");
  size = 0;
  for (int k = 0; k < ranges.length; k += 2){
      if ( ranges[k] <= ranges[k+1])
        size += (int)(ranges[k+1] - ranges[k] + 1);
      else
        throw new Exception("Invalid array of chars in an Alphabet constructor");
  } //for
}  //`Alphabet(stringDesc) constructor

/**
* This constructor creates an alphabet composed of characters form
*  one or more sets of Unicode blocks. 
* @param blocks  An array of Unicode blocks. For example,
*  [Character.Unicode.HIRAGAN, Character.Unicode.KATAKANA]. 
*/

public Alphabet(Character.UnicodeBlock[] blocks) throws Exception{
  int k = 0; 
  Character.UnicodeBlock aBlock; 
  StringBuffer sb = new StringBuffer();
  char ch = '\u0000';
  boolean chInAlph = false;
  while (ch < '\uFFFF') {
  //System.out.println("Start of while loop with ch = " + (int)(ch - 1));
       aBlock = Character.UnicodeBlock.of(ch);
       if (aBlock != null){
          for ( k = 0; k < blocks.length; k++) {
            if ( aBlock.equals(blocks[k])){
              chInAlph = true;
             // System.out.println("Subrange begun with ch = " + (int)ch);
            } //if - the second one
          }//for
      }//if
      if (!chInAlph)  //ch not in the alphabet, look at next char.
        ch++;
      else  { //We have found the beginning of a subrange.
          sb.append(ch);
          while (chInAlph && (ch < '\uFFFF')){
              ch++;
              chInAlph = false;
              aBlock = Character.UnicodeBlock.of(ch);
              if (aBlock != null){
                 for ( k = 0; k < blocks.length; k++)
                    if ( aBlock.equals(blocks[k]))
                       chInAlph = true;
              }//if
          }//while
          sb.append((char)((int)ch - 1)); //End of subrange found.
        //  System.out.println("Subrange ended with ch = " + (int)(ch - 1));
         // System.out.println("StringBuffer sb = " + sb.toString());
      }//else
  } //while
   //System.out.println("While loop ended with ch = " + (int)(ch - 1));
        // We have a String describing subranges so do as above
  String  rangeStr = sb.toString();
  char[] theRanges  = new char[rangeStr.length()];
  for ( k = 0; k < theRanges.length; k++){
      theRanges[k] = rangeStr.charAt(k);
  } //for
  ranges = theRanges;
 // System.out.println("ranges assigned of length = " + ranges.length);

  if (ranges.length % 2 == 1)
     throw new Exception("Invalid array of chars in an Alphabet constructor");
  size = 0;
  for ( k = 0; k < ranges.length; k += 2){
      if ( ranges[k] <= ranges[k+1])
        size += (int)(ranges[k+1] - ranges[k] + 1);
      else
        throw new Exception("Invalid array of chars in an Alphabet constructor");
  } //for
  //System.out.println("The number of Greek letters found = " + size);
} // Alphabet(unicodeBlocks) constructor

/**
* This constructor lets the user set the range using one of the
* static constants.
* @param rangeId an integer that specifies the character set
*/

public Alphabet(char[] ranges, int rangeId) throws Exception{
  this.ranges = ranges;
  if (rangeId < AlphabetFactory.ALPH_az || rangeId > AlphabetFactory.MAX_ALPH_NUM)
      throw new Exception("Invalid alphabet range descriptor: " + rangeId);
  this.rangeId = rangeId;
  size = 0;
  for (int k = 0; k < ranges.length; k += 2){
      if ( ranges[k] <= ranges[k+1])
        size += (int)(ranges[k+1] - ranges[k] + 1);
      else
        throw new Exception("Invalid array of chars in an Alphabet constructor");
  } //for
} //Alphabet(someRanges, intID) constructor

/**
* This constructor lets the user set the range using one of a
* set of descriptors.
* @param rangeDesc a String giving the range descriptor. It should be
* one of: az, AZ, azAZ, azAZ09, printable, ascii
*/

public Alphabet(char[] ranges, String rangeDesc) throws Exception {
  this.ranges = ranges;
  this.rangeId = AlphabetFactory.getRangeId(rangeDesc);
  size = 0;
  for (int k = 0; k < ranges.length; k += 2){
      if ( ranges[k] <= ranges[k+1])
        size += (int)(ranges[k+1] - ranges[k] + 1);
      else
        throw new Exception("Invalid array of chars in an Alphabet constructor");
  } //for
} //Alphabet(someRanges, stringDesc) constructor

/**
* returns the rangeId
*/

public int getRangeId() { return rangeId; }

/**
 * returns a descriptor that specifies the range of this alphabet
 */

public String getRangeDesc() throws Exception {
    return AlphabetFactory.getDesc(rangeId);
}

/**
 * returns the number of characters in this alphabet.
 */

public int getSize() { return size; }

/**
 * Returns the position of the parameter ch in the alphabet in the
 * range 0 to size - 1.  For example. For the alphabet a...z,
 * charToInt('c') returns 2 and charToInt('z') returns 25.
 * @param ch a char representing a character in the alphabet.
 * An Exception is thrown if ch is not in the alphabet.
 */

public int charToInt(char ch) throws Exception {
    if (!isInAlphabet(ch))
        throw new Exception("Character not in selected alphabet " + ch);
    int counter = 0;
    for (int k = 0; k < ranges.length; k += 2){
      if ((ch < ranges[k]) || (ch > ranges[k+1]))
          counter += (int)(ranges[k+1] - ranges[k] + 1);
      else
          return counter + (int)(ch - ranges[k]);
    }//for
    return -1; // Should never happen.
}

/**
* Returns the character in the alphabet which has the position
* of the parameter n.  For example, for the alphabet a...z,
* intToChar(3) returns 'd' and intToChar(25) returns 'z'.
* @param ch a char representing a character in the alphabet.
* An Exception is thrown if n is not in the range 0 to size - 1.
*/

public char intToChar(int n) throws Exception {
    if ((n < 0) || (n > size - 1))
        throw new Exception("There are not " + n + " characters in the alphabet.");
    int counter = n;
    for (int k = 0; k < ranges.length; k += 2){
      if (counter > (int)(ranges[k+1] - ranges[k]))
          counter -= (int)(ranges[k+1] - ranges[k] + 1);
      else
          return (char)( (int)ranges[k] + counter) ;
    }//for
    return ranges[0]; // Should never happen.
}


/**
* returns true iff its char parameter gives a character contained
* in this alphabet's characters set.
* @param ch a char giving the character to be tested for validity
*/

public boolean isInAlphabet(char ch) {
  for (int k = 0; k < ranges.length; k += 2)
    if (ch >= ranges[k] && ch <= ranges[k+1])
      return true;
  return false;
} //isInAlphabet()

/**
* Overrides the equals method of the Object class
* alph1.equals(alph2) returns true iff the arrays
* alph1.ranges and alph2.ranges have the same lengths
* and alph1.ranges[k] == alph2.ranges[k]for all k.
* @param alph2 an Object is assumed to be a member of Alphabet
*/

public boolean equals(Object alph2) {
  Alphabet alph = (Alphabet) alph2;
  if (ranges.length != alph.ranges.length)
      return false;
  for (int k = 0; k < ranges.length; k++)
      if (ranges[k] != alph.ranges[k])
          return false;
  return true;
}  //equals()

/**
* Returns a String of n characters of padding each equal to the  n-th
* character in the alphabet counting from 0.  This is called from
* BlockCipher.encrypt() to bring blocks up to the required blocksize.
* @param n an int indicates the number of characters needed.
* Example: If the character set is A..Z and 5 padding characters are
* needed, this method will return "FFFFF" since F is the sixth character.
*/

protected String getPadding(int n) throws Exception{
  char ch = intToChar(n);
  StringBuffer sb = new StringBuffer();
  for (int k = 0; k < n; k++)
      sb.append(ch);
  return sb.toString();
}

/**
* removes the padding characters from its string parameter assumed to have
* been decrypted so that the characters at the end were added while encrypting.
* MODIFIED 4/5/03 by RAM: To make this usable by cryptanalyzers it is
*  necessary to assume that the blocksize may be wrong.
* @param s a String giving a block of characters from which padding should be removed
* @param blocksize an int giving the blocksize for a particular cipher
*/

protected String removePadding(String s, int blocksize)throws Exception {
    int num = s.length()  - charToInt(s.charAt(s.length() - 1));
    //    System.out.println("removepadding: s= " + s + " blocksize= " + blocksize + " num = " + num);
    if (num >= 0)
	return  s.substring(0, num);
    else
	return s;
}


    /**
    * alph.strip(str) removes characters in the String str and returns the
    * resulting String.
    * @param str is a String which may have characters not in the alphabet.
    * @result is that String with chars not in the alphabet removed.
    */

    public String strip(String str){
        int len = str.length();
        StringBuffer sb = new StringBuffer(len);
        for (int k = 0; k < len; k++){
            if (isInAlphabet(str.charAt(k)))
                sb.append(str.charAt(k));
        }//for
        return sb.toString();
    } // strip

    /* removeDuplicateChars(str) removes any duplicate characters in a String
    */

    public static String removeDuplicateChars(String str){
        StringBuffer retStr = new StringBuffer();
        char ch; // for use in the loop
        boolean inBuff = false;
        for (int j = 0; j < str.length(); j++){
            ch = str.charAt(j);
            inBuff = false;
            for (int k = 0; k < retStr.length(); k++){
                if (retStr.charAt(k) == ch)
                    inBuff = true;
            } // for k
            if (!inBuff)
                retStr.append(ch);
        } // for j
        return retStr.toString();
    } //removeDuplicateChars()

    /**
    * Returns the string of characters which have the positions in the
    * alphabet of the array elements of arr.
    * @param arr is an array of int values corresponding to alphabet elements.
    * @result is a String of corresponding alphabet characters.
    */

    public String intArrayToString(int[] arr){
      try{
        int len = arr.length;
        StringBuffer sb = new StringBuffer(len);
        for (int k = 0; k < len; k++){
            sb.append(intToChar(arr[k]));
        }//for
        return sb.toString();
      } catch(Exception exc) {
        System.out.println(exc.getMessage());
        return "error";
      } // catch

    } // intArrayToString()

    /** createPermutation(str,permArr) writes a permutation to the
    * array permArr which extends the partial mapping described by the
    * string str. So k -> charToInt(str.charAt(k)) for 0 < k < str.length().
    * It is assumed that the number of the integers being permuted is
    * defined by the amount of memory assigned to permArr.
    * @param str is a string with characters in the alphabet.
    * @param permArr is an int array which has been assigned memory.
    */
    public void createPermutation(String str, int[] permArr){
      try{
        str = strip(str);
        str = removeDuplicateChars(str);

        // First copy the integers corresponding to keyStr characters
        for (int k = 0; k < str.length(); k++)
            permArr[k] = charToInt(str.charAt(k));

        //`Next add each integer that has not been used
        boolean inArray = false; // For  use in the loop below
        int  nextIndex = str.length();
        for (int m = 0; m < permArr.length; m++){
            inArray = false;
            for (int n = 0; n < nextIndex; n++){
                if (permArr[n] == m){
                        inArray = true;
                        break;
                } //  if
            } // for n
            if (!inArray)
                    permArr[nextIndex++] = m;
        } // for m

      } catch(Exception exc) {
        System.out.println(exc.getMessage());
      } // catch
    } //  createPermutation()

    /** permShift(permIn, shift) returns a permutation which is the composite
    * of permIn followed by shift, that is, k -> shift(permIn(k)). It is assumed
    * that permIn defines a valid permutation and that shift is a valid shift.
    * @param permIn is an int array which defines a valid permutation.
    * @param shift is an int shift between 0 and 1 less than the size of permIn.
    * @result is the int array defining the composite of permIn with shift.
    */
     public static int[] permShift(int[] permIn, int shift){
        if (permIn == null) return permIn;  // This error should not happen.
        int permSize = permIn.length;
        int[] permOut = new int[permSize];
        // Compute each new image in a loop
        for (int m = 0; m < permSize; m++){
            permOut[m] = (permIn[m] + shift)%permSize;
        } // for m
        return permOut;
     } //  permShift()

    /** shiftPerm(shift, permIn) returns a permutation which is the composite
    * of shift followed by permIn, that is, k -> permIn(shift(k)). It is assumed
    * that permIn defines a valid permutation and that shift is a valid shift.
    * @param shift is an int shift between 0 and 1 less than the size of permIn.
    * @param permIn is an int array which defines a valid permutation.
    * @result is an int array defining the composite of shift with permIn.
    */
     public static int[] shiftPerm(int shift, int[] permIn){
        if (permIn == null) return permIn;  // This error should not happen.
        int permSize = permIn.length;
        int[] permOut = new int[permSize];
        // Compute each new image in a loop
        for (int m = 0; m < permSize; m++){
            permOut[m] = permIn[(m + shift)%permSize];
        } // for m
        return permOut;
     } //  shiftPerm()

    /** normedShiftString(shStr) returns a String which defines shifts
    * like those defined by shStr only with the first shift subtracted from
    * all of the shifts. Thus for alphabet "az", normedShiftString("xyz")=="abc"
    * @param shStr is a String of chars in the alphabet.
    * @result is the String defining a normalized sequence of shifts in shStr.
    */
     public String normedShiftString(String shStr){
       try{
        if (shStr == null) return "error";  // This error should not happen.

        StringBuffer sb = new StringBuffer();
        int num; // For use in the loop.
        // Compute each new char in a loop.
        for (int m = 0; m < shStr.length(); m++){
            if (isInAlphabet(shStr.charAt(m))){
                num = charToInt(shStr.charAt(m)) - charToInt(shStr.charAt(0));
                num = (size + num)%size;
                sb.append(intToChar(num));
            } // if
        } // for m
        return sb.toString();

      } catch(Exception exc) {
        System.out.println(exc.getMessage());
        return "error";
      } // catch

     } //  normedShiftString()

    /** eAlbNormedPermString(permStr, shStr) returns a String which defines
    * the permutation of permStr followed by the first shift defined in shStr.
    * This permutation and the normed shift string of shStr are keys that give
    * precisely the same easy Alberti cipher as the keys permStr and shStr.
    * @param permStr is a String of chars in the alphabet defining a permutation.
    * @param shStr is a String of chars in the alphabet defining a sequence of shifts.
    * @result is the String defining the normalized easy Alberti permutation.
    */
     public String eAlbNormedPermString(String permStr, String shStr){
       try{
        int[] permIn = new int[size];
        createPermutation(permStr, permIn);
        int shift = charToInt(shStr.charAt(0));
        int[] permOut = permShift(permIn, shift);
        String strOut = intArrayToString(permOut);
        return strOut;

      } catch(Exception exc) {
        System.out.println(exc.getMessage());
        return "error";
      } // catch

     } //  eAlbNormedPermString()

    /** dAlbNormedPermString(shStr, permStr) returns a String which defines
    * the first shift defined in shStr followed by the  permutation of permStr.
    * This permutation and the normed shift string of shStr are keys that give
    * precisely the same difficult Alberti cipher as the keys shStr and permStr.
    * @param shStr is a String of chars in the alphabet defining a sequence of shifts.
    * @param permStr is a String of chars in the alphabet defining a permutation.
    * @result is the String defining the normalized difficult Alberti permutation.
    */
     public String dAlbNormedPermString(String shStr, String permStr){
       try{
        int[] permIn = new int[size];
        createPermutation(permStr, permIn);
        int shift = charToInt(shStr.charAt(0));
        int[] permOut = shiftPerm(shift, permIn);
        String strOut = intArrayToString(permOut);
        return strOut;
      } catch(Exception exc) {
        System.out.println(exc.getMessage());
        return "error";
      } // catch
     } //  dAlbNormedPermString()


//Temporary test method
    //public static void testAlphabet()throws Exception{
public static void main(String args[]) throws Exception{
     Alphabet myAlpha = AlphabetFactory.getInstance(AlphabetFactory.ALPH_az);
     System.out.println("size of az = " + myAlpha.getSize());
     System.out.println("charToInt('y') = " + myAlpha.charToInt('y'));
     System.out.println("intToChar(23) = " + myAlpha.intToChar(23)+ "\n");
    char [] myRanges ={'A', 'Z', 'a', 'z'};
     myAlpha = new Alphabet(myRanges);
     System.out.println("size of AZaz = " + myAlpha.getSize());
     System.out.println("charToInt('Y') = " + myAlpha.charToInt('Y'));
     System.out.println("charToInt('y') = " + myAlpha.charToInt('y'));
     System.out.println("intToChar(23) = " + myAlpha.intToChar(23));
     System.out.println("intToChar(51) = " + myAlpha.intToChar(51)+ "\n");

     String alphStr = "Az";
     myAlpha = new Alphabet(alphStr);
     System.out.println("size of Az = " + myAlpha.getSize());
     System.out.println("charToInt('Y') = " + myAlpha.charToInt('Y'));
     System.out.println("charToInt('y') = " + myAlpha.charToInt('y'));
     System.out.println("intToChar(23) = " + myAlpha.intToChar(23));
     System.out.println("intToChar(51) = " + myAlpha.intToChar(51));

     Character.UnicodeBlock[] theBlocks = { Character.UnicodeBlock.GREEK,Character.UnicodeBlock.BENGALI };
     myAlpha = new Alphabet(theBlocks);
     System.out.println("size of GREEK + BENGALI = " + myAlpha.getSize());
     System.out.println("intToChar(0) has int value = " + (int)myAlpha.intToChar(0));
     System.out.println("intToChar(size - 1) has int value = " + (int)myAlpha.intToChar(myAlpha.getSize()-1));
     if (Character.UnicodeBlock.of('\uFFFE') == null)
        System.out.println("\nCharacter.UnicodeBlock.of(\\uFFFE) == null");
     else
        System.out.println("Character.UnicodeBlock.of(\\uFFFE).toString()= " + Character.UnicodeBlock.of('\uFFFE').toString());
     if (Character.UnicodeBlock.of('\uFFFF') == null)
        System.out.println("\nCharacter.UnicodeBlock.of(\\uFFFF) == null");
     else
        System.out.println("Character.UnicodeBlock.of(\\uFFFF).toString()= " + Character.UnicodeBlock.of('\uFFFF').toString());

      //test the equals() method
    Alphabet alph1, alph2, alph3, alph4;
    char[] ranges1 = {'A','Z'};
    alph1 = new Alphabet(ranges1);
    alph2 = new Alphabet(ranges1);
    char[] ranges3 = {'a','z'};
    alph3 = new Alphabet(ranges3);
    char[] ranges4 = {'A','Z','a','z'};
    alph4 = new Alphabet(ranges4);
    if (alph1.equals(alph2)) System.out.println("alph1 equals alph2");
    else System.out.println("alph1 does not equal alph2");
    if (alph1.equals(alph3)) System.out.println("alph1 equals alph3");
    else System.out.println("alph1 does not equal alph3");
    if (alph1.equals(alph4)) System.out.println("alph1 equals alph4");
    else System.out.println("alph1 does not equal alph4");
} // main()

} //class Alphabet


