package hcrypto.cipher;

/**
 * This is a factory class for Alphabets. It defines static factory methods 
 * that create instances of the <A href="Alphabet.html">Alphabet</A> class.  
 * It also contains
 * static constants that describe the available Alphabets.
 * Appropriate variable names are given to unique non-negative int
 * values as ID values for the alphabets.
 * 
 * <P>This class can be modified to add new alphabets without having to modify
 * other classes that use the alphabets.
 */

import java.util.StringTokenizer;

public class AlphabetFactory {

    /**  
     *  The Alphabets are are numbered 0 through MAX_ALPH_NUM.
     *  This constant is increased as additional alphabets are defined.
     */
    public static final int MAX_ALPH_NUM = 8;
    public static final int MAX_CHARS = 128;

    // Each alphabet is assigned an integer and a descriptive variable name.
    public static final int ALPH_az         = 0;
    public static final int ALPH_AZ         = 1;
    public static final int ALPH_azAZ       = 2;
    public static final int ALPH_azAZ09     = 3;
    public static final int ALPH_printable  = 4;
    public static final int ALPH_ascii      = 5;
    public static final int ALPH_cryptogram = 6; // a-z + SPACE
    public static final int ALPH_unicode    = 7; // A set of Unicode blocks
    public static final int ALPH_hebrew    = 8; // A set of Unicode blocks

    /**
     * Returns an Alphabet instance, constructed from
     * an array subranges specified by their 16-bit Unicodes.
     * @param ranges -- an array of Unicode values specifying the
     *  disjoint subranges of the Alphabet.
     */
    public static Alphabet getInstance(char[] ranges) throws Exception{
        return (new Alphabet(ranges));
    }


    /**
     * Returns an Alphabet instance, constructed from an array of Unicode blocks
     * @param blocks -- an array of Unicode character blocks
     */
    public static Alphabet getInstance(Character.UnicodeBlock[] blocks) throws Exception {
        return new Alphabet(getRanges(blocks),ALPH_unicode);
    }


    /**
     * Returns an Alphabet instance, constructed from a range identifier
     * @param rangeId -- an int specifying one of the
     *  disjoint subranges of the Alphabet.
     */
    public static Alphabet getInstance(int rangeId) throws Exception {
        return new Alphabet(getRanges(rangeId), rangeId);
    }

    /* REWRITTEN BY RAM 5/24/02
     * Returns an Alphabet instance, constructed from the description given.
     * @param alphaDesc -- a String description of the alphabet
    public static Alphabet getInstance(String alphaDesc) throws Exception {
        Alphabet alpha;
        int rangeId = 0;
        try {
	    rangeId = getRangeId(alphaDesc);
	}
	catch (Exception e) {
            System.err.println(e.getMessage());
            e.printStackTrace();
	}
        return new Alphabet(getRanges(rangeId), rangeId);
    }
     */

    /**
     * This factory method returns an alphabet constructed from a string
     *  that describes the Unicode ranges that make up the alphabet.
     * @param alphaDesc a String of the form "az+AZ+UnicodeBlock.GREEK"
     *  where each of the substrings separated by '+' are descriptors
     *  for contiguous ranges of the Unicode character sequence.
     */
    public static Alphabet getInstance(String alphaDesc) throws Exception {
        return new Alphabet(getRanges(alphaDesc));
    }

    public static char getRangeLow(String alphDesc) throws Exception {
        if (alphDesc.equals("az"))
            return 'a';
        else if (alphDesc.equals("AZ"))
            return 'A';
        else if (alphDesc.equals("09"))
            return '0';
        else if (alphDesc.equals("printable"))
            return 32;
        else if (alphDesc.equals("ascii"))
            return 0;
        else if (alphDesc.equals("unicode"))
            return 0x0;
	else if (alphDesc.length() == 2)
	    return alphDesc.charAt(0);  // Must be a custome range e.g., ai OR kz
	else
            throw new Exception("AlphabetFactory.getRangeLow(): Illegal alphabet descriptor:" + alphDesc);
    }

    public static char getRangeHigh(String alphDesc) throws Exception {
        if (alphDesc.equals("az"))
            return 'z';
        else if (alphDesc.equals("AZ"))
            return 'Z';
        else if (alphDesc.equals("09"))
            return '9';
        else if (alphDesc.equals("printable"))
            return 126;
        else if (alphDesc.equals("ascii"))
            return 127;
        else if (alphDesc.equals("unicode"))
            return 0x7F;
	else if (alphDesc.length() == 2)
	    return alphDesc.charAt(1); // Must be a custom range
	else 
            throw new Exception("AlphabetFactory.getRangeHigh(): Illegal alphabet descriptor:" + alphDesc);
    }

    /**
     *  Returns a character array of starting and ending char values
     *   for each of the Unicode ranges included in this alphabet.
     *  @param alphaDesc a String of the form "az+AZ+Range.04+UnicodeBlock.GREEK"
     *   where "GREEK" must be a valid name of one of the 
     *   Character.UnicodeBlock ranges.
     */
    public static char[] getRanges(String alphaDesc) {
        StringTokenizer st = new StringTokenizer(alphaDesc, "+");
        char ranges[] = new char[st.countTokens() * 2];

	try {
            int k = 0;
            while (st.hasMoreTokens()) {
                String token = st.nextToken();
                if (token.startsWith("UnicodeBlock")) {
                    ranges[k++] = UnicodeRanges.getRangeLow(token);
                    ranges[k++] = UnicodeRanges.getRangeHigh(token);
                }
                else if (token.startsWith("Range")) { // e.g., Range.ai
		    String spec = token.substring(token.indexOf('.')+1);
                    ranges[k++] = getRangeLow(token.substring(token.indexOf('.')+1));
                    ranges[k++] = getRangeHigh(token.substring(token.indexOf('.')+1));
		}
                else { // Must be of the form "az" or "AZ" or "09"
                    ranges[k++] = getRangeLow(token);
                    ranges[k++] = getRangeHigh(token);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return ranges;
    }

    /**
     *  Returns a character array of starting and ending char values
     *   for each of the Unicode ranges contained in its parameter.
     *  @param blocks an array of UnicodeBlocks, such as
     *    [Character.UnicodeBlock.GREEK, Character.UnicodeBlock.KATAKANA]
     */
    public static char[] getRanges(Character.UnicodeBlock[] blocks) {
        char ranges[] = new char[blocks.length * 2];
        for (int k = 0; k < blocks.length; k++) {
            ranges[2*k] = UnicodeRanges.getRangeLow(blocks[k]);
            ranges[2*k+1] = UnicodeRanges.getRangeHigh(blocks[k]);
        }
        return ranges;
    }

    public static char[] getRanges(int id){
        char ranges[];
        switch(id) {
	case ALPH_az:
            ranges = new char[2];
            ranges[0] = 97; ranges[1] = 122;
            return ranges;
	case ALPH_AZ:
            ranges = new char[2];
            ranges[0] = 65; ranges[1] = 90;
            return ranges;
	case ALPH_azAZ:
            ranges = new char[4];
            ranges[0] = 65; ranges[1] = 90; ranges[2] = 97; ranges[3] = 122;
            return ranges;
	case ALPH_azAZ09:
            ranges = new char[6];
            ranges[0] = 48; ranges[1] = 57; ranges[2] = 65; ranges[3] = 90;
            ranges[4] = 97; ranges[5] = 122;
            return ranges;
	case ALPH_printable:
            ranges = new char[2];
            ranges[0] = 32; ranges[1] = 126;
            return ranges;
	case ALPH_ascii:
            ranges = new char[2];
            ranges[0] = 0; ranges[1] = 127;
            return ranges;
	case ALPH_cryptogram:
            ranges = new char[4];
            ranges[0] = 32; ranges[1] = 32; ranges[2] = 97; ranges[3]=122;
            return ranges;
	case ALPH_unicode:
            ranges = new char[2];
            ranges[0] = 0x0; ranges[1] = 0x7F;
            return ranges;
	case ALPH_hebrew:
            ranges = new char[2];
            ranges[0] = 0x590; ranges[1] = 0x5FF;
            return ranges;
	default:
            ranges = new char[2];
            ranges[0] = 97; ranges[1] = 122;
            return ranges;
	}
    }

    /**
     *  Returns a String description that specifies the range of the alphabet
     *  @param num is an int specifying the alphabet ID which should be between
     *  0 and MAX_ALPH_NUM.
     */
    public static String getDesc(int num){
        switch (num) {
        case ALPH_az:
            return "az";
        case ALPH_AZ:
            return "AZ";
        case ALPH_azAZ:
            return "azAZ";
        case ALPH_azAZ09:
            return "azAZ09";
        case ALPH_printable:
             return "printable";
        case ALPH_ascii:
             return "ascii";
        case ALPH_unicode:
             return "unicode block(s)";
        case ALPH_hebrew:
             return "hebrew";
        }
        return "invalid alphabet id number";
    }

    /**
     *  Returns the int ID for a given valid alphabet String descriptor and
     * returns -1  for an invalid descriptor.
     *  @param alphDesc a String specifying the descriptor, which should
     *   be one of: az, AZ, azAZ, azAZ09, printable, ascii
     */
    public static int getRangeId(String alphDesc) throws Exception {
        if (alphDesc.equals("az"))
            return ALPH_az;
        else if (alphDesc.equals("AZ"))
            return ALPH_AZ;
        else if (alphDesc.equals("azAZ"))
            return ALPH_azAZ;
        else if (alphDesc.equals("azAZ09"))
            return ALPH_azAZ09;
        else if (alphDesc.equals("printable"))
            return ALPH_printable;
        else if (alphDesc.equals("ascii"))
            return ALPH_ascii;
        else if (alphDesc.equals("cryptogram"))
            return ALPH_cryptogram;
        else if (alphDesc.equals("unicode"))
            return ALPH_unicode;
        else if (alphDesc.equals("hebrew"))
            return ALPH_hebrew;
        else 
            throw new Exception("AlphabetFactory.getRangeId() Invalid alphabet descriptor:" + alphDesc);
    }


    /**
     *  Returns true iff its String parameter represents a valid alphabet ID.
     *  @param spec a String specifying the descriptor, which should
     *   be one of: az, AZ, azAZ, azAZ09, printable, ascii
    */
    public static boolean isValidAlphabetSpec(String spec) {
        if (!spec.equals("az") && !spec.equals("AZ")
                && !spec.equals("azAZ09") && !spec.equals("printable")
	    && !spec.equals("ascii")  && !spec.equals("unicode") && !spec.equals("cryptogram")
	    && !spec.equals("hebrew"))
            return false;       
        return true;
    }
}
