/*
 * File: BlockCipher.java
 * @author: R. Morelli & R. Walde
 *
 * Copyright: This program is in the public domain. You can modify it as you 
 *  see fit as long as you properly acknowledge its original authors (Morelli
 *  and Walde). It would also be nice if you forwarded your changes to 
 *  <A HREF= "mailto:ralph.morelli@trincoll.edu">ralph.morelli@trincoll.edu</A> so 
 *  they can possibly be added to the "official" version.
 *
 * 6/4/2002: Added the engineEncryptFile(FileIn, FileOut, Encoding) and the
 *   engineDecryptFile(FileIn, FileOut, Encoding) methods to allow
 *   I/O of files that do not use the locale-dependent character encoding
 *   scheme.
 */
package hcrypto.cipher;

import java.io.*;

/**
 * This abstract class represents a basic block cipher. It provides 
 * implementations of methods to encrypt and decrypt files
 * and strings, for the entire range of cipher engines.
 * 
 * <P>Subclasses of this class must implement three methods:
 * <tt>engineInit(HistoricalKey)</tt>, <tt>engineEncode(String)</tt>
 * and <tt>engineDecode(String)</tt>. Of course, each particular cipher algorithm will
 * implement these methods in its own way. 
 *
 * <P>Traditional character ciphers, such as Caesar, Simple Substitution,
 * Vigenere, and Enigma, can be represented as special cases of a block
 * cipher by setting the <tt>blocksize</tt> to 1. This should be done
 * in the <tt>engineInit()</tt> method.
 *
 * <P>The <tt>engineEncrypt()</tt> and <tt>engineDecrypt()</tt> methods also 
 * handle <b>padding</b> of strings that are smaller than <b>blocksize</b>.
 * Details on the padding scheme are provided in <a href="Alphabet.html">Alphabet</a>.
 *
 * <P>The class contains algorithm-independent implementations of the following methods:
 * <tt>engineEncrypt(String)</tt>, <tt>engineDecrypt(String)</tt>,
 * <tt>engineEncryptFile(String, String)</tt>, and <tt>engineDecryptFile(String,String)</tt>.
 *
 */
public abstract class BlockCipher extends CipherEngine {

   /**
    * Stores the block size for the particular cipher. This variable
    *  should be initialized in the subclass.
    */
   protected int blocksize = 1;

   /**
    * Refers to the alphabet used by the plaintext of a particular cipher.
    * The cipher alphabet is defined as part of a <tt>HistoricalKey</tt>.
    * This variable should be initialized in <tt>engineInit()</tt>.
    */
   protected Alphabet alphabet;

    /**
    * Refers to the alphabet used for the ciphertext of a particular cipher.
    * For most ciphers, cipherAlphabet == alphabet.
    * The cipher alphabet is defined as part of a <tt>HistoricalKey</tt>.
    * This variable should be initialized in <tt>engineInit()</tt>.
    */
   protected Alphabet cipherAlphabet;

   /**
    * This abstract method must be implemented in the subclass to perform
    *  initializations required by a given cipher algorithm. Its parameter
    *  provides algorithm-specific information that is used to encrypt and
    *  decrypt. The <tt>key</tt> also provides a reference to the 
    *  <a href="Alphabet.html">Alphabet</a> that defines which plaintext
    *  characters are to be encrypted.
    * @param key an instance of a HistoricalKey subclass appropriate to
    *  the particular cipher algorithm
    */
   protected abstract void engineInit(HistoricalKey key) throws Exception;

   /**
    * This abstract method must be implemented in the subclass to perform
    *  basic encrypting step for a given cipher algorithm.  
    */
   protected abstract String engineEncode(String s) throws Exception;

   /**
    * This abstract method must be implemented in the subclass to perform
    *  basic decrypting step for a given cipher algorithm.  
    */
   protected abstract String engineDecode(String s) throws Exception;
   
   /**
    * This method encrypts its input file storing the encrypted text
    *  in the output file.   Note the use of InputStreamReader and OutputStreamReader.
    *  These classes perform automatic Unicode conversion to and from platform-dependent
    *  and locale-dependent encodings.  This is an important internationalization feature.
    * @param inFileName a String giving the name of the input file
    * @param outFileName a String giving the name of the output file
    */
   protected void engineEncryptFile(String inFileName, String outFileName) throws Exception {
       try {
           File inputFile = new File(inFileName);
           File outputFile = new File(outFileName);
	   InputStreamReader inStream = new InputStreamReader(new FileInputStream(inputFile));
           OutputStreamWriter outStream = new OutputStreamWriter(new FileOutputStream(outputFile));

           int length = (int)inputFile.length();            // Read the data
           char[] input = new char[length];
           inStream.read(input);
           inStream.close();

	   /************* TESTING THE ENCODING 
           System.out.println("File length = " + length);
           System.out.println("Encoding = " + inStream.getEncoding());
           for (int k = 0; k < input.length; k++)
               System.out.println((int)input[k]+ " ");
           ************/
            
           String stringIn = new String(input);            // Encrypt the data
           String stringOut = engineEncrypt(stringIn);

           outStream.write(stringOut,0,stringOut.length());  // Write the output
           outStream.close();
       } catch (FileNotFoundException e) {
           System.err.println("IOERROR: File not found either " + inFileName + " or " 
                                                                + outFileName);
           e.printStackTrace();
       } catch (IOException e) {
           System.err.println("IOERROR: " + e.getMessage());
           e.printStackTrace();
       }

   }


   /**
    * This method encrypts its input file storing the encrypted text
    *  in the output file.   Note the use of InputStreamReader and OutputStreamReader.
    *  These classes perform automatic Unicode conversion to and from platform-dependent
    *  and locale-dependent encodings.  This is an important internationalization feature.
    * @param inFileName a String giving the name of the input file
    * @param outFileName a String giving the name of the output file
    * @param encoding a String giving the name of the encoding scheme -- e.g., "ISO-2022-JP"
    */
   protected void engineEncryptFile(String inFileName, String outFileName, String encoding) throws Exception {
       try {
           File inputFile = new File(inFileName);
           File outputFile = new File(outFileName);
           InputStreamReader inStream = new InputStreamReader(new FileInputStream(inputFile), encoding);
           OutputStreamWriter outStream = new OutputStreamWriter(new FileOutputStream(outputFile), encoding);

           int length = (int)inputFile.length();            // Read the data
           char[] input = new char[length];
           inStream.read(input);
           inStream.close();

           System.out.println("File length = " + length);
           System.out.println("engineEncryptFile: Encoding = " + encoding);
           for (int k = 0; k < input.length; k++)
               System.out.println((int)input[k]+ " ");
            
           String stringIn = new String(input);            // Encrypt the data
           String stringOut = engineEncrypt(stringIn);

           outStream.write(stringOut,0,stringOut.length());  // Write the output
           outStream.close();
       } catch (FileNotFoundException e) {
           System.err.println("IOERROR: File not found either " + inFileName + " or " 
                                                                + outFileName);
           e.printStackTrace();
       } catch (IOException e) {
           System.err.println("IOERROR: " + e.getMessage());
           e.printStackTrace();
       }

   }



   /**
    * This method decrypts its input file storing the decrypted data
    *  in the output file. 
    * @param inFileName a String giving the name of the input file
    * @param outFileName a String giving the name of the output file
    */
   protected void engineDecryptFile(String inFileName, String outFileName) throws Exception {
       try {
           File inputFile = new File(inFileName);
           File outputFile = new File(outFileName);
           InputStreamReader inStream = new InputStreamReader(new FileInputStream(inputFile));
           OutputStreamWriter outStream = new OutputStreamWriter(new FileOutputStream(outputFile));

           int length = (int)inputFile.length();            // Read the data
           char[] input = new char[length];
           inStream.read(input);
           inStream.close();

           String stringIn = new String(input);      // Decrypt the data
           String stringOut = engineDecrypt(stringIn);

           outStream.write(stringOut,0,stringOut.length());  // Write the output
           outStream.close();
       } catch (FileNotFoundException e) {
           System.err.println("IOERROR: File not found either " + inFileName + " or " 
                                                                + outFileName);
           e.printStackTrace();
       } catch (IOException e) {
           System.err.println("IOERROR: " + e.getMessage());
           e.printStackTrace();
       }
   }


   /**
    * This method decrypts its input file storing the decrypted data
    *  in the output file. 
    * @param inFileName a String giving the name of the input file
    * @param outFileName a String giving the name of the output file
    * @param encoding a String giving the name of the encoding scheme -- e.g., "ISO-2022-JP"
    */
   protected void engineDecryptFile(String inFileName, String outFileName, String encoding) throws Exception {
       try {
           File inputFile = new File(inFileName);
           File outputFile = new File(outFileName);
           InputStreamReader inStream = new InputStreamReader(new FileInputStream(inputFile), encoding);
           OutputStreamWriter outStream = new OutputStreamWriter(new FileOutputStream(outputFile), encoding);

           int length = (int)inputFile.length();            // Read the data
           char[] input = new char[length];
           inStream.read(input);
           inStream.close();

           String stringIn = new String(input);      // Decrypt the data
           String stringOut = engineDecrypt(stringIn);

           outStream.write(stringOut,0,stringOut.length());  // Write the output
           outStream.close();
       } catch (FileNotFoundException e) {
           System.err.println("IOERROR: File not found either " + inFileName + " or " 
                                                                + outFileName);
           e.printStackTrace();
       } catch (IOException e) {
           System.err.println("IOERROR: " + e.getMessage());
           e.printStackTrace();
       }
   }

   /**
    * This method encrypts a String, returning a String. It invokes
    *  the abstract method <tt>encode(String)</tt> which is implemented
    *  in an algorithm-specific way in the subclass.
    * @param s the String to be encrypted
    */
   protected String engineEncrypt (String s) throws Exception {
       StringBuffer result = new StringBuffer();
       int sLength = s.length();
       int ptr = 0;
       while (ptr + blocksize <= sLength) {
           result.append( engineEncode(s.substring(ptr, ptr + blocksize))); 
           ptr += blocksize;
       }

       if (ptr < sLength) {                             // Add some padding 
           String tail = s.substring(ptr);

          String padString =    alphabet.getPadding(blocksize - tail.length());
          String encString =    engineEncode(tail + padString);
          result.append( encString);
       }
       else
          if (blocksize > 1)  // We always add padding for blockSize > 1.
               result.append( engineEncode(alphabet.getPadding(blocksize)) );
       return result.toString();
   }		

   /**
    * This method decrypts a String, returning a String. It invokes
    *  the abstract method <tt>decode(String)</tt> which is implemented
    *  in an algorithm-specific way in the subclass.
    * @param s the String to be encrypted
    */
   protected String engineDecrypt (String s) throws Exception {
       StringBuffer result = new StringBuffer();
       int sLength = s.length();
       int ptr = 0;
       while (ptr + blocksize < sLength) {
           result.append( engineDecode(s.substring(ptr, ptr + blocksize))); 
           ptr += blocksize;
       }
       String lastBlock = s.substring(ptr);
       lastBlock = engineDecode(lastBlock);

       if (blocksize > 1) // Remove padding for blockSize > 1
          lastBlock = alphabet.removePadding(lastBlock, blocksize);
       if (lastBlock.length() > 0)
           result.append( lastBlock);   // Last block

       return result.toString();
   }

 /**
    * This method shifts a character in the alphabet ahead the specified
    * number of characters in the alphabet. If the character is not in
    * the alphabet, the character itself is returned. Characters that are
    * shifted beyond the end of the alphabet are wrapped around past the
    * beginning of the alphabet.
    * @param ch the character to be shifted.
    * @param k the size (number of characters) of the shift.
    */
   protected char encodeShift (char ch, int k) throws Exception{
       if (!alphabet.isInAlphabet(ch)) 
           return ch;
       while (k < 0) 
           k += alphabet.getSize();
       return cipherAlphabet.intToChar((alphabet.charToInt(ch) + k) % alphabet.getSize());
   } // encodeShift

    /**
    * This method shifts a character in the alphabet ahead the specified
    * number of characters in the alphabet. If the character is not in
    * the alphabet, the character itself is returned. Characters that are
    * shifted beyond the end of the alphabet are wrapped around past the
    * beginning of the alphabet.
    * @param ch the character to be shifted.
    * @param k the size (number of characters) of the shift.
    */
   protected char decodeShift (char ch, int k) throws Exception{
       if (!cipherAlphabet.isInAlphabet(ch)) 
           return ch;
       while (k < 0) 
           k += cipherAlphabet.getSize();
       return alphabet.intToChar((cipherAlphabet.charToInt(ch) + k) % cipherAlphabet.getSize());
   } // decodeShift
}//end BlockCipher
