/*
 * File: Cipher.java
 * @author: R. Morelli and R. Walde
 *
 * Copyright: This program is in the public domain. You can modify it as you 
 *  see fit as long as you properly acknowledge its original authors (Morelli
 *  and Walde). It would also be nice if you forwarded your changes to 
 *  <A HREF= "mailto:ralph.morelli@trincoll.edu">ralph.morelli@trincoll.edu</A> so 
 *  they can possibly be added to the "official" version.
 *
 * 6/4/2002: Added the encryptFile(FileIn, FileOut, Encoding) and the
 *   decryptFile(FileIn, FileOut, Encoding) methods to allow
 *   I/O of files that do not use the locale-dependent character encoding
 *   scheme.
 */

package hcrypto.cipher;

import hcrypto.provider.*;

/**
 * This class provides the functionality for a wide range of historical ciphers
 * that can be used for encryption and decryption. 
 *
 * <P>To create a Cipher object, the application must call <tt>Cipher.getInstance()</tt>
 * passing it a String that gives the name of the cipher algorithm. For example, here
 * is how one would create a Caesar cipher. 
 *
 * <pre>
 *      Cipher cipher = Cipher.getInstance("Caesar");
 *      HistoricalKey key = HistoricalKey.getInstance("Caesar", cipher.getProvider());
 *      key.init("55/printable");
 *      cipher.init(key);
 *      String c1encrypt = cipher.encrypt(secret);
 * </pre>
 *     
 * The <tt>Cipher.getInstance()</tt> method searches a list of providers for an
 * implementation of Caesar cipher and will throw a "No such algorithm"
 * exception if none is found. Details on creating an algorithm-dependent
 * key for the cipher are described in the documentation of 
 * <a href="HistoricalKey.html">HistoricalKey</a> class.
 *
 * <P>See also:
 *  <BLOCKQUOTE> 
 *     <BR> <a href="CipherEngine.html">CipherEngine</a>
 *     <BR> <a href="HistoricalKey.html">HistoricalKey</a>
 *     <BR> <a href="../provider/Provider.html">Provider</a>
 *  </BLOCKQUOTE> 
 *
 */

public class Cipher {

   /**
    * provides a reference to the class that implements the 
    *  cipher algorithm
    */
   protected CipherEngine engine;    

   /**
    *  the name of the provider of this algorithm
    */
   protected String provider;  // Name of the provider

   /**
    *  the name of this algorithm
    */

   protected String algorithm; // Name of cipher algorithm

   /**
    *  the constructor is only used internally to create a Cipher object.
    *  It is passed an instance of the engine that provides an implementation 
    *  of the desired cipher algorithm. Applications should use one of the
    *  getInstance() factory methods to create a Cipher.
    */
   protected Cipher(CipherEngine engine) {
       this.engine = engine;
   }


   /**
    * searches for the provider named in the second parameter
    *  for an implementation of the algorithm named in its first parameter.
    * @param algorithm, a String giving the algorithm name
    * @param provider, a String giving the provider name
    */
   public static final Cipher getInstance(String algorithm, String provider) {
//       System.out.println("Getting instance " + algorithm + " " + provider);
       Provider p = new Provider();
       Cipher cipher = null;
       try {
           String className = p.getCipherEngine(algorithm, provider);
//           System.out.println("Using ENGINE: " + className);
           cipher = new Cipher( (CipherEngine) Class.forName(className).newInstance() );           
           cipher.algorithm = algorithm;
           cipher.provider = p.getName();
//           System.out.println("Using Provider: " + p.getName());
       } catch (Exception e) {
           System.err.println("Exception getting cipher instance: " + e.getMessage());
           e.printStackTrace();
       }
       return cipher;
   }

   /**
    * searches for a provider that provides an
    *  implementation of the algorithm named in its parameter,
    *  throwing a "no such algorithm" exception if none is found.
    * @param algorithm, a String giving the algorithm name
    */
   public static final Cipher getInstance(String algorithm) {
       Provider p = new Provider();
       Cipher cipher = null;
       try {
           String className = p.getCipherEngine(algorithm);
	   //           System.out.println("Using ENGINE: " + className);
           cipher = new Cipher( (CipherEngine) Class.forName(className).newInstance() );           
           cipher.algorithm = algorithm;
           cipher.provider = p.getName();
	   //           System.out.println("Using Provider: " + p.getName());
       } catch (Exception e) {
           System.err.println("Exception getting cipher instance: " + e.getMessage());
           e.printStackTrace();
       }
       return cipher;
   }

   /**
    * initializes the particular encrypting engine by calling
    *  the encapsulated CipherEngine and passing it a key of the appropriate 
    *  type. 
    * @param key, an instance of a HistoricalKey
    */
   public final void init(HistoricalKey key) throws Exception {
      engine.engineInit(key);
   }


   /**
    * encrypts a plaintext input file into an output file
    *  by simply calling the engineEncryptFile() method implemented in the ENGINE.
    * @param inFile, the name of the input plaintext file.
    * @param outFile, the name of the output encrypted file.
    */
   public final void encryptFile(String inFileName, String outFileName) throws Exception {
       engine.engineEncryptFile(inFileName, outFileName);
   }

   /**
    * encrypts a plaintext input file into an output file
    *  by simply calling the engineEncryptFile() method implemented in the ENGINE.
    *  Use this version if you want to encrypt a file that is NOT encoded with
    *  locale-dependent character encoding scheme.
    * @param inFile, the name of the input plaintext file.
    * @param outFile, the name of the output encrypted file.
    * @param encoding, the name of the encoding scheme -- "ISO-2022-JP"
    */
   public final void encryptFile(String inFileName, String outFileName, String encoding) throws Exception {
       engine.engineEncryptFile(inFileName, outFileName, encoding);
   }


   /**
    * decrypts a ciphertext input file into an output file
    *  by simply calling the engineDecryptFile() method implemented in the ENGINE.
    * @param inFile, the name of the input ciphertext file.
    * @param outFile, the name of the output plaintext file.
    */
   public final void decryptFile(String inFileName, String outFileName) throws Exception {
       engine.engineDecryptFile(inFileName, outFileName);
   }

   /**
    * decrypts a ciphertext input file into an output file
    *  by simply calling the engineDecryptFile() method implemented in the ENGINE.
    *  Use this version if you want to decrypt a file that is NOT encoded with
    *  locale-dependent character encoding scheme.
    * @param inFile, the name of the input ciphertext file.
    * @param outFile, the name of the output plaintext file.
    * @param encoding, the name of the encoding scheme  -- e.g.,  "ISO-2022-JP"
    */
   public final void decryptFile(String inFileName, String outFileName, String encoding) throws Exception {
       engine.engineDecryptFile(inFileName, outFileName, encoding);
   }

   /**
    * encrypts its String parameter returning the encrypted string.
    *  It simply calls the engineEncrypt() method implemented in the ENGINE.
    * @param s, a String to be encrypted
    * @result a String giving the encryption of s.
    */
   public String encrypt (String s ) throws Exception {
       return engine.engineEncrypt(s);
   }		

   /**
    * decrypts its String parameter returning the plaintext string.
    *  It simply calls the engineDecrypt() method implemented in the ENGINE.
    * @param s, a String to be decrypted
    * @result a String giving the decryption of s.
    */
   public final String decrypt (String s ) throws Exception {
       return engine.engineDecrypt(s);
   }

   /**
    *  returns a string giving the name of the cipher algorithm.
    */
   public final String getAlgorithm() {
       return this.algorithm;
   }

   /**
    *  returns a string giving the name of the provider that
    *   implements the cipher algorithm.
    */
   public final String getProvider() {
       return this.provider;
   }

   /**
    *  returns a string giving the alphabet range options for a particular
    *  cipher. The alphabet rangeOptions string should be set in the
    *  CipherEngine constructor.
    */
   public final String getAlphabetRangeOptions() {
       return engine.getAlphabetRangeOptions();
   }


} 

