/*
 * File: CipherEngine.java
 * @author: R. Morelli and R. Walde
 *
 * Copyright: This program is in the public domain. You can modify it as you 
 *  see fit as long as you properly acknowledge its original authors (Morelli
 *  and Walde). It would also be nice if you forwarded your changes to 
 *  <A HREF= "mailto:ralph.morelli@trincoll.edu">ralph.morelli@trincoll.edu</A> so 
 *  they can possibly be added to the "official" version.
 *
 * 6/4/2002: Added the engineEncryptFile(FileIn, FileOut, Encoding) and the
 *   engineDecryptFile(FileIn, FileOut, Encoding) methods to allow
 *   I/O of files that do not use the locale-dependent character encoding
 *   scheme.
 */

package hcrypto.cipher;

/**
 * The <i>CipherEngine</i> class defines the <i>interface</i> 
 * for the <i>Cipher</i> class. All of the methods in this class must be
 * implemented by a provider who wishes to provide an 
 * implementation of a particular cipher algorithm.
 * <P>
 * The Cipher class encapsulates an instance of this class. To
 * create a Cipher instance, an application would use the
 * <tt>Cipher.getInstance(String algorithm)</tt> <b>factory method</b>, 
 * where <i>algorithm</i> provides the name of the cipher algorithm 
 * -- e.g., "Caesar",  * "Playfair", and so on.
 * <P>
 * For example, here's sample code for instantiating and using a
 * Caesar cipher:
 * <pre>
 *      Cipher cipher = Cipher.getInstance("Caesar");
 *      HistoricalKey key = HistoricalKey.getInstance("Caesar", cipher.getProvider());
 *      key.init("55/printable");
 *      cipher.init(key);
 *      String c1encrypt = cipher.encrypt(secret);
 * </pre>
 *
 * The <tt>Cipher.getInstance()</tt> method searches the provider list for an
 * implementation of Caesar cipher and will throw a "No such algorithm"
 * exception if none is found. Details on creating an algorithm-dependent
 * key for the cipher are described in the documentation of 
 * <a href="HistoricalKey.html">HistoricalKey</a> class.
 *
 * <P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="Cipher.html">Cipher</a>
 *      <BR><a href="HistoricalKey.html">HistoricalKey</a>
 *  </BLOCKQUOTE> 
 */

public abstract class CipherEngine {

    /**
     * A string of 0s and 1s that describes the options for possible alphabet
     *  ranges. Currently there are six RANGE options (see above) and
     *  they are to be encoded in the order shown above so that RANGE_az,
     *  if that is an option for a particular alphabet, would be encoded
     *  by a '1' in the first position of the string. For example, a
     *  rangeOptions string of "111100" would allow the cipher to have
     *  any of the ranges RANGE_az, RANGE_AZ, RANGE_azAZ or RANGE_azAZ09,
     *  but not RANGE_printable or RANGE_ascii. This variable should be
     *  set in the cipher constructor. 
     */
    protected String alphabetRangeOptions;

   /**
    * engineInit() initializes the particular encrypting engine based on
    *  a key of the appropriate type. This method is implemented in 
    *  algorithm-specific ways in the subclass.
    * @param key, an instance of a HistoricalKey
    */
    protected abstract void engineInit(HistoricalKey key) throws Exception;

   /**
    * engineEncrypt() encrypts its String parameter returning the encrypted string.
    * When implemented, this method should call an engineEncode(String):String method 
    * that is implemented in the class that implements the specific cipher algorithm.
    * @param s, a String to be encrypted
    * @result a String giving the encryption of s.
    */
    protected abstract String engineEncrypt(String s) throws Exception;

   /**
    * engineDecrypt() encrypts its String parameter returning the encrypted string.
    * When implemented, this method should call an engineDecode(String):String method 
    * that is implemented in the class that implements the specific cipher algorithm.
    * @param s, a String to be encrypted
    * @result a String giving the encryption of s.
    */
    protected abstract String engineDecrypt(String s) throws Exception;

   /**
    * engineEncryptFile() encrypts a plaintext input file into an output file.
    * @param inFile, the name of the input plaintext file.
    * @param outFile, the name of the output encrypted file.
    */
    protected abstract void engineEncryptFile(String inFile, String outFile) throws Exception;

   /**
    * engineEncryptFile() encrypts a plaintext input file into an output file.
    * @param inFile, the name of the input plaintext file.
    * @param outFile, the name of the output encrypted file.
    * @param encoding, the name of the encoding scheme -- "ISO8859" or "ISO-2022-JP"
    */
    protected abstract void engineEncryptFile(String inFile, String outFile, String encoding) throws Exception;

   /**
    * engineDecryptFile() decrypts a ciphertext input file into an output file.
    * @param inFile, the name of the input ciphertext file.
    * @param outFile, the name of the output plaintext file.
    */
    protected abstract void engineDecryptFile(String inFile, String outFile) throws Exception;

   /**
    * engineDecryptFile() decrypts a ciphertext input file into an output file.
    * @param inFile, the name of the input ciphertext file.
    * @param outFile, the name of the output plaintext file.
    * @param encoding, the name of the encoding scheme -- "ISO8859" or "ISO-2022-JP"
    */
    protected abstract void engineDecryptFile(String inFile, String outFile, String encoding) throws Exception;

    public String getAlphabetRangeOptions() {
	return alphabetRangeOptions;
    }

}
