package hcrypto.engines;

import hcrypto.cipher.*;

/**
 * This class implements an Affine cipher algorithm (a special case of a Substitution Cipher that 
 * encrypts and decrypts a character based on two Affine functions) for any of three alphabet 
 * ranges including az, AZ, azAZ.
 * The AffineKey is composd of two integers: <i>A_E_Key</i> and <i>B_E_Key</i> which represent 
 * "encyption key A" and "encryption key B" respectfully.  Two additional keys are created in this class
 * for decrypting: <i>A_D_Key</i> and <i>B_D_Key</i>.
 * <p>A_D_Key is created by computing the multiplicative inverse (mod 26) of the A_E_Key with a private function.
 * <p>B_D_Key is created by subtracting the B_E_Key from 26, modulus 26. 
 * <pre>
 *     B_D_Key = (26 - B_E_Key) % 26 )
 * </pre>
 * 
 * <p>engineEncode(String s) performs the following function:
 * <pre>
        encrypt(x) = ax + b mod 26
   </pre>
   <p>engineDecode(String s) performs the following function:
   <pre>
        decrypt(y) = inverse of (a) * (y - b) mod 26
   </pre>   
 
 <P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="AffineKey.html">AffineKey</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 */
 
public class AffineEngine extends BlockCipher {  //DOCUMENTED BY GREGG--DONE
 
    private AffineKey key;

       // These two keys can be set through a GUI.                                           
    private int A_E_Key;   //encrypt : key needs to be an odd integer from 1-25, excluding 13.
    private int B_E_Key;  //encrypt : key needs to be an integer between 0-26.

      // These two keys are derived from above keys and are used for decrypting.
    private int A_D_Key;     //decrypt : key is calculated by a function below.
    private int B_D_Key;     //decrypt : key is calculated by a function below.

    /**
    * Creates an AffineEngine and sets the <i>alphabetRangeOptions</i> instance variable
    * to "110000", which translates to the "az+AZ"alphabet option for both the
    * plaintext and ciphertext alphabets.
    */
    
    public AffineEngine() {
//        alphabetRangeOptions = "111000";  // Affine just allows 3 options
        alphabetRangeOptions = "110000/110000";  // Affine just allows az+AZ
    }

    
   /**
    * Initializes the AffineEngine with the specified hKey.
    * @param hKey a AffineKey.
    */
    
    protected void engineInit(HistoricalKey hKey) throws Exception {
        if ( !(hKey instanceof AffineKey))
            throw new Exception("InvalidKey: Affine requires AffineKey");
        AffineKey key = (AffineKey) hKey;
        blocksize = key.getBlocksize();
        alphabet = key.getAlphabet();
        cipherAlphabet = key.getCTAlphabet();
        A_E_Key = key.getKeyA();
        B_E_Key = key.getKeyB();
        int aSize = alphabet.getSize();
        A_D_Key =  getA_D_Key(aSize);
        B_D_Key =  getB_D_Key(aSize);
    }

    
    
   /**
    * Returns an encoded String for the specified String.  Characters which 
    *  are not part of the chosen alphabet set are ignored.
    * @param s the String to be encrypted
    */
    
    public String engineEncode(String s) throws Exception {
        if (blocksize != 1)
            throw new Exception("Invalid blocksize for Affine cipher " + blocksize);
        char ch = s.charAt(0);
        char cipherChar = ch;
    
        if (alphabet.isInAlphabet(ch)) {
            cipherChar =cipherAlphabet.intToChar( (A_E_Key * alphabet.charToInt(ch) + B_E_Key) 
                                                   % alphabet.getSize() );
//            System.out.println(ch + " " + low + " " + (int)cipherChar);
        }
        return "" + cipherChar;
    }

   /**
    * Returns an encoded String for the specified String.  
    *   Characters which are not part of the chosen alphabet set are ignored.
    * @param s the String to be decrypted
    */
   
    public String engineDecode(String s) throws Exception {
        if (blocksize != 1)
            throw new Exception("Invalid blocksize for Affine cipher " + blocksize);
        char ch = s.charAt(0);
        char cipherChar = ch;
   
        if (cipherAlphabet.isInAlphabet(ch)) {
           cipherChar =alphabet.intToChar( (A_D_Key * cipherAlphabet.charToInt(ch) + B_D_Key) 
                                                      % cipherAlphabet.getSize() );
//            System.out.println(ch + " " + low + " " + (int)cipherChar);
        }
        return "" + cipherChar;
    }

    /**
     * Computes the multiplicative inverse (mod aSize) of A_E_Key
     */
    private int getA_D_Key(int aSize)throws Exception {
        for (int k = 1; k < aSize; k++) {
            if((A_E_Key*k) % aSize == 1)       
                return k;
        } // for
       throw new Exception("InvalidKey for Affine Cipher: Invalid A_E_Key");
    }

    /**
     * Computes the value of B_D_Key for decrypting.
     */
    private int getB_D_Key( int aSize) {    
        return ( (aSize  - ((A_D_Key *B_E_Key) % aSize) % aSize ));
    }

}  // end class AffineEngine
