package hcrypto.engines;

import java.util.StringTokenizer;
import hcrypto.cipher.*;

/* File: AffineKey.java   Last modified by REW on 2/4/2002.
Written by RAM and GM during the summer of 2000.
*/

/**
 * This class represents a key for an Affine cipher. The key
 * is entirely represented by two integers, <i>a</i> and <i>b</i>, which
 * serve as variables in the Affine functions:  <b>encrypt(x) = ax + b mod N</b>
 * and <b>decrypt(y) = inverse_of(a) * (y - b) mod N</b> where N = size of the
 * alphabet.
 *
 * <p>This implementation of the Affine cipher is restricted to condition that
 * the plaintext alphabet and ciphertext alphabet must have the same size.
 *<P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="AffineEngine.html">AffineEngine</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 */

public class AffineKey extends HistoricalKey { 

   /**
    * Serves as one of two variables for the Affine function. Value must be an
    * an integer 1 and N - 1 which has a multiplicative inverse mod N, that is,
    * there must be an integer C between 1 and N-1 with keyA*C % N = 1.  As
    * above, N = size of the alphabet.
    */
    
    private int keyA;
    
   /**
    * Serves as one of two variables for the Affine function.  Value can be any integer from 0
    * to N -1.
    */
    
    private int keyB;

   /**
    * A description for an Affine Key.
    */
    
    public final static String DEFAULT_KEY_DESCRIPTOR_PROMPT_STRING 
         = ("an odd integer between 0-25(with the exception of 13), and any integer from 0-25.  Example: 3,19");
        
   /**
    * Default key values.
    */
     
    public final static String DEFAULT_KEYWORD_STRING = ("3,19");
    
    // keySpec will be something like "5,21/az"
    // and will be mapped into keyword="5,21", alphabet="az"   
    
    /** 
    * Creates an AffineKey with the specified keyspec.
    * @param keyspec takes the form "integer,integer/alphabet", i.e.
    * "5,21/az" for example, which would be mapped into the instance variables
    * <i>keyword</i> as a String with the value "5,21", and <i>alphabet</i> as
    * a String with the value "az".
    */   

    public void init(String keyspec) throws Exception {
        initKey(keyspec, false);         // Inherited from superclass
        this.blocksize = 1;
        this.keyA = Integer.parseInt(keyword.substring(0,keyword.indexOf(',')));
        this.keyB = Integer.parseInt(keyword.substring(keyword.indexOf(',') + 1));
    }

   /**
    * Initializes the AffineKey given a keyword and one or more alphabets.
    * @param keyword -- a string version of the keyword
    * @param alpha1, alpha2 -- Alphabets for plain/ciphertext respectively. May
    *  be identical.
    */ 
    public void init(String keyword, Alphabet alpha1, Alphabet alpha2) throws Exception { 
	if (alpha1 == null || alpha2 == null)
	    throw new Exception("AffineKey.init(): Null alphabet reference passed");
        this.keyword = keyword;
        this.blocksize = 1;
        this.keyA = Integer.parseInt(keyword.substring(0,keyword.indexOf(',')));
        this.keyB = Integer.parseInt(keyword.substring(keyword.indexOf(',') + 1));
        super.initKey(keyword, alpha1, alpha2);
    }

   /**
    * Returns the algorithm name "Affine".
    */
    
    public String getAlgorithm() { 
        return "Affine"; 
    }
    
    
   /**
    * Returns the keyA value for the Affine function.
    */
     
    public int getKeyA() { 
        return keyA; 
    }
    
    
   /**
    * Returns the keyB value for the Affine function.
    */
    
    public int getKeyB() { 
        return keyB; 
    }
}
