package hcrypto.engines;
//package plugins;

import hcrypto.cipher.*;

import java.util.StringTokenizer;

/**
 * This class represents a key for a Alberti cipher. The key
 * is entirely represented by a keyword String composed of characters from the chosen alphabet. 
 * <p>For this implementation of Alberti cipher any of the alphabet ranges
 * can be used.
 * <P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="AlbertiEngine.html">AlbertiEngine</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 */

public class AlbertiKey extends HistoricalKey 
{
    /**
     * A keyword description.
     */  
    public final static String DEFAULT_KEY_DESCRIPTOR_PROMPT_STRING = ("permutation,shift");
        
   
    /**
     * A default keyword.
     */ 
    public final static String DEFAULT_KEYWORD_STRING = ("permutation,shift");

    /**
     * Strings to hold the two keywords, used to make the cipheralphabet and period of rotation.
     */
	
    private String shiftkey, permutekey;

    /**
     * Initializes the AlbertiKey with the specified keyspec.
     * @param keySpec takes the form "keyword/alphabet", where
     * "76TrombonesLED,THEBIGPARADE/az+AZ+09" for example,
     * keyword consists of 'permutationkey,shiftkey'
     * and <i>alphabet</i> as a String with a value such as "az+AZ+09".
     */ 
    public void init(String keyspec) throws Exception { 
	super.initKey(keyspec, false);       // Extracts keyword (permutekey,shiftkey) false = don't remove duplicates 
                                            // initKey() calls initKeyArrays which extracts permutekey and shiftkey
	if (!isValidKeyword(shiftkey))
	    throw new Exception("Alberti shiftkey contains invalid characters for selected alphabet");
	if (!isValidKeyword(permutekey))
	    throw new Exception("Alberti permutekey contains invalid characters for selected alphabet");
	this.blocksize = 1;
    }

    /**
     *  NOT COMPATIBLE WITH 1.3
     * Initializes the AlbertiKey given the keyword (permutationkey,shiftkey) and one or more alphabets.
     * @param keyword -- a string version of the keyword
     * @param alpha1, alpha2 -- Alphabets for plain/ciphertext respectively. May
     *  be identical, only one alphabet is nescessary, however.
     */ 

    public void init(String keyword, Alphabet alpha1, Alphabet alpha2) throws Exception { 
	if (alpha1 == null || alpha2 == null)
	    throw new Exception("AlbertiKey.init(): Null alphabet reference passed");
	splitKeyspec(keyword);         // Splits keyword into the permutation and shift key and assigns
                                       //  permutekey to this.keyword
	if (!isValidKeyword(shiftkey))
	    throw new Exception("Alberti keyword contains invalid characters for selected alphabet");       
	if (!isValidKeyword(permutekey))
	    throw new Exception("Alberti permutekey contains invalid characters for selected alphabet");
        super.initKey(this.keyword,alpha1,alpha2);
	this.blocksize = 1;
	//System.out.println(shiftkey +  " " + blocksize);
    }

    /**
     *  This method constructs char arrays consisting of
     *  the characters in the plaintext and ciphertext alphabets.
     *  Note: This method won't work if the keyword has duplicate
     *  characters. It will throw and exception.
     */ 
    protected void initKeyArrays() throws Exception  {
	if (this.keyword.indexOf(',') != -1) {  // Keyword may already be split into permute and shift keys
	    splitKeyspec(keyword);
	    keyword = permutekey;
	}
        super.initKeyArrays();
    } //initKeyArrays()

    /**
     * Takes in the initial keyword and splits it around the comma
     * @param String kywrd takes in the initial keyword to be split
     */
    public void splitKeyspec(String kywrd) throws Exception {
	StringTokenizer ke = new StringTokenizer(kywrd, ",");
	permutekey = ke.nextToken();
	if(!ke.hasMoreTokens())
	    throw new Exception("Alberti requires two keywords seperated by a comma:" + kywrd); 
	shiftkey = ke.nextToken();    
	//	System.out.println(shiftkey + "|" + permutekey + "|");
    }

    /**
     * @return shiftkey
     */
    public String getShiftKey() {
	return shiftkey;
    }	
   
    /**
     * @return permutekey
     */
    public String getPermutationKey() {
	return permutekey;
    }

    /**
     * @return the name of the algorithm "Alberti".
     */
    public String getAlgorithm() { 
	return "Alberti"; 
    }   
	
    /**
     * @param String s 
     * @return boolean true if the String s contains only letters from
     * the given alphabet, false otherwise.
     */
    private boolean isValidKeyword(String s)  {
	// System.out.println(s + "");
	for (int k = 0; k < s.length(); k++) {
	    if (!alphabet.isInAlphabet(s.charAt(k))) {
		System.out.println("ch = " + s.charAt(k));
		return false;
	    }
	}
	return true;
    }
} //AlbertiKey
