package hcrypto.engines;

import hcrypto.cipher.*;

/**
 * BeaufortEngine.java implements a traditional Beaufort Cipher for
 *  the full range of alphabets -- az, AZ, azAZ, azAZ09, printable characters, and all ASCII.
 *  The encoding method is based upon the following function:
 *      CipherChar = PlainChar - KeyChar
 *  The decoding method is based upon the following function:
 *      PlainChar = CipherChar + KeyChar
 *
 * A Beaufort cipher is a variation of Vigenere. A Beaufort cipher is
 * self-reciprocal, meaning that the same tableaux can be used for
 * encryption and decryption.  For a 26 letter alphabet, with a key of
 * k, Beaufort encryption and decryption are given by the formula: 
 *         c = (k - p) % 26
 * where 'c' is the cipher text and 'p' is the plaintext.
 *  
 *<P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="BeaufortKey.html">BeaufortKey</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 *
 */

public class BeaufortEngine extends VigenereEngine {  
 
   /**
    * Creates a BeaufortEngine and sets the <i>alphabetRangeOptions</i> instance variable
    * to "111111", which translates to all six alphabet options.
    */ 
    public BeaufortEngine() {
	super();
    }

   /**
    * Initializes the BeaufortEngine with the specified hKey.
    * @param hKey a BeaufortKey.
    */
    
    protected void engineInit(HistoricalKey hKey) throws Exception {
        if ( !(hKey instanceof BeaufortKey))
            throw new Exception("InvalidKey: Beaufort requires BeaufortKey");
	super.engineInit(hKey);
    }
   
   
   /**
    * engineEncode() is simply defined in terms of Vigenere decode()
    *  Characters which are not part of the chosen alphabet set are ignored.
    * @param s the String to be encrypted
    */

    public String engineEncode( String s ) throws Exception {
        if (blocksize != 1)
            throw new Exception("Invalid blocksize for Vignere cipher " + blocksize);
        char ch = s.charAt(0);

	if (alphabet.isInAlphabet(ch)) {
            char keyCh = keyword.charAt(keyPtr);
	    //            char newCh = encodeShift(ch,alphabet.charToInt(keyCh));
	    char newCh = encodeShift(keyCh,alphabet.getSize() - alphabet.charToInt(ch));
            keyPtr = (keyPtr + 1) % keyword.length();
//            System.out.println(ch + " " + low + " " + high + " " + keyCh + " " + range + " " + theShift + " " + newCh);
            return "" + newCh;
        }
        else
            return s;
    }  //engineEncode

    

    /**
    * engineDecode() is exactly the same as encode().
    *  Characters which are not part of the chosen alphabet set are ignored.
    * @param s the String to be decrypted
    */
    
    public String engineDecode( String s ) throws Exception {
	return engineEncode(s);
    } //engineDecode
}  //BeaufortEngine class


