package hcrypto.engines;

import hcrypto.cipher.*;

/**
 * This class implements the Caesar cipher algorithm for any
 * range of alphabets, including az, AZ, azAZ, azAZ09, printable, and ASCII.
 * As in the traditional Caesar cipher, letters in the plaintext are shifted
 * by an integral value. Thus for the traditional alphabet, az or AZ, if the
 * shift is 3, plaintext 'a' would become 'd' and plaintext 'A' would become
 * 'D'. 
 *
 * <p>This implementation of Caesar cipher is generalized to work for any
 * ASCII characters by scaling the modulus to the appropriate character range
 * depending on the plaintext or ciphertext character. For example, if the
 * character is the digit '5', then the modulus must be 10 in order to insure
 * that '5' is mapped onto another digit by the formula:
 * <pre>
 *       (char)((('5' - '0') + shift) % 10 + '0')
 * </pre>
 * 
 * The <a href="../cipher/Alphabet.html">Alphabet</a> class contains methods, <tt>getLowBound('5')</tt> 
 * and <tt>getRange('5')</tt>, which allow this mapping to be generalized to:
 * <pre>
 *       (char)(((ch - low) + shift) % range + low)
 * </pre>
 * 
 * so that the shift will apply to any character in the character set. Thus the 
 * mapping is always based on the low character in the appropriate character range --
 * 'a' for range az, '0' for range 09, ASCII 32 for range printable -- and the
 * appropriate modulus for the range -- 10, 26, etc.
 * <P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="CaesarKey.html">CaesarKey</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 */
 

public class CaesarEngine extends BlockCipher {  
 
    private int shift;             // Caesar shift
    private CaesarKey key;

   /**
    * Creates a CaesarEngine and sets the <i>alphabetRangeOptions</i> instance variable
    * to "111111", which translates to all six alphabet options for both the plaintext
    * and ciphertext alphabets.
    */ 
    
    public CaesarEngine() {
        alphabetRangeOptions = "111111/111111";  // Caesar allows all 6 possible alphabet ranges
    }
    
    
   /**
    * Initializes the CaesarEngine with the specified hKey.
    * @param hKey a CaesarKey.
    */
    
    protected void engineInit(HistoricalKey hKey) throws Exception {
        if ( !(hKey instanceof CaesarKey))
            throw new Exception("InvalidKey: Caesar requires CaesarKey");
        key = (CaesarKey) hKey;
        alphabet = key.getAlphabet();
        cipherAlphabet = key.getCTAlphabet();
        shift = key.getShift();
        this.blocksize = key.getBlocksize();
    }

    
   /**
    * Returns an encoded String for the specified String.  Characters which are not part of the chosen alphabet set are ignored.
    * @param s the String to be encrypted
    */

    public String engineEncode(String s ) throws Exception {
        if (blocksize != 1)
            throw new Exception("Invalid blocksize for Caesar cipher " + blocksize);
        char ch = s.charAt(0);
        if (alphabet.isInAlphabet(ch)) {
           return "" + encodeShift(ch, shift);
        }
        else
           return s;
    }


   /**
    * Returns an encoded String for the specified String.  Characters which are not part of the chosen alphabet set are ignored.
    * @param s the String to be decrypted
    */
         
    public String engineDecode( String s ) throws Exception {
        if (blocksize != 1)
            throw new Exception("Invalid blocksize for Caesar cipher " + blocksize);
        char ch = s.charAt(0);
	      if (cipherAlphabet.isInAlphabet(ch)) {
            return "" + decodeShift(ch, cipherAlphabet.getSize() - shift);
        }
        else
            return s;
    }
}//End of class.


