package hcrypto.engines;

import hcrypto.cipher.*;

/**
 * This class represents a key for a Caesar (shift) cipher. The key
 * is entirely represented by an integer, <i>shift</i>, which specifies
 * by how many characters each letter of the plaintext is supposed to be
 * shifted to the right in the alphabet. For example, with a shift of 3,
 * the letter 'A' would become 'D' and the digit '1' would become '4'.
 *
 * <p>For this implementation of Caesar cipher any of the alphabet ranges
 * can be used and any nonnegative shift value can be specified. If the
 * shift exceeds the range of characters in the alphabet -- e.g., if the
 * shift is 100 and the alphabet is 'a'..'z', then the shift is scaled modulo 26.
 * Negative shift values are rejected, resulting in an exception.
 *<P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="CaesarEngine.html">CaesarEngine</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 */
  

public class CaesarKey extends HistoricalKey {
    
   /**
    * A description for a Caesar Key.
    */ 
    
    public final static String DEFAULT_KEY_DESCRIPTOR_PROMPT_STRING = "a positive integer";
    
    
   /**
    * A default key value.
    */
     
    public final static String DEFAULT_KEYWORD_STRING = "5";
    
    private int shift;
   
    /**
     * Default constructor required.
     */
    //    public CaesarKey() { }
   
   /**
    * Creates the CaesarKey with the specified shift.  The default alphabet is set to "az".
    * @param shift is a positive integer, i.e. "5" which would be stored in the 
    * instance variables <i>shift</i> as an integer and <i>keyword</i> as a String.
    public CaesarKey(int shift) throws Exception {  // Default Constructor
        init(shift + "/" + "az");
    }
    */

   /**
    * Initializes the CaesarKey with the specified keyspec (which takes the form "4/az" 
    *  or "4/az/AZ" for example).
    * @param keySpec takes the form "<i>integer/alphabet</i>", i.e. "5/azAZ09"
    * which would be mapped into the instance variables <i>shift</i> as an integer with the value of 5, 
    * <i>keyword</i> as a String with the value of "5", and <i>alphabet</i> as a String with the value "azAZ09".
    */ 
    public void init(String keyspec) throws Exception { 
        initKey(keyspec, false);             // Inherited from superclass
        this.blocksize = 1;
        this.keyDescriptorPrompt = "positive integer";
        this.shift = Integer.parseInt(keyword);
        if (shift < 0)
            throw new Exception ("Invalid shift value for Caesar cipher: " + shift);
    }

   /**
    * Initializes the CaesarKey given a keyword and one or more alphabets.
    * @param keyword -- a string version of the keyword
    * @param alpha1, alpha2 -- Alphabets for plain/ciphertext respectively. May
    *  be identical.
    */ 
    public void init(String keyword, Alphabet alpha1, Alphabet alpha2) throws Exception { 
	if (alpha1 == null || alpha2 == null)
	    throw new Exception("CaesarKey.init(): Null alphabet reference passed");
        super.initKey(keyword, alpha1, alpha2);
        this.keyword = keyword;
        this.blocksize = 1;
        this.keyDescriptorPrompt = "positive integer";
        this.shift = Integer.parseInt(keyword);
        if (shift < 0)
            throw new Exception ("Invalid shift value for Caesar cipher: " + shift);
    }

   /**
    * Returns shift value.
    */
     
    public int getShift() { 
        return shift; 
    }

   /**
    * Returns the algorithm name "Caesar".
    */
     
    public String getAlgorithm() { 
        return "Caesar"; 
    }   
}
