package hcrypto.engines;

import java.util.*;
import hcrypto.cipher.*;

/**
 * <p>Title: EnigmaEngine</p>
 * <p>Description: A Simulation of the German Military Enigma Machine.
 *   Specifications of rotors and reflectors obtained from
 *   http://www.codesandciphers.org.uk/enigma/rotorspec.htm and
 *   http://homepages.tesco.net/~andycarlson/enigma/simulating_enigma.html</p>
 * @author Meghan Emilio
 * @version 1.0
 */
public class EnigmaEngine extends BlockCipher {

    //STATIC ROTORS                         01234567890123456789012345
    public final static String rotors[] = {"ABCDEFGHIJKLMNOPQRSTUVWXYZ",
					   "EKMFLGDQVZNTOWYHXUSPAIBRCJ",  // Rotor I
					   "AJDKSIRUXBLHWTMCQGZNPYFVOE",  // Rotor II
					   "BDFHJLCPRTXVZNYEIWGAKMUSQO",  // Rotor III
					   "ESOVPZJAYQUIRHXLNFTGKDCMWB",  // Rotor IV
					   "VZBRGITYUPSDNHLXAWMJQOFECK",  // Rotor V
					   "JPGVOUMFYQBENHZRDKASXLICTW",  // Rotor VI
					   "NZJHGRCXMYSWBOUFAIVLPEKQDT",  // Rotor VII
					   "JPGVOUMFYQBENHZRDKASXLICTW"}; // Rotor VIII

    public final String reflector0  = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";    //STATIC REFLECTORS
    public final String reflectorB  = "YRUHQSLDPXNGOKMIEBFZCWVJAT";    
    public final String reflectorC  = "FVPJIAOYEDRZXWGCTKUQSBNMHL";

    private int firstRotor, secondRotor, thirdRotor;  // Rotor numbers
    private int p1=0, p2=0, p3=0;                     // Current start positions of three rotors
    private int notch1, notch2;                       //  Enigma Ring (Notch) Sittings:  Q, E, V, J, Z, Z, Z, Z, 
    private String reflector;

    //CURRENT PLUGBOARD SETTINGS
    private char[] plugBoard; 

    private EnigmaKey key;

    public EnigmaEngine() {
        alphabetRangeOptions = "110000/110000";  // Upper and lowercase A-Z
    }

    /**
     * Class Constructor
     *@param r1 rotor to be used as first rotor
     *@param r2 rotor to be used as second rotor
     *@param r3 rotor to be used as third rotor
     *@param r reflector to be used
     */
    public EnigmaEngine(String r1, String r2, String r3, String r) {
	firstRotor = r1.charAt(0) - '0';
	secondRotor = r2.charAt(0) - '0';
	thirdRotor = r3.charAt(0) - '0';
	reflector = reflectorB;
	//	notch1 = getValue(r1)[1];
	//	notch2 = getValue(r2)[1];
    }
  
    protected void engineInit(HistoricalKey hKey) throws Exception {
        if (!(hKey instanceof EnigmaKey))
            throw new Exception("InvalidKey: Enigma requires EnigmaKey");
        key = (EnigmaKey)hKey;
        alphabet = key.getAlphabet();
        cipherAlphabet = key.getCTAlphabet();
        String rotors = key.getRotors();
	String ringsettings = key.getRingSettings();
	String keysettings = key.getKeySettings();

        firstRotor = rotors.charAt(0) - '0';
        secondRotor = rotors.charAt(1) - '0';
        thirdRotor = rotors.charAt(2) - '0';

	p1 = keysettings.charAt(0) - 'A';
	p2 = keysettings.charAt(1) - 'A';
	p3 = keysettings.charAt(2) - 'A';

        notch1 = ringsettings.charAt(0) - 'A';
        notch2 = ringsettings.charAt(1) - 'A';

        reflector = reflectorB;
        plugBoard = new char[26];
        initPlugboard(plugBoard);
	//	printPlugboard();
        if (!setPlugBoard(key.getPlugboardSettings()))
	    throw new Exception("Error in Plugboard Settings " + key.getPlugboardSettings());
	//	System.out.println(firstRotor +" "+ secondRotor +" "+ thirdRotor +" "+ notch1 +" "+ notch2);
    }

    private void initPlugboard(char pb[]) {
        for (int k = 0; k < 26; k++) 
	    pb[k] = (char)('A' + k);
    }

    /**
     *Creates a plubboard connection between two letters 
     *@param x first character to be connected
     *@param y second character to be connected
     *@return void
     */
    public void setPlugBoard(char x, char y){
	for(int i=0; i<plugBoard.length; i++){
	    if(plugBoard[i] == x)
		plugBoard[i] = y;
	    else if(plugBoard[i] == y)
		plugBoard[i] = x;
	}
    }

    /**
     *Sets the plug board settings
     *@param str plug board settings formatted in pairs, 
     *each pair seperated by a space
     *@return boolean if str entered was in correct format
     *and if the plugboard was set accordingly
     */
    public boolean setPlugBoard(String str){
	//      System.out.println("Plugboard settings = |" + str + "|");
	if (str.equals(""))  // Empty string
	    return true; 
	String s;
	StringTokenizer tokenCheck = new StringTokenizer(str, "-");
	//    StringTokenizer tokenCheck = new StringTokenizer(str, " ");
	while(tokenCheck.hasMoreTokens()){
	    s = tokenCheck.nextToken();
	    //      System.out.println("PLUG |" + s + "|");
	    if (s.length() != 2)
		return false;
	    if(s.charAt(0)>90 || s.charAt(0)<65 || s.charAt(1)>90 || s.charAt(1)<65)
		return false;
	}

	//    StringTokenizer token = new StringTokenizer(str, " ");
	StringTokenizer token = new StringTokenizer(str, "-");
	while(token.hasMoreTokens()){
	    s = token.nextToken();
	    if(s.length()==2)
		setPlugBoard(s.charAt(0), s.charAt(1));
	    else
		return false;
	}
	return true;
    }

    /**
     * Rotates the rotors according to their current settings
     *@param void
     *@return void
     */
    public void rotate(){
	int current1 = p1;
	int current2 = p2;
	p1 = (p1 + 1) % 26;   	//Rotate first rotor
	if (current1 == notch1) {	//If first rotor was at notch
	    p2 = (p2 + 1) % 26;
	    if (current2 == notch2)    //If second rotor was at notch
		p3 = (p3 + 1) % 26;
	}
    }

    /** 
     *Returns the result of passing c through
     *the plugboard with its current settings
     *@param c the inputted character
     *@return char the result of passing c through
     *the plugboard with its current settings
     */	
    public char plugBoard(char c){
	//	int i = (int)(c) - 65; 
	//	return plugBoard[i];
	return plugBoard[c - 'A'];
    }

    private void printPlugboard() {
	for (int k = 0; k < plugBoard.length; k++)
	    System.out.print(plugBoard[k]);
	System.out.println();
    }


    /** 
     *Returns the current setting of the first rotor.
     *@param void
     *@return char that is the current setting of the first rotor
     */
    //    public char getFRSetting(){
    //	return firstRotorT.charAt(0);
    //    }

    /** 
     *Returns the current setting of the second rotor.
     *@param void
     *@return char that is the current setting of the second rotor
     */
    //    public char getSRSetting(){
    //	return secondRotorT.charAt(0);
    //    }

    /** 
     *Returns the current setting of the third rotor.
     *@param void
     *@return char that is the current setting of the third rotor
     */
    //    public char getTRSetting(){
    //	return thirdRotorT.charAt(0);
    //    }

    public String engineDecode(String s) throws Exception {
        return engineEncode(s);
    }

  /** 
   *Encrypts/Decrypts the inputted string using the 
   *machine's current settings
   *@param p the text to be encrypted/decrypted
   *@return void
   */
  public String engineEncode(String s) throws Exception {
      s = s.toUpperCase();
      if (blocksize != 1)
          throw new Exception("Invalid blocksize for Enigma " + blocksize);
      char c = s.charAt(0);
      if (alphabet.isInAlphabet(c)) {
	  //	  printPlugboard();
           rotate();        	   //rotate the rotors
	   //	   System.out.print(c + ">");
           c = plugBoard(c);                                 // Plugboard
	   //	   System.out.print(c + ">");
	   c = rotors[firstRotor].charAt(((c-'A') + 26 - p1) % 26);   // Rotor #1
	   //	   System.out.print(c + ">");
	   c = rotors[secondRotor].charAt(((c-'A') + 26 - p2) % 26);   // Rotor #2
	   //	   System.out.print(c + ">");
	   c = rotors[thirdRotor].charAt(((c-'A') + 26 - p3) % 26);   // Rotor #3
	   //	   System.out.print(c + ">");
	   c = reflector.charAt(c-'A');                      // Reflector
	   //	   System.out.print(c + ">");
	   c = (char)('A' + ((rotors[thirdRotor].indexOf(c) + p3) % 26)); // Rotor #3
	   //	   System.out.print(c + ">");
	   c = (char)('A' + ((rotors[secondRotor].indexOf(c) + p2) % 26)); // Rotor #2
	   //	   System.out.print(c + ">");
	   c = (char)('A' + ((rotors[firstRotor].indexOf(c) + p1) % 26)); // Rotor #1
	   //	   System.out.print(c + ">");
           c = plugBoard(c);                                 // Plugboard
	   //	   System.out.println(c + ">");
           return "" + c;
       } else
	   return s;
  }

  /**
   *Parses Plugboard input to check for repeated letters
   *as each letter can only be used once in the plugboard 
   *@param str the inputted plug board settings
   *@return void
   */
  public boolean pbParser(String str){
  	//if no plug board settings were input, then continue
    if(str.length()<=0 || str.equals(null) || str == null){
      return true;
    }
    
    //otherwise, check to make sure letters are not repeated
    for(int i=0; i<str.length()-1; i++){
      //if not a letter, continue	
      if(str.charAt(i)>90 || str.charAt(i)<65)
        i++;
      //if the current letter appears in the rest of the string
      else if(str.substring(i+1).indexOf(str.charAt(i)) != -1)
        return false;
    }
    
    //otherwise, return true
    return true;
  }
  

  }

