package hcrypto.engines;

import hcrypto.cipher.*;

/**
 * This class implements the Playfair encryption algorithm. In this algorithm, 
 * pairs of individual letters from the range a..z are mapped into pairs of symbols 
 * taken from a 2-dimensional square matrix. The letter 'j' is dropped from the 
 * ciphertext alphabet. Thus, in this implementation, the ciphertext alphabet consists
 * of the letters a..z, minus the letter 'j'. A keyword is used to vary the 
 * distribution of the letters in the matrix.

 *
 *<P>For example: If the keyword is "keyword", the matrix would look like this:  
 *<pre>   
         k e y w o 
         r d a b c 
         f g h i l
         m n p q s
         t u v x z
 *</pre>
 *

 * <P>The following rules are used to perform the mapping:
 *
 *<p>1. If both letters happen to be in the same <i>row</i>, the
 *letters immediately to the right of each letter are used.  The
 *letter to the "right" of the last letter in a <i>row</i> is the
 *first letter in that same <i>row</i>.  For example, given the
 *above square, "ey" would be encrypted as "yw".

 *<p>2. If both letters are in the same <i>column</i>, the letters
 *immediately below each letter are used.  The letter "below" a bottom
 *letter is the top letter of the same <i>column</i>.  For example,
 *given the above square, "ed" would be encrypted by "dg".

 *<p>3. If the two letters are in different rows and in different
 *columns, each letter is replaced by the letter in the <i>same row
 *that is also in the column occupied by the other letter</i>. For example,
 *for the above square, "dh" would be encrypted by "ag".

 * 
 *<BLOCKQUOTE> 
 *      <BR><a href="PlayfairKey.html">PlayfairKey</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *</BLOCKQUOTE> */

public class PlayfairEngine extends BlockCipher {  //DOCUMENTED BY GREGG--NOT DONE

    private PlayfairKey key;

   /**
    * Creates a PlayfairEngine and sets the <i>alphabetRangeOptions</i> instance variable
    * to "100000", which translates to the "az" alphabet option only for both plaintext
    * and ciphertext.
    */ 
    
    public PlayfairEngine() {
        alphabetRangeOptions = "100000/100000";  // Just az
    }

    
   /**
    * Initializes the PlayfairEngine with the specified hKey.
    * @param hKey a PlayfairKey.
    */
    
    protected void engineInit(HistoricalKey hKey) throws Exception {
        if ( !(hKey instanceof PlayfairKey))
            throw new Exception("InvalidKey: Playfair requires PlayfairKey");
        key = (PlayfairKey) hKey;
        alphabet = key.getAlphabet();
        cipherAlphabet = key.getCTAlphabet();
        this.blocksize = key.getBlocksize();
    
    }
    
    /**
    * Returns an encoded String for the specified String.  
    *  Characters which are not part of the chosen alphabet set are ignored.
    * @param s the String to be encrypted
    */
    
    public String engineEncode( String s ) throws Exception {
        if (s.length() < blocksize)
             return s;
//            throw new Exception("ERROR: blocksize error in engineDecode");
 

        char ch1 = s.charAt(0);
        char ch2 = s.charAt(1);

        if (!alphabet.isInAlphabet(ch1) || !alphabet.isInAlphabet(ch2))
            return s;

        char newCh1;
        char newCh2;

        if (ch1 == 'j') ch1 = 'i';
        if (ch2 == 'j') ch2 = 'i';
 
        int indexCh1 = key.findCharPosition(ch1);
        int indexCh2 = key.findCharPosition(ch2);

        if ( key.getRowIndex(indexCh1) == key.getRowIndex(indexCh2)) {
            newCh1 = key.getChar( key.getRightColIndex(indexCh1) );            
            newCh2 = key.getChar( key.getRightColIndex(indexCh2) );            
        }
        else if ( key.getColIndex(indexCh1) == key.getColIndex(indexCh2)) {
            newCh1 = key.getChar( key.getRowBelowIndex(indexCh1) );            
            newCh2 = key.getChar( key.getRowBelowIndex(indexCh2) );            
        }
        else {
            newCh1 = key.getChar(key.getRowIndex(indexCh1), key.getColIndex(indexCh2));
            newCh2 = key.getChar(key.getRowIndex(indexCh2), key.getColIndex(indexCh1));
        }
//        System.out.println(s + " --> " + newCh1 + newCh2);
        return "" + newCh1 + newCh2;
    }

    
    /**
    * Returns an encoded String for the specified String.  
    *  Characters which are not part of the chosen alphabet set are ignored.
    * @param s the String to be decrypted
    */
    
    public String engineDecode( String s ) throws Exception {
	//	key.printKeySquare();
        if (s.length() < blocksize)
             return s;

        char ch1 = s.charAt(0);
        char ch2 = s.charAt(1);


        if (!cipherAlphabet.isInAlphabet(ch1) || !cipherAlphabet.isInAlphabet(ch2))
            return s;

        char newCh1;
        char newCh2;

        if (ch1 == 'j') ch1 = 'i';
        if (ch2 == 'j') ch2 = 'i';
 
        int indexCh1 = key.findCharPosition(ch1);
        int indexCh2 = key.findCharPosition(ch2);

        if ( key.getRowIndex(indexCh1) == key.getRowIndex(indexCh2)) {
            newCh1 = key.getChar( key.getLeftColIndex(indexCh1) );            
            newCh2 = key.getChar( key.getLeftColIndex(indexCh2) );            
        }
        else if ( key.getColIndex(indexCh1) == key.getColIndex(indexCh2)) {
            newCh1 = key.getChar( key.getRowAboveIndex(indexCh1) );            
            newCh2 = key.getChar( key.getRowAboveIndex(indexCh2) );            
        }
        else {
            newCh1 = key.getChar(key.getRowIndex(indexCh1), key.getColIndex(indexCh2));
            newCh2 = key.getChar(key.getRowIndex(indexCh2), key.getColIndex(indexCh1));
        }
//        System.out.println(s + " --> " + newCh1 + newCh2);
        
        return "" + newCh1 + newCh2;

    }

} 

