package hcrypto.engines;

import hcrypto.cipher.*;

/**
* This class represents a key for a Playfair cipher. The key is
* entirely represented by a keyword. The key is inserted into a matrix
* of letters in the <A href="PlayfairEngine.html">PlayfairEngine</A>
* class for encrypting and decrypting.  <P>For this implementation of
* the Playfair cipher the alphabet range is restricted to "az". In the
* ciphertext alphabet, the letter 'j' is omitted and j's in the plaintext
* message are changed to i's. 

 *
 *<P>For example: If the keyword is "keyword", the matrix would look like this:  
 *<pre>   
         k e y w o 
         r d a b c 
         f g h i l
         m n p q s
         t u v x z
 *</pre>
 *

 * <P>The following rules are used to perform the mapping:
 *
 *<p>1. If both letters happen to be in the same <i>row</i>, the
 *letters immediately to the right of each letter are used.  The
 *letter to the "right" of the last letter in a <i>row</i> is the
 *first letter in that same <i>row</i>.  For example, given the
 *above square, "ey" would be encrypted as "yw".

 *<p>2. If both letters are in the same <i>column</i>, the letters
 *immediately below each letter are used.  The letter "below" a bottom
 *letter is the top letter of the same <i>column</i>.  For example,
 *given the above square, "ed" would be encrypted by "dg".

 *<p>3. If the two letters are in different rows and in different
 *columns, each letter is replaced by the letter in the <i>same row
 *that is also in the column occupied by the other letter</i>. For example,
 *for the above square, "dh" would be encrypted by "ag".

* <P> See also:
* <BLOCKQUOTE>
* <BR><A href="PlayfairEngine.html">PlayfairEngine</A>
* <BR><A href="../cipher/Alphabet.html">Alphabet</A>
* </BLOCKQUOTE> 
*/

public class PlayfairKey extends HistoricalKey {

/**
* A keyword description.
*/

public final static String DEFAULT_KEY_DESCRIPTOR_PROMPT_STRING = ("a keyword");

/**
* A default keyword.
*/

public final static String DEFAULT_KEYWORD_STRING = ("keyword");

private static int SQUARE_SIZE = 26;
private static int SIDE = 5;

// Length of side of square
private char keySquare[];

/**
* keyspec takes the form "keyword/az"
*/

/**
* Initializes the PlayfairKey with the specified keyspec.
* @param keySpec takes the form "keyword/alphabet", i.e. "ralph/az" for example,
* which would be mapped into the instance variables <I>keyword</I> as a String 
* with the value of "ralph" and <I>alphabet</I> as a String with the value "az".
*/

public void init(String keyspec) throws Exception {
  initKey(keyspec, true); // true = remove duplicates from keyword
  blocksize = 2;
  if (alphabet.getRangeId() != AlphabetFactory.ALPH_az)
    throw new Exception("Invalid alphabet for Playfair key: " + alphabet.getRangeDesc());
  initKeySquare();
}

   /**
    * Initializes the PlayfairKey given a keyword and one or more alphabets.
    * @param keyword -- a string version of the keyword
    * @param alpha1, alpha2 -- Alphabets for plain/ciphertext respectively. May
    *  be identical.
    */ 
    public void init(String keyword, Alphabet alpha1, Alphabet alpha2) throws Exception { 
	if (alpha1 == null || alpha2 == null)
	    throw new Exception("PlayfairKey.init(): Null alphabet reference passed");
        super.initKey(keyword, alpha1, alpha2);
        this.keyword = HistoricalKey.removeDuplicateChars(keyword);
        this.blocksize = 2;
	if (alphabet.getRangeId() != AlphabetFactory.ALPH_az)
	    throw new Exception("Invalid alphabet for Playfair key: " + alphabet.getRangeDesc());
	initKeySquare();
    }

/**
* Returns the algorithm name "Playfair".
*/

public String getAlgorithm() {
  return "Playfair";
}

/**
* Returns the position of the specified character. Called from <A href="">PlayfairCipherSpi</A>.
*/

public int findCharPosition(char ch) throws Exception {
  int k;
  for (k = 0; keySquare[k] != ch && k < keySquare.length; k++) ; // NULL BODY
  if (k == keySquare.length)
    throw new Exception("Invalid character in Playfair " + ch);
  return k;
}

/**
* Returns the character at the specified index. Called from <A href="">PlayfairCipherSpi</A>.
*/

public char getChar(int index) {
  return keySquare[ index ];
}

/**
* Returns the character at the specified row and column. Called from <A href="">PlayfairCipherSpi</A>.
*/

public char getChar(int row, int col) {
  return keySquare[ getIndex(row, col) ];
}

/**
* Returns the row index for the specified index. Called from <A href="">PlayfairCipherSpi</A>.
*/

public int getRowIndex(int index) {
  return index / SIDE;
}

/**
* Returns the column index for the specified index. Called from <A href="">PlayfairCipherSpi</A>.
*/

public int getColIndex(int index) {
  return index % SIDE;
}

/**
* Returns the right column index for the specified index. Called from <A href="">PlayfairCipherSpi</A>.
*/

public int getRightColIndex(int index) {
  return getIndex( getRowIndex(index), (getColIndex(index) + 1) % SIDE);
}

/**
* Returns the left column index for the specified index. Called from <A href="">PlayfairCipherSpi</A>.
*/

public int getLeftColIndex(int index) {
  int newCol = getColIndex(index) - 1;
  if (newCol < 0)
    newCol = SIDE - 1;
  return
  getIndex( getRowIndex(index), newCol );
}

/**
* Returns the row below the index for the specified index. Called from <A href="">PlayfairCipherSpi</A>.
*/

public int getRowBelowIndex(int index) {
  return getIndex( (getRowIndex(index) + 1) % SIDE, getColIndex(index));
}

/**
* Returns the row above the index for the specified index. Called from <A href="">PlayfairCipherSpi</A>.
*/

public int getRowAboveIndex(int index) {
  int newRow = getRowIndex(index) - 1;
  if (newRow < 0)
      newRow = SIDE - 1;
  return getIndex( newRow, getColIndex(index));
}

/********* PRIVATE UTILITY METHODS ***********************/

private void initKeySquare() throws Exception {
  if (alphabet.getRangeId() != AlphabetFactory.ALPH_az)
      throw new Exception ("Invalid alphabet for Playfair cipher");

  keySquare = new char[SQUARE_SIZE];
  boolean inserted[] = new boolean[SQUARE_SIZE];
  for (int k = 0; k < keySquare.length; k++) {
      keySquare[k] = '*';
      inserted[k] = false;
  }

  // Insert the keyword into the square
  for (int k = 0; k < keyword.length(); k++) {
      char ch = keyword.charAt(k);
      if (ch == 'j')
          ch = 'i';
      if (!inserted[ch - 'a']) {
          keySquare[k] = ch;
          inserted[ch - 'a'] = true;
      }
      if (ch == 'i') // Mark both j and i
          inserted['j' - 'a'] = true;
  }

  int ptr = 0;
  for (int k = keyword.length(); k < keySquare.length; k++) {
      while (ptr < inserted.length && inserted[ptr])
          ptr++;       // Find the next unused letter
      if (ptr == ('i' - 'a')) { // i and j are special cases
          keySquare[k] = 'i';
          inserted['j' - 'a'] = true;
      }
      else if (ptr == ('j' - 'a'))
          keySquare[k] = 'i';
      else keySquare[k] = (char)(ptr + 'a');
      ptr++;
              // System.out.println(ptr + " " + k);
  }//for
   // printKeySquare();
}

private int getIndex(int rowIndx, int colIndx) {
  return SIDE * rowIndx + colIndx;
}

/******************* PUBLIC UTILITY METHODS ****************************/

/**
* Performs exhaustive tests of the methods that
* perform various indexing functions on the key.
*/

public void testMethods() {
  for (int k = 0; k < keySquare.length - 1; k++) {
    System.out.println("right " + k + " " + getRightColIndex(k));
    System.out.println("left " + k + " " + getLeftColIndex(k));
    System.out.println("up " + k + " " + getRowAboveIndex(k));
    System.out.println("down " + k + " " + getRowBelowIndex(k));
    }
}

/**
* Prints the playfair key to the System console.
*/

public void printKeySquare() {
  for (int k = 0; k < keySquare.length - 1; k++) {
      if (k % SIDE == 0)
          System.out.println();
      System.out.print(keySquare[k] + " ");
  }
}

} //class PlayfairKey
