package hcrypto.engines;

import hcrypto.cipher.*;

/* File: PolySubstitutionEngine.java  last modified on 2/4/2002.
*/

 /**
 * This class implements a PolySubstitution cipher algorithm (which many
 * authors call a Homophonic Substitution cipher). The key is represented
 * by a keyword, which is inserted at the beginning of the decoding alphabet.

 * <P>More precisely, the cipher alphabet is likely to be considerably larger
 * than the plaintext alphabet with certain plaintext characters mapped into
 * several different ciphertext characters, that is, the encryption mapping
 * is one-to-many. On the other hand, each ciphertext character represents a
 * unique plaintext character.  The keyword will be expanded to a string with
 * length equal to the size of the cipher alphabet with all characters in the
 * plaintext alphabet. The kth character of the cipher alphabet will be decoded
 * into the kth character of the keyword.  This map also uniquely determines
 * the set of characters that each plaintext character can be mapped into.

 * <P>The encoding algorithm maintains a variable that, for each character in the
 * plaintext message, stores the position of that character.  The value of that
 * variable modulo the number of ciphertext characters that plaintext character
 * can be mapped into determines which ciphertext character it is mapped into.

 * <p>   For example, suppose that the plaintext alphabet is a..z of size 26 and
 * the cipher alphabet is a..zA..Z of size 52.  Suppose the user chooses the
 * keyword: "nowisthetimetolearncryptography" of length 31. The 10 characters
 * "bdfjkquvxz" are not contained in this keyword so add them to the end of it.
 * Now add the first 11 characters of the resulting keyword to the end of itself
 * to get a new keyword of length 52.  Thus we have the following table for
 * decoding:

 *<PRE>
 * abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ  -- The cipher alphabet
 * nowisthetimetolearncryptographybdfjkquvxznowisthetim  -- The decoding alphabet
 * </PRE>
 
 * Thus for encryption, for example, "a" can be mapped into "q" or "B"
 * and "t" can be mapped into any of the letters of "fimxUX".  If the plaintext
 * message starts with the word "that", the first "t" of "that" would be encoded
 * as "f", the second would be encoded as "i", and so on. Since there are six
 * different encodings for "t", after six encodings, the seventh occurrence of
 * 't' would be encoded as "f".

 * The PolySubstution cipher is one that requires that two different
 * alphabets be used for the plaintext and ciphertext alphabets 
 * respectively. As just explained, the ciphertext alphabet must
 * be larger than the plaintext alphabet. The following code segment 
 * provides an example of how to create and use a Polysubstitution cipher:
 *
 * <PRE>
 *      Cipher cipher = Cipher.getInstance("PolySubstitution");
 *      HistoricalKey key = HistoricalKey.getInstance("PolySubstitution", cipher.getProvider());
 *      key.init("nowisthetimetolearncryptography/az/azAZ");
 *      cipher.init(key);
 *      String c1encrypt = cipher.encrypt("this is a secret message");
 * </PRE> 

 * In this case, the plaintext alphabet consists of the letters a..z, and
 * the ciphertext alphabet consists of the letters a..zA..Z.  The decoding
 * alphabet would be constructed from the keyphrase as described above.

 *<P>See also:
 *  <BLOCKQUOTE>
 *      <BR><a href="PolySubstitutionKey.html">PolySubstitutionKey</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 */

public class PolySubstitutionEngine extends BlockCipher { 

    private char[] decodingAlphabet;
    private String[] codingAlphabet;
    private PolySubstitutionKey key;
    private static int counter = 0;
    
   /**
    * Creates a PolySubstitutionEngine and sets the <i>alphabetRangeOptions</i>
    * instance variable to "111111", which translates to all six alphabet options.
    */ 
    
    public PolySubstitutionEngine() {
        alphabetRangeOptions = "111111/111111";  // Substitution allows all 6 possible alphabet ranges
    }

    
   /**
    * Initializes the PolySubstitutionEngine with the specified hKey.
    * @param hKey a PolySubstitutionKey.
    */
    
    protected void engineInit(HistoricalKey hKey) throws Exception {
        if ( !(hKey instanceof PolySubstitutionKey))
            throw new Exception ("InvalidKey: Substitution requires SubstitutionKey");
        key = (PolySubstitutionKey) hKey;
        alphabet = key.getAlphabet();
        cipherAlphabet =key.getCTAlphabet();
        this.blocksize = key.getBlocksize();
        decodingAlphabet = key.getDecodingAlphabet();
        codingAlphabet = key.getCodingAlphabet();
    }

   /**
    * Returns an encoded String for the specified String.  Characters which are
    * not part of the chosen alphabet set are ignored.
    * @param s the String to be encoded
    */
    
    
    public String engineEncode(String s) throws Exception {
        if (blocksize != 1)
            throw new Exception("Invalid blocksize for Substitution cipher " + blocksize);
        char ch = s.charAt(0);
   
	      if (alphabet.isInAlphabet(ch)){
            int num = alphabet.charToInt(ch);
            int len = codingAlphabet[num].length();
            return "" + codingAlphabet[num].charAt(counter++ % len);
        } //if
        else return s;
    }//engineEncode()

   /**
    * Returns an decoded String for the specified encode String.
    * Characters which are not part of the chosen alphabet set are ignored.
    * @param s the String to be decoded.
    */

    public String engineDecode(String s) throws Exception {
        if (blocksize != 1)
            throw new Exception("Invalid blocksize for Substitution cipher " + blocksize);
        char ch = s.charAt(0);

	      if (cipherAlphabet.isInAlphabet(ch))
            return "" + decodingAlphabet[cipherAlphabet.charToInt(ch)];
        else return s;
    }//engineDecode()

}//PolySubstitutionEngine class
