package hcrypto.engines;

import hcrypto.cipher.*;

/* File: PolySubstitutionKey.java  last modified on 2/4/2002.
written in Jan.2002 by REW by editing he file SubstitutionKey.java.
*/

/**
 * This class represents a key for a PolySubstitution cipher (which many
 * authors call a Homophonic Substitution). The key is entirely represented by
 * a keyword, which is inserted at the beginning of the decoding alphabet. More
 * precisely, the cipher alphabet is likely to be considerably larger
 * than the plaintext alphabet with certain plaintext characters mapped into
 * several different ciphertext characters, that is, the encryption mapping
 * is one-to-many. On the other hand, each ciphertext character represents a
 * unique plaintext character.  The keyword will be expanded to a string with
 * length equal to the size of the cipher alphabet with all characters in the
 * plaintext alphabet. The kth character of the cipher alphabet will be decoded
 * into the kth character of the keyword.  This map also uniquely determines
 * the set of characters that each plaintext character can be mapped into.

 * <P>The keyword provided by the user is expanded to length equal to the size
 * of the cipher alphabet in the following manner.  First a string is created
 * of all plaintext characters not in the keyword and this string is added to
 * the end of the keyword.  If this would create a keyword longer than the size
 * of the cipher alphabet, the keyword is truncated before adding the missing
 * plaintext characters.  On the other hand, if the keyword plus missing
 * characters creates a new keyword with length less than the size of the cipher
 * alphabet, copies of the new keyword are added to itself till it is longer
 * than the size of the cipher alphabet before it is truncated to be exactly
 * of length equal to the size of the cipher alphabet.
 * <p>   For example, suppose that the plaintext alphabet is a..z of size 26 and
 * the cipher alphabet is a..zA..Z of size 52.  Suppose the user chooses the
 * keyword: "nowisthetimetolearncryptography" of length 31. The 10 characters
 * "bdfjkquvxz" are not contained in this keyword so add them to the end of it.
 * Now add the first 11 characters of the resulting keyword to the end of itself
 * to get a new keyword of length 52.  Thus we have the following table for
 * decoding:
 * <PRE>
 * abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ  -- The cipher alphabet
 * nowisthetimetolearncryptographybdfjkquvxznowisthetim  -- The decoding alphabet
 * </PRE>
 * Thus for encryption, for example, "a" can be mapped into "q" or "B"
 * and "t" can be mapped into any of the letters of "fimxUX"

 * The PolySubstution cipher is one that requires that two different
 * alphabets be used for the plaintext and ciphertext alphabets 
 * respectively. As just explained, the ciphertext alphabet must
 * be larger than the plaintext alphabet. The following code segment 
 * provides an example of how to create and use a Polysubstitution cipher:
 *
 * <PRE>
 *      Cipher cipher = Cipher.getInstance("PolySubstitution");
 *      HistoricalKey key = HistoricalKey.getInstance("PolySubstitution", cipher.getProvider());
 *      key.init("nowisthetimetolearncryptography/az/azAZ");
 *      cipher.init(key);
 *      String c1encrypt = cipher.encrypt("this is a secret message");
 * </PRE> 

 * In this case, the plaintext alphabet consists of the letters a..z, and
 * the ciphertext alphabet consists of the letters a..zA..Z.  The decoding
 * alphabet would be constructed from the keyphrase as described above.

 *<P>See also:
 *  <BLOCKQUOTE>
 *      <BR><a href="PolySubstitutionEngine.html">PolySubstitutionEngine</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 */ 

public class PolySubstitutionKey extends HistoricalKey {
    
   /**
    * A keyword description.
    */ 
    public final static String DEFAULT_KEY_DESCRIPTOR_PROMPT_STRING = ("a plaintext keyword");
        
   /**
    * A default keyword.
    */ 
    public final static String DEFAULT_KEYWORD_STRING = ("keyword");
   
    private char[] decodingAlphabet;  //stores the plaintext characters
    private String[] codingAlphabet;    //stores the ciphertext strings

   /**
    * Creates a PolySubstitutionKey with the specified keyspec.
    * @param keySpec takes the form "keyword/alphabet/cipheralphabet",
    * i.e. "nowisthetimetolearncryptography/az/azAZ" for example,
    * which would be mapped into the instance variables <i>keyword</i> 
    * as a String with the value of "nowisthetimetolearncryptography",
    * <i>alphabet</i> as a String with the value "az" and
    * and <i>cipheralphabet</i> as a String with the value "azAZ".
    */ 
    public void init(String keyspec) throws Exception {
        initKey(keyspec, false);  // Inherited from superclass. Assigns keyword
	                                // false = don't remove duplicates in keyword
        this.blocksize = 1;
    }
    
   /**
    * Initializes the PolySubstitutionKey given a keyword and one or more alphabets.
    * @param keyword -- a string version of the keyword
    * @param alpha1, alpha2 -- Alphabets for plain/ciphertext respectively. May
    *  be identical.
    */ 
    public void init(String keyword, Alphabet alpha1, Alphabet alpha2) throws Exception { 
	if (alpha1 == null || alpha2 == null)
	    throw new Exception("PolySubstitution.init(): Null alphabet reference passed");
        super.initKey(keyword, alpha1, alpha2);
        this.keyword = keyword;
        this.blocksize = 1;
    }

   /**
    * Returns the algorithm name "PolySubstitution".
    */
    public String getAlgorithm() { 
        return "PolySubstitution";
    }

   /**
    * Returns the decoding alphabet.
    */
    public char[] getDecodingAlphabet() {
        return decodingAlphabet;
    }

   /**
    * Returns the plaintext alphabet.
    */
    public String[] getCodingAlphabet() {
        return codingAlphabet;
    }


/******************* PRIVATE UTILITY METHODS *****************/

    protected void initKeyArrays() throws Exception {
	//    private void initKeyArrays()throws Exception {
        char ch;  //Used in loops for storing a character
        int index; //Used in a loop
        codingAlphabet = new String[alphabet.getSize()];
        decodingAlphabet = new char[cipherAlphabet.getSize()];
        String keychars = "";

       //Initialize strings in the array.
        for (int k = 0; k < codingAlphabet.length; k++) {
              codingAlphabet[k] = "";
        }//for

        //Eliminate keyword characters not in alphabet
        for (int k = 0; k < keyword.length(); k++) {
            ch = keyword.charAt(k);
            if (alphabet.isInAlphabet(ch))
                keychars = keychars + ch;
        }//for

        //Add any alphabet characters not in keychars
        String missingChars = "";
        for (int k = 0; k < alphabet.getSize(); k++) {
            ch = alphabet.intToChar(k);
            if (keychars.indexOf(ch) == -1)
                missingChars = missingChars + ch;
        }//for
        //Worry about subtle problem for keyword which is too long.
        while(keychars.length()+missingChars.length() > cipherAlphabet.getSize()){
            keychars = keychars.substring(0,cipherAlphabet.getSize()-missingChars.length());
            missingChars = "";
            for (int k = 0; k < alphabet.getSize(); k++) {
                ch = cipherAlphabet.intToChar(k);
                if (keychars.indexOf(ch) == -1)
                      missingChars = missingChars + ch;
            }//for
        }//while
        keychars = keychars + missingChars;
        // keychars now contains each alphabet character at least once

         //Expand keychars to the lenghth of the cipher alphabet.
        while(keychars.length() < cipherAlphabet.getSize()){
             keychars = keychars + keychars;
        }//while
        keychars = keychars.substring(0,cipherAlphabet.getSize());
        for (int k = 0; k < keychars.length(); k++) {
            ch = keychars.charAt(k);
            decodingAlphabet[k] = ch;
            index = alphabet.charToInt(ch);
            codingAlphabet[index] = codingAlphabet[index]+ cipherAlphabet.intToChar(k);
        }//for

       // System.out.println("DEBUG: PRINITNG ALPHABETS ");
        // printAlphabets();

  } //initKeyArrays()}


/******************* PUBLIC UTILITY METHODS *****************/

   /**
    *  Prints both the cipher and plain alphabets to the System console.
    */

    public void printAlphabets() {
        for (int k = 0; k < cipherAlphabet.getSize(); k++)
            System.out.print(decodingAlphabet[k]);
        System.out.println();
        for (int k = 0; k < alphabet.getSize(); k++)
            System.out.print(codingAlphabet[k]+" ");
        System.out.println();
    }

}//PolySubstitutionKey


