package hcrypto.engines;

import hcrypto.cipher.*;

/**
 * This class implements a <i>keyword-based</i> Substitution cipher algorithm for any
 * range of alphabets, including az, AZ, azAZ, azAZ09, printable characters, all ASCII,
 * and any of the Unicode character sets.
 * A <i>plaintext alphabet</i> is retrieved from the <a href="SubstitutionKey.html">SubstitutionKey</a> 
 * class and stored in a character array.  A <i>cipher alphabet</i> is created by 
 * inserting the characters from the keyword
 * into the first elements of the array, followed by all remaining characters in the particular alphabet.
 *  
 *
 * <p>Encryption involves replacing a plaintext character from a given message with a character from the 
 *    cipher alphabet with the same index.  Decrypting works in the reverse order.
 * <pre>
 *      plainKey = [a,b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,......];
 *      keyword = calm;
 *      cipherAlphabet = [c,a,l,m,b,d,e,f,g,h,i,j,k,n,o,p,......];
 *      
 *      engineEncrypt("cab") == "lca"
 *      engineDecrypt("lca") == "cab"
 * </pre>
 * 
 * <P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="SubstitutionKey.html">SubstitutionKey</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 */
 
public class SubstitutionEngine extends BlockCipher {  //DOCUMENTED BY GREGG -- DONE

    private char[] cipherKey;
    private char[] plainKey;
    private SubstitutionKey key;
    
   /**
    * Creates a SubstitutionEngine and sets the <i>alphabetRangeOptions</i> instance variable
    * to "111111", which translates to all six alphabet options.
    */ 
    
    public SubstitutionEngine() {
        alphabetRangeOptions = "111111/111111";  // Substitution allows all 6 possible alphabet ranges
    }

    
   /**
    * Initializes the SubstitutionEngine with the specified hKey.
    * @param hKey a SubstitutionKey.
    */
    
    protected void engineInit(HistoricalKey hKey) throws Exception {
        if ( !(hKey instanceof SubstitutionKey))
            throw new Exception ("InvalidKey: Substitution requires SubstitutionKey");
        key = (SubstitutionKey) hKey;
        alphabet = key.getAlphabet();
        cipherAlphabet =key.getCTAlphabet();
        this.blocksize = key.getBlocksize();
        cipherKey = key.getCipherKey();
        plainKey = key.getPlainKey();
    }

    
   /**
    * Returns an encoded String for the specified String.  
    * Characters which are not part of the chosen alphabet set are ignored.
    * @param s the String to be encrypted
    */
    
    
    public String engineEncode(String s) throws Exception {
        if (blocksize != 1)
            throw new Exception("Invalid blocksize for Substitution cipher " + blocksize);
        char ch = s.charAt(0);
   
	if (alphabet.isInAlphabet(ch)) 
            return "" + cipherKey[alphabet.charToInt(ch)];
        else return s;
    }
    
    
    
   /**
    * Returns an encoded String for the specified String.  
    *  Characters which are not part of the chosen alphabet set are ignored.
    * @param s the String to be decrypted
    */
    
    
    public String engineDecode(String s) throws Exception {
        if (blocksize != 1)
            throw new Exception("Invalid blocksize for Substitution cipher " + blocksize);
        char ch = s.charAt(0);

	if (cipherAlphabet.isInAlphabet(ch))
            return "" + plainKey[cipherAlphabet.charToInt(ch)];
        else return s;
    } 
} 
