package hcrypto.engines;

import hcrypto.cipher.*;

/**
 * This class represents a key for a Simple Substitution cipher. The key
 * is entirely represented by a keyword, which which is inserted at the beginning
 * of a SubstitutionAlphabet. For example, the keyword "ralph" would begin the substitution
 * alphabet and the remaining letters would would follow in their natural progression: 
 * "ralphbcdefgijkmnoqstuvwxyz". 

 *<P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="SubstitutionEngine.html">SubstitutionEngine</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 */ 

public class SubstitutionKey extends HistoricalKey {  
    
   /**
    * A prompt for the keyword
    */ 
    public final static String DEFAULT_KEY_DESCRIPTOR_PROMPT_STRING = ("a keyword");
        
   /**
    * A default keyword.
    */ 
    public final static String DEFAULT_KEYWORD_STRING = ("keyword");

   /**
    * Creates a SubstitutionKey with the specified keyspec.
    * @param keySpec takes the form "keyword/alphabet", 
    * i.e. "76TrombonesLEDTHEBIGPARADE/az+AZ+09" for example,
    * which would be mapped into the instance variables <i>keyword</i> 
    * as a String with the value of "76TrombonesLEDTHEBIGPARADE",
    * and <i>alphabet</i> as a String with the value "azAZ09".
    */ 
    public void init(String keyspec) throws Exception {
        super.initKey(keyspec, true);  // true => remove duplicates
        this.blocksize = 1;
    }

   /**
    * Initializes the SubstitutionKey given a keyword and one or more alphabets.
    * @param keyword -- a string version of the keyword
    * @param alpha1, alpha2 -- Alphabets for plain/ciphertext respectively. May
    *  be identical.
    */ 
    public void init(String keyword, Alphabet alpha1, Alphabet alpha2) throws Exception { 
	if (alpha1 == null || alpha2 == null)
	    throw new Exception("SubstitutionKey.init(): Null alphabet reference passed");
	//	System.out.println("Substitution key: keyword " + keyword);
        super.initKey(keyword, alpha1, alpha2);
        blocksize = 1;
	//        System.out.println(keyword +  " " + blocksize);
    }

   /**
    * Returns the algorithm name "Substitution".
    */
    public String getAlgorithm() { 
        return "Substitution"; 
    }

} //SubstitutionKey class




