package hcrypto.engines;

import hcrypto.cipher.*;

/**
 * This class implements a Transposition cipher algorithm for any
 * range of alphabets, including az, AZ, azAZ, azAZ09, printable characters, and all ASCII.
 * A keyword is retrieved from the <a href>"TranspositionKey.html">TranspositionKey</a> class and stored in
 * a String instance variable.  The keyword consists of a series of positive integers, i.e. "3021".  
 * 
 * <p>Encryption involves transposing a plaintext character from a given message with another character from the 
 *    same plain text message.  The basic algorithm for encrypting is:
 * <pre>
 *      StringBuffer sb = new StringBuffer(blocksize);
 *      for (int k = 0; k < keyword.length(); k++) {
 *          sb.setCharAt((int)(keyword.charAt(k) - '0'), s.charAt(k));
 *      }
 *      return sb;
 * </pre>
 * 
 * <p>Decryption basically involves the same algorithm, only with an inverted keyword (created and retrieved from the TranspositionKey class).
 * <pre>
 *      StringBuffer sb = new StringBuffer(blocksize);
 *      for (int k = 0; k < keyword.length(); k++) {
 *          sb.setCharAt((int)(invertedKeyword.charAt(k) - '0'), s.charAt(k));
 *      }
 *      return sb;
 *</pre>
 *<p>Padding and blocksize are handled in the <tt>engineEncode(String s)</tt> and <tt>engineDecode(String s)</tt> methods
 <P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="TranspositionKey.html">TranspositionKey</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 */


public class TranspositionEngine extends BlockCipher { //DOCUMENTED BY GREGG -- NOT DONE
 
    private TranspositionKey key;
    private String keyword;
    private String invertedKeyword;

    
    /**
    * Creates a TranspositionEngine and sets the <i>alphabetRangeOptions</i> instance variable
    * to "111111", which translates to all six alphabet options.
    */ 
    
    public TranspositionEngine() {
        alphabetRangeOptions = "111111/111111";  // Allows all 6
    }

    
    
   /**
    * Initializes the TranspositionEngine with the specified hKey.
    * @param hKey a TranspositionKey.
    */
    
    protected void engineInit(HistoricalKey hKey) throws Exception {
        if ( !(hKey instanceof TranspositionKey))
            throw new Exception("InvalidKey: Transposition requires TranspositionKey");
        key = (TranspositionKey) hKey;
        alphabet = key.getAlphabet();
        cipherAlphabet = key.getCTAlphabet();
        this.blocksize = key.getBlocksize();
        keyword = key.getKeyword();
        invertedKeyword = key.getInvertedKeyword();
//        System.out.println("blocksize = " + blocksize + " " +  keyword + " " + invertedKeyword);
   }

    
    
   /**
    * Returns an encoded String for the specified String.  Characters which are not part of the chosen alphabet set are ignored.
    * @param s the String to be encrypted
    */
    
    public String engineEncode( String s ) throws Exception {
//        System.out.print(s);
        char ch = 'a'; //for use in the second loop below
        StringBuffer sb = new StringBuffer(blocksize);
        for (int k = 0; k < blocksize; k++)
            sb.append(" ");
        for (int k = 0; k < keyword.length(); k++) {
	    if (alphabet.isInAlphabet(s.charAt(k))) {
		ch = cipherAlphabet.intToChar(alphabet.charToInt(s.charAt(k)));
		sb.setCharAt((int)(keyword.charAt(k) - '0'), ch);
	    }
	    else
		sb.setCharAt((int)(keyword.charAt(k) - '0'), s.charAt(k));
        }
//        System.out.println("-->" + sb.toString());
        return sb.toString();
    }

   
   
    /**
    * Returns an encoded String for the specified String.  Characters which are not part of the chosen alphabet set are ignored.
    * @param s the String to be decrypted
    */   
   
    public String engineDecode( String s ) throws Exception {
//        System.out.print(s);
        char ch = 'a'; //for use in the second loop below
        if (s.length() < blocksize)                    // It must be the last block
            return s;
        StringBuffer sb = new StringBuffer();
        for (int k = 0; k < blocksize; k++)
            sb.append(" ");
        for (int k = 0; k < invertedKeyword.length(); k++) {
	    if (alphabet.isInAlphabet(s.charAt(k))) {
		ch = alphabet.intToChar(cipherAlphabet.charToInt(s.charAt(k)));
		sb.setCharAt((int)(invertedKeyword.charAt(k) - '0'), ch);
	    }
	    else
		sb.setCharAt((int)(keyword.charAt(k) - '0'), s.charAt(k));
        }
//        System.out.println("-->" + sb.toString());
        return sb.toString();
    }
} 

