package hcrypto.engines;

import hcrypto.cipher.*;

/**
 * This class represents a key for a Transposition cipher. The key
 * is entirely represented by a keyword, a String with a set of positive contiguous 
 * integers always including zero: such as "2013". 
 * <p>For this implementation of Transposition cipher any of the alphabet ranges
 * can be used.
 * <P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="TranspositionKey.html">TranspositionKey</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 */
 public class TranspositionKey extends HistoricalKey { 
   /**
    * A keyword description.
    */ 
    
    public final static String DEFAULT_KEY_DESCRIPTOR_PROMPT_STRING = ("a series of positive integers");
        
    
   /**
    * A default key.
    */
    
    public final static String DEFAULT_KEYWORD_STRING = ("2013");
  
    private String invertedKeyword;     //used for decrypting
      
   /**
    * Initializes the TranspositionKey with the specified keyspec.
    * @param keySpec takes the form "keyword/alphabet", i.e. "2013/azAZ09" for example,
    * which would be mapped into the instance variables <i>keyword</i> as a String with the value of "2013",
    * and <i>alphabet</i> as a String with the value "azAZ09".
    */ 
    public void init(String keyspec) throws Exception { 
        super.initKey(keyspec, true);  // true = remove duplicates from keyword
        blocksize = keyword.length();
        invertedKeyword = getInverseKey(keyword);
	//        System.out.println(keyword + " " + invertedKeyword + " " + blocksize);
    }
    
   /**
    * Initializes the TranspositionKey given a keyword and one or more alphabets.
    * @param keyword -- a string version of the keyword
    * @param alpha1, alpha2 -- Alphabets for plain/ciphertext respectively. May
    *  be identical.
    */ 
    public void init(String keyword, Alphabet alpha1, Alphabet alpha2) throws Exception { 
	if (alpha1 == null || alpha2 == null)
	    throw new Exception("TranspositionKey.init(): Null alphabet reference passed");
	//	System.out.println("Transposition key: keyword " + keyword);
        super.initKey(keyword, alpha1, alpha2);
        blocksize = keyword.length();
        invertedKeyword = getInverseKey(keyword);
	//        System.out.println(keyword + " " + invertedKeyword + " " + blocksize);
    }


   /**
    * Returns the name of the algorithm "Transposition".
    */
    
    public String getAlgorithm() { 
        return "Transposition"; 
    }   
    
   /**
    * Returns the inverted keyword.
    */
     
    public String getInvertedKeyword() { 
        return invertedKeyword; 
    }   

    private String getInverseKey(String s) {
        StringBuffer sb = new StringBuffer(blocksize);
        for (int k = 0; k < s.length(); k++)
            sb.append(" ");
        for (int k = 0; k < s.length(); k++) {
            char ch = (char)(s.indexOf((char)('0' + k)) + '0');
            sb.setCharAt(k, ch);
        }
        return sb.toString();
    }

}
