package hcrypto.engines;
//package plugins;

import hcrypto.cipher.*;
import hcrypto.provider.*;
import java.util.StringTokenizer;

/**
 * TypeIVEngine.java implements a modified TypeIV Cipher for
 *  the full range of alphabets -- az, AZ, azAZ, azAZ09, printable characters, and all ASCII.
 *
 *<P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="TypeIVKey.html">AlbertiKey</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 */

public class TypeIVEngine extends BlockCipher   //DOCUMENTED BY Will
{
    private String shiftkeywd;
    private String plainkeywd;
    private String cipherkeywd;
    private TypeIVKey key;
    private char[] cipherKey;
    private char[] cipherKeyInverse;
    private char[] plainKey;
    private int[] encryptIndex;
    private int[] decryptIndex;
    private char[] plainKeyInverse;
    private char[][] encryptTable;
    private char[][] decryptTable;
    private int ShiftPtr;
    private int keyPtr;        // Cycles through chars in keyword
    
	/**
	 * Creates a TypeIVEngine and sets the <i>alphabetRangeOptions</i> instance variable
	 * to "111111", which translates to all six alphabet options.
	 */ 
    	
    
	public TypeIVEngine() 
	{
        	alphabetRangeOptions = "111111/111111";  // TypeIV allows all 6 possible alphabet ranges
	}

	/**
	 * Initializes the TypeIVEngine with the specified hKey.
	 * @param hKey a TypeIVKey.
	 */
    
	protected void engineInit(HistoricalKey hKey) throws Exception 
	{
		if ( !(hKey instanceof TypeIVKey))
        		throw new Exception("InvalidKey: TypeIV requires TypeIVKey");
		key = (TypeIVKey) hKey;
	        alphabet = key.getAlphabet();
        	cipherAlphabet = key.getCTAlphabet();
	        shiftkeywd = key.getShiftKeywd();
		plainkeywd = key.getPlainKeywd();
		cipherkeywd = key.getCipherKeywd();
	        this.blocksize = key.getBlocksize();
		cipherKey = key.getCipherKey();
		plainKey = key.getPlainKey();
		cipherKeyInverse = key.getCipherKeyInverse();
		plainKeyInverse = key.getPlainKeyInverse();
		ShiftPtr = 0;				// Initializes ShiftPtr
	        keyPtr = 0;                         // Initialize keyPtr

		encryptTable = makeEncryptTable(plainKey, cipherKey, shiftkeywd);
		//		decryptTable = makeDecryptTable(plainKey, cipherKey, shiftkeywd);
		//		decryptTable = makeDecryptTable(plainKeyInverse, cipherKeyInverse, shiftkeywd);
	}
	
        private char[][] makeEncryptTable(char[] plain, char[] cipher, String shiftwd) {
	    encryptIndex = makeIndex(plain);
            char table[][] = new char[shiftwd.length()][plain.length];   // Two dimensional table
	    for (int k = 0; k < shiftwd.length(); k++)
		table[k] = rotate(cipher, shiftwd.charAt(k) - 'a');
	    //	    printTable(table,plain);
	    return table;
        }

        private char[][] makeDecryptTable(char[] plain, char[] cipher, String shiftwd) {
	    decryptIndex = makeIndex(cipher);
            char table[][] = new char[shiftwd.length()][plain.length];   // Two dimensional table
	    for (int k = 0; k < shiftwd.length(); k++)
		//		table[k] = rotate(plain, 26 - (shiftwd.charAt(k) - 'a'));
		table[k] = rotate(plain, shiftwd.charAt(k) - 'a');
	    //	    printTable(table,cipher);
	    return table;
        }


    /**
     * complementWord() computes the 26-ch complement of a shift keyword
     */
    private String complementWord(String s) {
	StringBuffer result = new StringBuffer();             
	for (int k = 0; k < s.length(); k++) 
	    result.append((char)('a' + ((26 - s.charAt(k) + 'a') % 26) )); // Complement of s
	return result.toString();
    }

	/**
	 * Returns an encoded String for the specified String.  
	 *  Characters which are not part of the chosen alphabet set are ignored.
	 * @param s the String to be encrypted
	 * @return String representing the encrypted input
	 */

	public String engineEncode( String s ) throws Exception {
	    if (blocksize != 1)
		throw new Exception("Invalid blocksize for Alberti cipher " + blocksize);        
	    char ch = s.charAt(0);
	    if (alphabet.isInAlphabet(ch)) {
		char encodeCh = encryptTable[keyPtr][encryptIndex[alphabet.charToInt(ch)]];
		keyPtr = (keyPtr + 1) % shiftkeywd.length();
		return "" + encodeCh;
	    }
	    else
		return s;
    	}  //engineEncode

	/**
	 * Finds the index of a given character
	 * @param char c in the char key[]
	 * @return int the index of char c
	 */
	public int indexOf(char[] key, char c) {
	    int index = 0;	
	    for(int i = 0; i < plainKey.length; i++) {
		if(c == key[i])
		    return index;
		index++;
	    }
	    return -1;
	}

    private int[] makeIndex(char[] key) {
	int[] index = new int[key.length];
	for (char ch = 'a'; ch <= 'z'; ch++) {
            index[ch - 'a'] = indexOf(key,ch);
	}
	return index;
    }

    public String engineDecode( String s ) throws Exception {
	if (blocksize != 1)
	    throw new Exception("Invalid blocksize for Alberti cipher " + blocksize);
	char ch = s.charAt(0);
	if (cipherAlphabet.isInAlphabet(ch))  {
	    int indx = indexOf(encryptTable[keyPtr],ch);
	    //	    char decodeCh = decryptTable[keyPtr][decryptIndex[cipherAlphabet.charToInt(ch)]];
	    keyPtr = (keyPtr + 1) % shiftkeywd.length();
	    //   System.out.println(ch + " " + low + " " + high + " " + keyCh + " " + range + " " + theShift + " " + newCh);
	    return "" + plainKey[indx];
	}
	else
	    return s;
    } //engineDecode

    /**
     * rotate() rotates a character array until the
     *  character representing the shift is in the first
     *  location.
     */
    private char[] rotate(char[] key, int shift) {
        char[] newkey = new char[key.length];
	int p = 0;
	while (key[p] - 'a' != shift) 
	    ++p;

	for (int k = 0; k < key.length; k++) {
	    newkey[k] = key[p];
            p = (p + 1) % key.length;
	}
	return newkey;
    }

    /**
     * prints the encrypt or decrypt table
     */
    protected void printTable(char[][] table, char[]key) {
        for (int j = 0; j < key.length; j++) 
	    System.out.print(key[j] + " ");
	System.out.println("\n--------------------------------------------------");
        for (int j = 0; j < table.length; j++) {
            for (int k = 0; k < table[j].length; k++) 
		System.out.print(table[j][k] + " ");
	    System.out.println();
	}
	System.out.println();
    }
}  //TypeIVEngine class
