package hcrypto.engines;
//package plugins;

import hcrypto.cipher.*;

import java.util.StringTokenizer;

/**
 * This class represents a key for a TypeIV Quagmire cipher. The key
 * is entirely represented by a keyword String composed of characters from the chosen alphabets. 
 * <p>For this implementation of TypeIV cipher any of the alphabet ranges
 * can be used.
 * <P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="TypeIVEngine.html">AlbertiEngine</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 */

public class TypeIVKey extends HistoricalKey 
{
    /**
     * A keyword description.
     */  
    public final static String DEFAULT_KEY_DESCRIPTOR_PROMPT_STRING = ("plainkey,cipherkey,shiftkey");
        
   
    /**
     * A default keyword.
     */ 
    public final static String DEFAULT_KEYWORD_STRING = ("plainkey,cipherkey,shiftkey");

    /**
     * Strings to hold the two keywords, used to make the cipheralphabet and period of rotation.
     */
	
    private String shiftkeywd, plainkeywd, cipherkeywd;
    private char[] plainKeyInverse, cipherKeyInverse;

    /**
     * Initializes the TypeIVKey with the specified keyspec.
     * @param keySpec takes the form "keyword/alphabet", where
     * "plainkeywd,cipherkeywd,shiftwd/az" for example, where
     * keyword consists of 'plainkeywd,cipherkeywd,shiftwd'
     * and <i>alphabet</i> as a String with a value such as "az+AZ+09".
     */ 
    public void init(String keyspec) throws Exception { 
        this.keyspec = keyspec;
        if (keyspec.indexOf("/") == -1)
            throw new Exception("Invalid key specification: " + keyspec);
        if (keyspec.indexOf("/") == 0)
            throw new Exception("Keyword missing in key specification: " + keyspec);

        StringTokenizer st = new StringTokenizer(keyspec, "/");
	keyword = st.nextToken();                   // e.g., "plain,cipher,shiftwd"

        alphabet = AlphabetFactory.getInstance(st.nextToken());  // e.g., "az"
        if (st.hasMoreTokens())              
            cipherAlphabet = AlphabetFactory.getInstance(st.nextToken());
        else 
            cipherAlphabet = alphabet;

	initKeyArrays();

	if (!isValidKeyword(shiftkeywd))
	    throw new Exception("TypeIV shiftkeywd contains invalid characters for selected alphabet");
	if (!isValidKeyword(plainkeywd))
	    throw new Exception("TypeIV plainkeywd contains invalid characters for selected alphabet");
	if (!isValidKeyword(cipherkeywd))
	    throw new Exception("TypeIV cipherkeywd contains invalid characters for selected alphabet");
	this.blocksize = 1;
    }

    /**
     *  NOT COMPATIBLE WITH 1.3
     * Initializes the TypeIVKey given the keyword (plainkeywd,cipherkeywd,shiftkeywd) and one or more alphabets.
     * @param keyword -- a string version of the keyword
     * @param alpha1, alpha2 -- Alphabets for plain/ciphertext respectively. May
     *  be identical, only one alphabet is nescessary, however.
     */ 

    public void init(String keyword, Alphabet alpha1, Alphabet alpha2) throws Exception { 
	if (alpha1 == null || alpha2 == null)
	    throw new Exception("TypeIVKey.init(): Null alphabet reference passed");
	splitKeywords(keyword);         // Splits keyword into the permutation and shift key and assigns
                                       //  permutekey to this.keyword
	if (!isValidKeyword(shiftkeywd))
	    throw new Exception("TypeIV keyword contains invalid characters for selected alphabet");       
	if (!isValidKeyword(plainkeywd))
	    throw new Exception("TypeIV plainkeywd contains invalid characters for selected alphabet");
	if (!isValidKeyword(cipherkeywd))
	    throw new Exception("TypeIV cipherkeywd contains invalid characters for selected alphabet");
        super.initKey(this.keyword,alpha1,alpha2);
	this.blocksize = 1;
	//System.out.println(shiftkeywd +  " " + blocksize);
    }

    /**
     *  This method constructs char arrays consisting of
     *  the characters in the plaintext and ciphertext alphabets.
     *  Note: This method won't work if the keyword has duplicate
     *  characters. It will throw and exception.
     */ 
    protected void initKeyArrays() throws Exception {
	//	if (this.keyword.indexOf(',') != -1) {  // Keyword may already be split into keys
	    splitKeywords(keyword);
	    //	    keyword = permutekey;
	    //	}

	if (alphabet.getSize() != cipherAlphabet.getSize()) 
	    throw new Exception ("ERROR: Plain and cipher alphabets must be the same size");

        char ch;  //Used in loops for storing a character

	plainKey = new char[alphabet.getSize()];
        plainKeyInverse = new char[alphabet.getSize()];
	cipherKey = new char[alphabet.getSize()];
	cipherKeyInverse = new char[alphabet.getSize()];

	initKeyArray(alphabet, plainKey, plainKeyInverse, plainkeywd);
	initKeyArray(cipherAlphabet, cipherKey, cipherKeyInverse, cipherkeywd);
    } //initKeys()}

    protected void initKeyArray(Alphabet alpha, char[] keyarr, char[] keyarrinv, String keywd) throws Exception {
        StringBuffer keychars = new StringBuffer();

        //Eliminate characters not in the alphabet
        for (int k = 0; k < keywd.length(); k++) {
            char ch = keywd.charAt(k);
            if (alpha.isInAlphabet(ch))
                keychars.append(ch);
        }//for

	String keycharStr = keychars.toString(); 

        //Add any cipher alphabet characters not in keychars
        for (int k = 0; k < alpha.getSize(); k++) {
            char ch = alpha.intToChar(k);
            if (keycharStr.indexOf(ch) == -1) 
		keychars.append(ch);
        }//for

        // keychars now contains each alphabett character exactly once
	//        System.out.println("Keychars = " + keychars);

	if (keychars.length() != keyarr.length) 
	    throw new Exception ("ERROR: Unable to construct alphabets from keyword " + keywd);

	for (int k = 0; k < keychars.length(); k++) {
            char ch = keychars.charAt(k);
//	    System.out.println("keychars.len = " + keychars.length() + " ch= " + ch);
            keyarr[k] = ch;
            keyarrinv[cipherAlphabet.charToInt(ch)] = alphabet.intToChar(k);
        }//for
    }

    /**
     * Takes in the initial keyword and splits it around the comma
     * @param String kywrd takes in the initial keyword to be split
     */
    public void splitKeywords(String kywrd) throws Exception {
	StringTokenizer st = new StringTokenizer(kywrd, ",");
	plainkeywd = st.nextToken();
	if(!st.hasMoreTokens())
	    throw new Exception("TypeIV requires three keywords seperated by a comma:" + kywrd); 
	cipherkeywd = st.nextToken();
	if(!st.hasMoreTokens())
	    throw new Exception("TypeIV requires three keywords seperated by a comma:" + kywrd); 
	shiftkeywd = st.nextToken();    
	//	System.out.println(plainkeywd + "|" + cipherkeywd + "|" + shiftkeywd + "|");
    }

    /**
     * @return shiftkeywd
     */
    public String getShiftKeywd() {
	return shiftkeywd;
    }	
   
   /**
    * Returns the inverse cipher alphabet.
    */
    public char[] getCipherKeyInverse() {
        return cipherKeyInverse;
    }

   /**
    * Returns the inverse plain alphabet.
    */
    public char[] getPlainKeyInverse() { 
        return plainKeyInverse; 
    }

    /**
     * @return plainkeywd
     */
    public String getPlainKeywd() {
	return plainkeywd;
    }

    /**
     * @return cipherkeywd
     */
    public String getCipherKeywd() {
	return plainkeywd;
    }

    /**
     * @return the name of the algorithm "TypeIV".
     */
    public String getAlgorithm() { 
	return "TypeIV"; 
    }   
	
    /**
     * @param String s 
     * @return boolean true if the String s contains only letters from
     * the given alphabet, false otherwise.
     */
    private boolean isValidKeyword(String s)  {
	// System.out.println(s + "");
	for (int k = 0; k < s.length(); k++) {
	    if (!alphabet.isInAlphabet(s.charAt(k))) {
		System.out.println("ch = " + s.charAt(k));
		return false;
	    }
	}
	return true;
    }
} //TypeIVKey
