package hcrypto.engines;

import hcrypto.cipher.*;

/**
 * VigenereEngine.java implements a traditional Vigenere Cipher for
 *  the full range of alphabets -- az, AZ, azAZ, azAZ09, printable characters, and all ASCII.
 *  The encoding method is based upon the following function:
 *      CipherChar = PlainChar + KeyChar
 *  The decoding method is based upon the following function:
 *      PlainChar = CipherChar - KeyChar
 *
 *  A traditional Vigenere assumes that the alphabet is AZ and that
 *  the key characters is also in the range AZ. However, for this
 *  system the key character may not be in the same range as the
 *  cipher of plain character. In that case, the key character must
 *  be scaled to fall into the range of the plain or cipher character.
 *  For example, if the plain character is a digit, '5', and the key
 *  character is the letter 't', then 't' must be scaled to a value
 *  between 0 and 9. Since 't' is character 19 in a..z, we scale it
 *  to 9 ( = 19 % 10). This allows us to shift '5' by 9 (% 10), giving '4'.
 *  On decoding, the key characters 't' will occur above '4'. Again
 *  we scale to 't' to 9 and perform subtraction (% 10) to get '5'.
 *  
 *  A similar kind of scaling occurs for other alphabet ranges.
 *
 * NOTE: Added 12/15/2004.  The encode() and decode() methods
 *  used by this implementation of VigenereEngine were tested using
 *  cryptograms in A. Sinkov, Elementary Cryptanalysis, p59 and p77.
 *  They produced identical encryptions to those used in Sinkov.
 * 
 *<P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="VigenereKey.html">VigenereKey</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 *
 */

public class VigenereEngine extends BlockCipher {  //DOCUMENTED BY GREGG--DONE
 
    protected String keyword;
    protected VigenereKey key;
    protected int keyPtr;        // Cycles through chars in keyword
    
    
    
   /**
    * Creates a VigenereEngine and sets the <i>alphabetRangeOptions</i> instance variable
    * to "111111", which translates to all six alphabet options.
    */ 
    
    
    public VigenereEngine() {
        alphabetRangeOptions = "111111/111111";  // Vigenere allows all 6 possible alphabet ranges
    }

   /**
    * Initializes the VigenereEngine with the specified hKey.
    * @param hKey a VigenereKey.
    */
    
    protected void engineInit(HistoricalKey hKey) throws Exception {
        if ( !(hKey instanceof VigenereKey))
            throw new Exception("InvalidKey: Vigenere requires VigenereKey");
        key = (VigenereKey) hKey;
        alphabet = key.getAlphabet();
        cipherAlphabet = key.getCTAlphabet();
        keyword = key.getKeyword();
        this.blocksize = key.getBlocksize();
        keyPtr = 0;                         // Initialize keyPtr
	printSquare();
    }
   
   
   /**
    * Returns an encoded String for the specified String.  
    *  Characters which are not part of the chosen alphabet set are ignored.
    * @param s the String to be encrypted
    */

    public String engineEncode( String s ) throws Exception {
        if (blocksize != 1)
            throw new Exception("Invalid blocksize for Vignere cipher " + blocksize);
        char ch = s.charAt(0);

	if (alphabet.isInAlphabet(ch)) {
            char keyCh = keyword.charAt(keyPtr);
            char newCh = encodeShift(ch,alphabet.charToInt(keyCh));
            keyPtr = (keyPtr + 1) % keyword.length();
//            System.out.println(ch + " " + low + " " + high + " " + keyCh + " " + range + " " + theShift + " " + newCh);
            return "" + newCh;
        }
        else
            return s;
    }  //engineEncode

    

    /**
    * Returns an decoded String for the specified String.  
    *  Characters which are not part of the chosen alphabet set are ignored.
    * @param s the String to be decrypted
    */
    
    public String engineDecode( String s ) throws Exception {
        if (blocksize != 1)
            throw new Exception("Invalid blocksize for Caesar cipher " + blocksize);
        char ch = s.charAt(0);
        if (cipherAlphabet.isInAlphabet(ch)) {
          char keyCh = keyword.charAt(keyPtr);
          char newCh = decodeShift(ch,alphabet.getSize() - alphabet.charToInt(keyCh));
          keyPtr = (keyPtr + 1) % keyword.length();
//   System.out.println(ch + " " + low + " " + high + " " + keyCh + " " + range + " " + theShift + " " + newCh);
          return "" + newCh;
       }
       else
            return s;
    } //engineDecode

    protected void printSquare() {
	String primary = "abcdefghijklmnopqrstuvwxyz";
	for (int j = 0; j < 26; j++) {
	    for (int k = 0; k < primary.length(); k++) {
		//		System.out.print((char)('a' + ((primary.charAt(k) -'a' + j) % 26)) + " ");
		System.out.print(primary.charAt((k + j) % primary.length()) + " ");
	    }
	    System.out.println();
	}
    }

}  //VigenereEngine class


